/*
Copyright (C) 1995 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the 
terms of the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _array_store_h_
#define _array_store_h_

char* _edelim_ = "\t";                 // element output delimiter
char* _ldelim_ = "\n";                 // line output delimiter

template<class T> class mask_pointer;
template<class T> class value_pointer;
template<class T> class value_mask_pointer;
template<class T> class scalar_pointer;

template<class T> 
class mask_allocator 
{
public:
  typedef T			value_type;
  typedef mask_pointer<T>	pointer;
  typedef size_t		size_type;

  pointer allocate(size_type n) { 
    pointer t; t.a_=new T[n]; t.o_=t.a_; t.ma_=0; return t; }
  void deallocate(pointer p) { delete [] p.a_; delete [] p.ma_; }
};

template<class T>
class value_allocator
{
public:
  typedef T                     value_type;
  typedef value_pointer<T>	pointer;
  typedef size_t                size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class T>
class value_mask_allocator
{
public:
  typedef T                     value_type;
  typedef value_mask_pointer<T>	pointer;
  typedef size_t                size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class T>
class scalar_allocator
{
public:
  typedef T                     value_type;
  typedef scalar_pointer<T>	pointer;
  typedef size_t                size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class T>
class scalar_pointer 
{
private:
  T a_;
public:
  scalar_pointer() {}
  scalar_pointer(const size_t s, const T& a=T()) : a_(a) {}
  scalar_pointer(const scalar_pointer& a) : a_(a.a_) {}
  ~scalar_pointer() {}
  scalar_pointer& operator=(const scalar_pointer& a) { a_=a.a_; return *this; }
  T& operator[](size_t n) { return a_; }
  T operator[](size_t n) const { return a_; }
  scalar_pointer& operator++() { return *this; }
  scalar_pointer& operator--() { return *this; }
  scalar_pointer& operator+=(size_t n) { return *this; }
  scalar_pointer& operator-=(size_t n) { return *this; }
};

template<class T>
class value_pointer 
{
private:
  size_t s_;
  T *a_,*o_;
public:
  value_pointer() : a_(0), o_(0), s_(0) {}
  value_pointer(const size_t s, const T& a=T()) : s_(s), o_(new T[s]) {
    a_=o_; for(size_t j=0;j<s_;j++) a_[j]=a; }
  value_pointer(const value_pointer& a) : s_(a.s_), o_(new T[a.s_]) {
    a_=o_; for(size_t j=0;j<s_;j++) a_[j]=a.o_[j]; }
  ~value_pointer() { delete [] o_; }
  value_pointer& operator=(const value_pointer& a) {
    this->~value_pointer();
    s_=a.s_; o_=new T[a.s_]; a_=o_; 
    for(size_t j=0;j<s_;j++) a_[j]=a.o_[j]; }
  T& operator[](size_t n) { return a_[n]; }
  T operator[](size_t n) const { return a_[n]; }
  value_pointer& operator++() { a_++; return (*this); }
  value_pointer& operator--() { a_--; return (*this); }
  value_pointer& operator+=(size_t n) { a_+=n; return (*this); }
  value_pointer& operator-=(size_t n) { a_-=n; return (*this); }
};

template<class T>
class mask_pointer 
{
private:
  T *a_,*o_; 
  size_t ms_,*ma_;               // array for indirect element access
public:
  friend mask_allocator<T>;

  mask_pointer() : a_(0), o_(0), ma_(0) {}
  mask_pointer& operator++() { a_++; return (*this); }
  mask_pointer& operator--() { a_--; return (*this); }
  mask_pointer& operator+=(size_t n) { a_+=n; return (*this); }
  mask_pointer& operator-=(size_t n) { a_-=n; return (*this); }

  T& operator[](size_t n) { if(ma_) return a_[ma_[n]]; return a_[n]; }
  T operator[](size_t n) const { if(ma_) return a_[ma_[n]]; return a_[n]; }

  void mask();
  void mask(size_t length,size_t *m);
  void mask(size_t offset,size_t length,size_t stride);
  void mask(size_t offset,size_t n,size_t *length,size_t *stride);
  void mask(size_t length,bool *m);
};

template<class T>
class value_mask_pointer : public mask_pointer 
{
private: 
  size_t s_;
public:
  value_mask_pointer() : s_(0) {}
  value_mask_pointer(const size_t s, const T& a=T()) : s_(s), o_(new T[s]), ma_(0), ms_(0) {
    a_=o_; for(size_t j=0;j<s_;j++) a_[j]=a; }
  value_mask_pointer(const value_mask_pointer& a) : s_(a.s_), o_(new T[a.s_]), ma_(0), ms_(a.ms_) {
    a_=o_; 
    for(size_t j=0;j<s_;j++) a_[j]=a.o_[j];   // deep copy the array
    if(a.ma_) {             // if masked, deep copy the masking array
      ma_=new size_t[ms_];
      for(size_t j=0;j<ms_;j++) ma_[j]=a.ma_[j];
    } }
  value_mask_pointer& operator=(const value_mask_pointer& a) {
    this->~value_mask_pointer();
    s_=a.s_; o_(new T[a.s_]); a_=o_; 
    for(size_t j=0;j<s_;j++) a_[j]=a.o_[j]; 
    ms_=a.ms_;
    if(a.ma_) {             // if masked, deep copy the masking array
      ma_=new size_t[ms_];
      for(size_t j=0;j<ms_;j++) ma_[j]=a.ma_[j];
    } }
  ~value_mask_pointer() { delete [] o_; delete [] ma_; }
  value_mask_pointer& operator++() { a_++; return (*this); }
  value_mask_pointer& operator--() { a_--; return (*this); }
  value_mask_pointer& operator+=(size_t n) { a_+=n; return (*this); }
  value_mask_pointer& operator-=(size_t n) { a_-=n; return (*this); }
};

template<class T>
void 
mask_pointer<T>::mask()
{                             // reset to default
  delete [] ma_; 
  ma_=0;
}

template<class T>
void 
mask_pointer<T>::mask(size_t length,size_t *m)
{ // copy from a preexisting array
  ms_=length;
  size_t* store=ma_;
  ma_=new size_t[ms_]; 
  for(size_t j=0;j<length;j++) ma_[j]=m[j]; 
  if(store!=(size_t*)0) {
    for(size_t j=0;j<ms_;j++) ma_[j]=store[ma_[j]];
    delete [] store;
  }
}

template<class T>
void 
mask_pointer<T>::mask(size_t offset,size_t length,size_t stride)
{ // stride
  ms_=length;
  size_t* store=ma_;
  ma_=new size_t[ms_]; 
  for(size_t j=0;j<length;j++) ma_[j]=offset+j*stride; 
  if(store!=(size_t*)0) {
    for(size_t j=0;j<ms_;j++) ma_[j]=store[ma_[j]];
    delete [] store;
  }
}

template<class T>
void 
mask_pointer<T>::mask(size_t offset,size_t n,size_t *length,size_t *stride)
{ 
  // generalized stride, n=# of indeces
  ms_=1;
  for(size_t j=0;j<n;j++) ms_*=length[j];    // # of elements
  size_t* store=ma_;
  ma_=new size_t[ms_]; 
  size_t v[n];       // array to loop over indeces
  for(size_t j=0;j<n;j++) v[j]=0;   // initialize to 0
  size_t j=0;
  while(v[0]<length[0]) { // loop over all possible indeces
    ma_[j]=offset;    // compute element position for this v
    for(size_t k=0;k<n;k++) ma_[j]+=v[k]*stride[k];
    v[n-1]++;            // increment last index of v
    for(size_t l=n-1;l>0;l--) {  // no index of v may exceed length index
      if(v[l]>=length[l]) {  // if it does, 
        v[l]=0;              // set it to 0 
        v[l-1]++;            // and increment next index
      } else break; 
    } 
    j++;       // next mask element
  } 
  if(store!=(size_t*)0) {
    for(size_t j=0;j<ms_;j++) ma_[j]=store[ma_[j]];
    delete [] store;
  }
}

template<class T>
void 
mask_pointer<T>::mask(size_t length,bool *m)
{ // initialize from a bitmask
  ms_=0;
  for(size_t j=0;j<length;j++) if(m[j]) ms_++;
  size_t* store=ma_;
  ma_=new size_t[ms_]; 
  for(size_t j=0,c=0;j<ms_;j++) 
    if(m[j]) ma_[c++]=j; 
  if(store!=(size_t*)0) {
    for(size_t j=0;j<ms_;j++) ma_[j]=store[ma_[j]];
    delete [] store;
  }
}

#endif         _array_store_h_

