/*
     This file is part of GNUnet.
     (C) 2010, 2012, 2013 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 3, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/peerinfo/gnunet-peerinfo-gtk.c
 * @brief Main function of gnunet-peerinfo-gtk
 * @author Christian Grothoff
 */
#include "gnunet_gtk.h"
#include <gnunet/gnunet_peerinfo_service.h>
#include <gnunet/gnunet_transport_service.h>
#include <gnunet/gnunet_ats_service.h>
#include <gnunet/gnunet_friends_lib.h>
#include "gnunet-peerinfo-gtk-flags.h"

#if HAVE_LIBUNIQUE
#include <unique/unique.h>
#endif


/**
 * Columns in the peerinfo model.
 */
enum PEERINFO_ModelColumns
{
  /**
   * A gchararray
   */
  PEERINFO_MC_PEER_IDENTITY_STRING = 0,

  /**
   * A guint
   */
  PEERINFO_MC_NUMBER_OF_ADDRESSES = 1,

  /**
   * A gchararray
   */
  PEERINFO_MC_COUNTRY_NAME = 2,

  /**
   * A GdkPixbuf
   */
  PEERINFO_MC_COUNTRY_FLAG = 3,

  /**
   * A guint64
   */
  PEERINFO_MC_BANDWIDTH_IN = 4,

  /**
   * A guint64
   */
  PEERINFO_MC_BANDWIDTH_OUT = 5,

  /**
   * A gchararray
   */
  PEERINFO_MC_ADDRESS_AS_STRING = 6,

  /**
   * A GdkPixbuf
   */
  PEERINFO_MC_CONNECTIVITY_LED = 7,

  /**
   * A gboolean
   */
  PEERINFO_MC_CONNECTED_STATUS = 8,

  /**
   * A gboolean
   */
  PEERINFO_MC_IS_FRIEND = 9,

  /**
   * A `struct PeerInfo *`
   */
  PEERINFO_MC_PEERINFO = 10
};


/**
 * Information we track for each peer outside of the model.
 */
struct PeerInfo
{
  /**
   * Reference to the peer in the view.
   */
  GtkTreeRowReference *rr;

  /**
   * Handle to an active lookup for addresses of this peer, or NULL.
   */
  struct GNUNET_TRANSPORT_PeerIterateContext *palc;

  /**
   * Handle for address to string conversion.
   */
  struct GNUNET_TRANSPORT_AddressToStringContext *tos;

  /**
   * Identity of the peer for this entry.
   */
  struct GNUNET_PeerIdentity pid;

  /**
   * Did we get any address?
   */
  int got_address;
};


/**
 * Handle to our main loop.
 */
static struct GNUNET_GTK_MainLoop *ml;

/**
 * Handle for our notifications from peerinfo about new peers.
 */
static struct GNUNET_PEERINFO_NotifyContext *pnc;

/**
 * Handle to ATS service.
 */
static struct GNUNET_ATS_PerformanceHandle *ats;

/**
 * Map of peer identities to the respective PeerInfo for our view.
 */
static struct GNUNET_CONTAINER_MultiPeerMap *peer2info;

/**
 * Should gnunet-peerinfo-gtk start in tray mode?
 */
static int tray_only;

/**
 * Green status led (connected)
 */
static GdkPixbuf *led_green;

/**
 * Red status led (disconnected)
 */
static GdkPixbuf *led_red;

/**
 * Main window list store.
 */
static GtkListStore *ls;

/**
 * Map of all of our friends.
 */
static struct GNUNET_CONTAINER_MultiPeerMap *friends;


#if HAVE_LIBUNIQUE
static UniqueApp *unique_app;
#endif


/**
 * Get cfg.
 */
static const struct GNUNET_CONFIGURATION_Handle *
get_configuration ()
{
  return GNUNET_GTK_main_loop_get_configuration (ml);
}


/**
 * Get an object from the main window.
 *
 * @param name name of the object
 * @return NULL on error
 */
static GObject *
get_object (const char *name)
{
  return GNUNET_GTK_main_loop_get_object (ml, name);
}


/**
 * Function called on each entry in the 'peer2info' map
 * to free the associated path.
 *
 * @param cls unused
 * @param key peer identity
 * @param value the 'struct PeerInfo'
 * @return #GNUNET_OK (continue to iterate)
 */
static int
free_paths (void *cls,
	    const struct GNUNET_PeerIdentity *key,
	    void *value)
{
  struct PeerInfo *info = value;

  if (NULL != info->palc)
  {
    GNUNET_TRANSPORT_peer_get_active_addresses_cancel (info->palc);
    info->palc = NULL;
  }
  if (NULL != info->tos)
  {
    GNUNET_TRANSPORT_address_to_string_cancel (info->tos);
    info->tos = NULL;
  }
  gtk_tree_row_reference_free (info->rr);
  GNUNET_free (info);
  return GNUNET_OK;
}


/**
 * Task run on shutdown.
 *
 * @param cls unused
 * @param tc scheduler context, unused
 */
static void
shutdown_task (void *cls, const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  GNUNET_GTK_tray_icon_destroy ();
  GNUNET_GTK_main_loop_quit (ml);
  ml = NULL;
  if (NULL != pnc)
  {
    GNUNET_PEERINFO_notify_cancel (pnc);
    pnc = NULL;
  }
  if (NULL != ats)
  {
    GNUNET_ATS_performance_done (ats);
    ats = NULL;
  }
  GNUNET_CONTAINER_multipeermap_iterate (peer2info, &free_paths, NULL);
  GNUNET_CONTAINER_multipeermap_destroy (peer2info);
  peer2info = NULL;
  GNUNET_PEERINFO_GTK_flags_shutdown ();
}



/**
 * Function to call with the text format of an address
 *
 * @param cls the 'struct PeerInfo' for which this is a valid address
 * @param address address as a string, NULL on error
 */
static void
peer_address_string_cb (void *cls, const char *address)
{
  struct PeerInfo *info = cls;
  GtkTreeIter iter;
  GtkTreePath *path;
  char *country;
  const char *colon;
  const char *dot;

  path = gtk_tree_row_reference_get_path (info->rr);
  GNUNET_assert (NULL != path);
  GNUNET_assert (TRUE == gtk_tree_model_get_iter (GTK_TREE_MODEL (ls), &iter, path));
  gtk_tree_path_free (path);
  if (NULL == address)
  {
    /* error */
    if (GNUNET_NO == info->got_address)
      gtk_list_store_set (ls, &iter,
			  PEERINFO_MC_NUMBER_OF_ADDRESSES, (guint) 1,
			  PEERINFO_MC_COUNTRY_NAME, NULL,
			  PEERINFO_MC_COUNTRY_FLAG, NULL,
			  PEERINFO_MC_ADDRESS_AS_STRING, "<no address>",
			  PEERINFO_MC_CONNECTIVITY_LED, led_green,
			  PEERINFO_MC_CONNECTED_STATUS, TRUE,
			  -1);
    info->tos = NULL;
    return;
  }
  /* last address, store information in model */
  country = NULL;
  colon = strstr (address, ":");
  if (NULL != colon)
  {
    for (dot = colon - 1; dot != address; dot--)
      if ('.' == *dot)
	break;
    if ('.' == *dot)
      country = GNUNET_strndup (&dot[1], (colon - dot) - 1);
  }
  gtk_list_store_set (ls, &iter,
		      PEERINFO_MC_NUMBER_OF_ADDRESSES, 1,
		      PEERINFO_MC_COUNTRY_NAME, country,
		      PEERINFO_MC_COUNTRY_FLAG, GNUNET_PEERINFO_GTK_get_flag (country),
		      PEERINFO_MC_ADDRESS_AS_STRING, address,
		      PEERINFO_MC_CONNECTIVITY_LED, led_green,
		      PEERINFO_MC_CONNECTED_STATUS, TRUE,
		      -1);
  GNUNET_free_non_null (country);
  info->got_address = GNUNET_YES;
}


/**
 * Function to call with a binary format of an address
 *
 * @param cls the `struct PeerInfo` for which this is a valid address
 * @param peer peer the update is about
 * @param address NULL on disconnect, otherwise 0-terminated printable UTF-8 string
 */
static void
peer_address_cb (void *cls,
                 const struct GNUNET_PeerIdentity *peer,
                 const struct GNUNET_HELLO_Address *address)
{
  struct PeerInfo *info = cls;
  GtkTreeIter iter;
  GtkTreePath *path;

  path = gtk_tree_row_reference_get_path (info->rr);
  GNUNET_assert (NULL != path);
  GNUNET_assert (TRUE == gtk_tree_model_get_iter (GTK_TREE_MODEL (ls), &iter, path));
  gtk_tree_path_free (path);
  if (NULL == address)
  {
    /* disconnect */
    gtk_list_store_set (ls, &iter,
			PEERINFO_MC_NUMBER_OF_ADDRESSES, (guint) 0,
			PEERINFO_MC_CONNECTIVITY_LED, led_red,
			PEERINFO_MC_CONNECTED_STATUS, FALSE,
                        -1);
    return;
  }
  gtk_list_store_set (ls, &iter,
		      PEERINFO_MC_NUMBER_OF_ADDRESSES, 1,
		      PEERINFO_MC_CONNECTIVITY_LED, led_green,
		      PEERINFO_MC_CONNECTED_STATUS, TRUE,
		      -1);
  if (NULL != info->tos)
    GNUNET_TRANSPORT_address_to_string_cancel (info->tos);
  info->got_address = GNUNET_NO;
  info->tos =
      GNUNET_TRANSPORT_address_to_string (get_configuration (), address,
                                          GNUNET_NO,
                                          GNUNET_TIME_UNIT_FOREVER_REL,
                                          &peer_address_string_cb, info);
}


/**
 * Function called for peers that we know about.
 *
 * @param cls closure
 * @param peer id of the peer, NULL for last call
 * @param hello hello message for the peer (can be NULL)
 * @param err_msg NULL if successful, otherwise contains error message
 */
static void
peerinfo_processor (void *cls,
                    const struct GNUNET_PeerIdentity *peer,
                    const struct GNUNET_HELLO_Message *hello,
                    const char *err_msg)
{
  GtkTreeIter iter;
  struct PeerInfo *info;
  GtkTreePath *path;

  info = GNUNET_CONTAINER_multipeermap_get (peer2info, peer);
  if (NULL == info)
  {
    info = GNUNET_new (struct PeerInfo);
    info->pid = *peer;
    gtk_list_store_append (ls, &iter);
    gtk_list_store_set (ls, &iter,
			PEERINFO_MC_PEER_IDENTITY_STRING, GNUNET_i2s (peer),
			PEERINFO_MC_NUMBER_OF_ADDRESSES, (guint) 0,
                        PEERINFO_MC_COUNTRY_NAME, "",
                        PEERINFO_MC_COUNTRY_FLAG, NULL,
                        PEERINFO_MC_BANDWIDTH_IN, (guint64) 0,
                        PEERINFO_MC_BANDWIDTH_OUT, (guint64) 0,
			PEERINFO_MC_ADDRESS_AS_STRING, "",
			PEERINFO_MC_CONNECTIVITY_LED, led_red,
			PEERINFO_MC_CONNECTED_STATUS, FALSE,
                        PEERINFO_MC_IS_FRIEND, GNUNET_CONTAINER_multipeermap_contains (friends,
                                                                                       peer),
                        PEERINFO_MC_PEERINFO, info,
                        -1);
    path = gtk_tree_model_get_path (GTK_TREE_MODEL (ls), &iter);
    info->rr = gtk_tree_row_reference_new (GTK_TREE_MODEL (ls), path);
    GNUNET_assert (NULL != info->rr);
    gtk_tree_path_free (path);
    GNUNET_CONTAINER_multipeermap_put (peer2info, peer, info,
                                       GNUNET_CONTAINER_MULTIHASHMAPOPTION_UNIQUE_ONLY);
  }
  if (NULL == info->palc)
  {
    info->palc =
        GNUNET_TRANSPORT_peer_get_active_addresses (get_configuration (), peer,
                                                    GNUNET_NO,
                                                    GNUNET_TIME_UNIT_FOREVER_REL,
                                                    &peer_address_cb, info);
  }
}


/**
 * Method called whenever a given peer has a QoS status change.
 *
 * @param cls closure
 * @param address the address
 * @param address_active is this address actively used to maintain a connection
 * 				to a peer
 * @param bandwidth_in available amount of inbound bandwidth
 * @param bandwidth_out available amount of outbound bandwidth
 * @param ats performance data for the address (as far as known)
 * @param ats_count number of performance records in @a ats
 */
static void
status_cb (void *cls,
	   const struct GNUNET_HELLO_Address *address,
	   int address_active,
           struct GNUNET_BANDWIDTH_Value32NBO bandwidth_in,
           struct GNUNET_BANDWIDTH_Value32NBO bandwidth_out,
           const struct GNUNET_ATS_Information *ats,
	   uint32_t ats_count)
{
  struct PeerInfo *info;
  GtkTreeIter iter;
  GtkTreePath *path;

  if (0 == address_active)
    return;
  info = GNUNET_CONTAINER_multipeermap_get (peer2info, &address->peer);
  if (NULL == info)
  {
    /* should rarely happen... */
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
                "Entry for peer %s not found\n",
                GNUNET_i2s (&address->peer));
    return;
  }
  path = gtk_tree_row_reference_get_path (info->rr);
  GNUNET_assert (NULL != path);
  GNUNET_assert (TRUE == gtk_tree_model_get_iter (GTK_TREE_MODEL (ls), &iter, path));
  gtk_tree_path_free (path);
  gtk_list_store_set (ls, &iter,
		      PEERINFO_MC_BANDWIDTH_IN, (guint) ntohl (bandwidth_in.value__),
		      PEERINFO_MC_BANDWIDTH_OUT, (guint) ntohl (bandwidth_out.value__),
		      -1);
}


/**
 * Write a friend to the friends file.
 *
 * @param cls the `struct GNUNET_FRIENDS_Writer`
 * @param friend friend to write to file
 * @param value unused
 * @return #GNUNET_OK if the writing succeeded
 */
static int
write_friend (void *cls,
              const struct GNUNET_PeerIdentity *friend,
              void *value)
{
  struct GNUNET_FRIENDS_Writer *w = cls;

  return GNUNET_FRIENDS_write (w, friend);
}


/**
 * Write an updated friends file out to disk.
 */
static void
write_friends ()
{
  struct GNUNET_FRIENDS_Writer *w;

  w = GNUNET_FRIENDS_write_start (get_configuration ());
  if (NULL == w)
  {
    GNUNET_break (0);
    return;
  }
  GNUNET_CONTAINER_multipeermap_iterate (friends,
                                         &write_friend,
                                         w);
  if (GNUNET_OK !=
      GNUNET_FRIENDS_write_stop (w))
  {
    GNUNET_break (0);
    return;
  }
}


/**
 * The user has toggled the 'is friend' column for one of the peers.
 * Update everything.
 *
 * @param cell_renderer the cell renderer that issued the toggle signal
 * @param path which cell was toggled
 * @param user_data our main window builder
 */
void
GNUNET_PEERINFO_GTK_main_window_friends_cellrenderertoggle_toggled_cb (GtkCellRendererToggle *cell_renderer,
                                                                       gchar *path,
                                                                       gpointer user_data)
{
  GtkListStore *ls;
  GtkTreeIter old;
  struct PeerInfo *info;
  gboolean oldvalue;

  ls = GTK_LIST_STORE (get_object ("GNUNET_PEERINFO_GTK_list_store"));
  if (NULL == ls)
  {
    GNUNET_break (0);
    return;
  }
  if (TRUE !=
      gtk_tree_model_get_iter_from_string (GTK_TREE_MODEL (ls), &old, path))
  {
    GNUNET_break (0);
    return;
  }
  gtk_tree_model_get (GTK_TREE_MODEL (ls), &old,
                      PEERINFO_MC_PEERINFO, &info,
                      PEERINFO_MC_IS_FRIEND, &oldvalue,
                      -1);
  gtk_list_store_set (ls, &old,
                      PEERINFO_MC_IS_FRIEND, ! oldvalue,
                      -1);
  if (oldvalue)
  {
    GNUNET_break (1 ==
                  GNUNET_CONTAINER_multipeermap_remove_all (friends,
                                                            &info->pid));
  }
  else
  {
    GNUNET_break (GNUNET_YES ==
                  GNUNET_CONTAINER_multipeermap_put (friends,
                                                     &info->pid,
                                                     "true",
                                                     GNUNET_CONTAINER_MULTIHASHMAPOPTION_UNIQUE_ONLY));

  }
  write_friends ();
}


/**
 * Callback invoked if the application is supposed to exit.
 *
 * @param object
 * @param user_data unused
 */
void
GNUNET_PEERINFO_GTK_quit_cb (GObject * object, gpointer user_data)
{
  GNUNET_SCHEDULER_shutdown ();
}


/**
 * Load LED image from resource file.
 *
 * @param color color of the LED to load
 * @return the image as a GdkPixbuf
 */
static GdkPixbuf *
load_led (const char *color)
{
  GdkPixbuf *pixbuf;
  char *dir;
  char *fn;

  dir = GNUNET_GTK_installation_get_path (GNUNET_OS_IPK_DATADIR);
  GNUNET_asprintf (&fn, "%s%s.png", dir,
                   color);
  GNUNET_free (dir);
  pixbuf = gdk_pixbuf_new_from_file (fn, NULL);
  GNUNET_free (fn);
  return pixbuf;
}


/**
 * Add a friend to our friends peer map.
 *
 * @param cls NULL
 * @param friend the friend to add
 */
static void
add_friend (void *cls,
            const struct GNUNET_PeerIdentity *friend)
{
  GNUNET_break (GNUNET_YES ==
                GNUNET_CONTAINER_multipeermap_put (friends,
                                                   friend,
                                                   "true",
                                                   GNUNET_CONTAINER_MULTIHASHMAPOPTION_UNIQUE_ONLY));
}


/**
 * Actual main function run right after GNUnet's scheduler
 * is initialized.  Initializes up GTK and Glade.
 *
 * @param cls NULL
 * @param tc schedule context
 */
static void
run (void *cls, const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  GtkWidget *main_window;

  ml = cls;
  if (GNUNET_OK !=
      GNUNET_GTK_main_loop_build_window (ml, NULL))
    return;
  led_green = load_led ("green");
  led_red = load_led ("red");
  GNUNET_GTK_set_icon_search_path ();
  GNUNET_GTK_setup_nls ();
  friends = GNUNET_CONTAINER_multipeermap_create (128, GNUNET_NO);
  if (GNUNET_OK !=
      GNUNET_FRIENDS_parse (get_configuration (),
                            &add_friend,
                            NULL))
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                _("Failed to parse list of friends\n"));
  peer2info = GNUNET_CONTAINER_multipeermap_create (256, GNUNET_NO);
  pnc =
      GNUNET_PEERINFO_notify (get_configuration (),
			      GNUNET_NO,
			      &peerinfo_processor, NULL);
  if (NULL == pnc)
  {
    fprintf (stderr,
             _("Failed to initialize communication with peerinfo service!\n"));
    GNUNET_SCHEDULER_shutdown ();
    return;
  }
  ats = GNUNET_ATS_performance_init (get_configuration (), &status_cb, NULL);
  /* setup main window */
  main_window = GTK_WIDGET (get_object ("GNUNET_PEERINFO_GTK_main_window"));
  main_window = GNUNET_GTK_plug_me ("GNUNET_PEERINFO_GTK_PLUG",
                                    main_window);
  ls = GTK_LIST_STORE (get_object ("GNUNET_PEERINFO_GTK_list_store"));
  GNUNET_assert (NULL != ls);
  gtk_window_maximize (GTK_WINDOW (main_window));
  if (NULL == getenv ("GNUNET_PEERINFO_GTK_PLUG"))
    GNUNET_GTK_tray_icon_create (ml,
                                 GTK_WINDOW (main_window),
                                 "gnunet-gtk" /* FIXME: different icon? */ ,
                                 "gnunet-peerinfo-gtk");

#if HAVE_LIBUNIQUE
  unique_app_watch_window (unique_app, GTK_WINDOW (main_window));
#endif
  /* make GUI visible */
  if (!tray_only)
  {
    gtk_widget_show (main_window);
    gtk_window_present (GTK_WINDOW (main_window));
  }
  GNUNET_SCHEDULER_add_delayed (GNUNET_TIME_UNIT_FOREVER_REL,
				&shutdown_task, NULL);
}


/**
 * Main function of gnunet-peerinfo-gtk.
 *
 * @param argc number of arguments
 * @param argv arguments
 * @return 0 on success
 */
int
main (int argc, char **argv)
{
  static struct GNUNET_GETOPT_CommandLineOption options[] = {
    {'t', "tray", NULL,
     gettext_noop ("start in tray mode"), 0,
     &GNUNET_GETOPT_set_one, &tray_only},
    GNUNET_GETOPT_OPTION_END
  };

#if HAVE_LIBUNIQUE
  gtk_init (&argc, &argv);
  unique_app = unique_app_new ("org.gnunet.gnunet-peerinfo-gtk", NULL);
  if (unique_app_is_running (unique_app))
  {
    UniqueResponse response;

    response = unique_app_send_message (unique_app, UNIQUE_ACTIVATE, NULL);
    g_object_unref (unique_app);
    return (UNIQUE_RESPONSE_OK == response) ? 0 : 1;
  }
#endif

  if (GNUNET_OK !=
      GNUNET_GTK_main_loop_start ("gnunet-peerinfo-gtk",
                                  "GTK GUI for inspecting GNUnet Peers", argc,
                                  argv, options,
                                  "gnunet_peerinfo_gtk_main_window.glade",
                                  &run))
    return 1;

#if HAVE_LIBUNIQUE
  g_object_unref (unique_app);
#endif
  return 0;
}


/* end of gnunet-peerinfo-gtk.c */
