#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2003-2009 Free Software Foundation
#
# $Id: Writer.py 6764 2004-12-01 00:34:26Z jcater $

__all__ = ['Writer']


import string
from PrinterDef import PrinterDefinition


class Writer:

  def __init__(self, formatter):
    self.formatter = formatter
    self.printdef = PrinterDefinition(formatter)
    
  def initialize(self, report):
    self.reportHeight = report.height
    self.reportWidth  = report.width

    self.pageNumber = 1
    self.currLine = 0

    self.pageHeaderDone = False
    self.pageFooterDone = False

    if self.formatter == "xml":
      self.output.write(self.printdef['job_begin'] % gConfig('textEncoding'))
    else:
      self.output.write(self.printdef['job_begin'])


  def finalize(self):
    if self.pageNumber > 1: 
      self.output.write(self.printdef['page_end'])
    self.output.write(self.printdef['job_end'])


  def renderBand(self, band): 
#    self.output.write("[[ Begin Band %s ]]" % band._type)
    for line in band.buffer: 
      line.sort()
      ll = len(line)
      cx = 0
      outputLine = ''
      # Make sure fields don't overlap
      for i in xrange(ll): 
        x, text, prefix, postfix = line[i]
        if i < ll - 1: 
          text = text[:line[i+1][0] - x]
        # Move us to correct X position
        if cx < x: 
          if self.formatter != "xml":
            outputLine += self.printdef.escape(' '*(x-cx))
          cx = x

        if self.formatter == "xml" and prefix[:6] == '<label':
          prefix = prefix % cx

        outputLine += prefix + \
                      self.printdef.escape(text).encode(gConfig('textEncoding'),'replace') + \
                      postfix
        cx += len(text)

      self.printLine(outputLine, band._type)
#    self.output.write("[[ End Band %s ]]" % band._type)

    
  def render(self, object, x, y, width, band,
                  align='none', condensed=False, bold=False,
                  italic=False, underline=False):

    try:
      curline = band.buffer[y]
    except: 
      curline = band.buffer[y] = []
    
    prefix = self.printdef['label_begin']
    # start style
    if condensed:
      prefix += self.printdef['condensed_begin']
    if bold:
      prefix += self.printdef['bold_begin']
    if italic:
      prefix += self.printdef['italic_begin']
    if underline:
      prefix += self.printdef['underline_begin']


    # end style... these should be the 
    # reverse of the "Start" lines above.
    postfix = ""
    if underline:
      postfix += self.printdef['underline_end']
    if italic:
      postfix += self.printdef['italic_end']
    if bold:
      postfix += self.printdef['bold_end']
    if condensed:
      postfix += self.printdef['condensed_end']

    postfix += self.printdef['label_end']
    
    curline.append((x, _alignFunc[align](object[:width], width), prefix, postfix))

    
  def newPage(self):
    if self.pageNumber > 1: 
      self.output.write(self.printdef['page_end'])
    self.pageNumber += 1
    self.currLine = 0
    self.pageHeaderDone = False
    self.pageFooterDone = False

    self.output.write(self.printdef['page_begin'])


  def printLine(self, line, bandtype): 
    if self.PageFooter and (not self.pageFooterDone):
      if self.currLine + 1 > self.reportHeight - self.PageFooter.height:
        self.PageFooter.chProcess(self)
        self.newPage()

    if self.PageHeader and (not self.pageHeaderDone) and bandtype != 'CHSummaryPage':
      self.PageHeader.chProcess(self)

    fullline = self.printdef['line_begin'] +  \
                     line +  \
                     self.printdef['line_end']
    if self.formatter == "xml":
      fullline = fullline % self.currLine

    self.output.write(fullline)
    self.currLine += 1


  def printLines(self, lines, bandtype): 
    for line in lines:
      if self.formatter == "xml":
        self.currLine += 1
      else:
        self.printLine(line, bandtype)
      
  
_alignFunc = {'left'   : lambda obj, width: string.ljust(obj, width),
              'right'  : lambda obj, width: string.rjust(obj, width),
              'center' : lambda obj, width: string.center(obj, width),
              'none'   : lambda obj, width: string.ljust(obj, width),
             }


