/* inode-sig.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/char/str.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/vu/safe.h"
#include "tla/libfsutils/ensure-dir.h"
#include "tla/libfsutils/dir-listing.h"
#include "tla/libfsutils/string-files.h"
#include "tla/libawk/relational.h"
#include "tla/libawk/relassoc.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/invent.h"
#include "tla/libarch/inode-sig.h"



#define MAX_INODE_SIG_FILES 5


/* __STDC__ prototypes for static functions */
static void inode_sig_callback (t_uchar * path, struct stat * stat_buf,
                                enum arch_inventory_category category,
                                t_uchar * tag, int has_source_name,
                                void * closure);
static t_uchar * arch_inode_sig_dir (t_uchar * tree_root);
static t_uchar * arch_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision);
static t_uchar * arch_inode_sig_tmp_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision);
static int arch_creat_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision);
static void arch_finish_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision, int fd);
static void arch_prune_inode_sig_dir (t_uchar * tree_root);



t_uchar *
arch_statb_inode_sig (struct stat * statb)
{
  int fd = make_output_to_string_fd ();

  safe_printfmt (fd, "dev=%lu:ino=%lu:mtime=%lu:ctime=%lu:size=%lu",
                 (t_ulong)statb->st_dev,
                 (t_ulong)statb->st_ino,
                 (t_ulong)statb->st_mtime,
                 (t_ulong)statb->st_ctime,
                 (t_ulong)statb->st_size);

  return string_fd_close (fd);
}


rel_table
arch_tree_inode_sig (t_uchar * tree_root)
{
  struct arch_inventory_options options = {0, };
  int here_fd;
  rel_table answer = 0;

  here_fd = safe_open (".", O_RDONLY, 0);
  safe_chdir (tree_root);

  options.categories = arch_inventory_source;
  options.want_tags = 1;
  options.include_excluded = 1;

  arch_get_inventory_naming_conventions (&options, ".");

  arch_inventory_traversal (&options, ".", inode_sig_callback, (void *)&answer);

  arch_free_inventory_naming_conventions (&options);

  rel_sort_table_by_field (0, answer, 0);

  safe_fchdir (here_fd);
  safe_close (here_fd);

  return answer;
}

void
arch_snap_inode_sig (t_uchar * tree_root, t_uchar * archive, t_uchar * revision)
{
  rel_table sig = arch_tree_inode_sig (tree_root);
  int fd = arch_creat_inode_sig_file (tree_root, archive, revision);

  rel_print_table (fd, sig);
  arch_finish_inode_sig_file (tree_root, archive, revision, fd);

  rel_free_table (sig);
}


void
arch_read_inode_sig (rel_table * as_table, assoc_table * as_assoc,
                     t_uchar * tree_root, t_uchar * archive, t_uchar * revision)
{
  t_uchar * sig_file = arch_inode_sig_file (tree_root, archive, revision);

  if (as_table)
    *as_table = 0;

  if (as_assoc)
    *as_assoc = 0;

  if (!safe_access (sig_file, F_OK))
    {
      int fd = safe_open (sig_file, O_RDONLY, 0);
      rel_table the_table;

      the_table = rel_read_table (fd, 2, "arch", sig_file);

      if (as_assoc)
        *as_assoc = rel_to_assoc (the_table, 0, 1);

      if (as_table)
        *as_table = the_table;
      else
        rel_free_table (the_table);

      safe_close (fd);
    }
}


void
arch_read_tag_shortcut (assoc_table * tags_shortcut, t_uchar * tree_root)
{
  t_uchar * inode_sig_dir = arch_inode_sig_dir (tree_root);
  rel_table dir_listing = maybe_directory_files (inode_sig_dir);
  assoc_table answer = 0;
  int x;

  for (x = 0; x < rel_n_records (dir_listing); ++x)
    {
      t_uchar * path = file_name_in_vicinity (0, inode_sig_dir, dir_listing[x][0]);
      t_uchar * pct = str_chr_index (dir_listing[x][0], '%');

      if (pct && str_cmp (".", dir_listing[x][0]) && str_cmp ("..", dir_listing[x][0]))
        {
          t_uchar * maybe_archive = str_save_n (0, dir_listing[x][0], pct - dir_listing[x][0]);
          t_uchar * maybe_revision = str_save (0, pct + 1);

          if (arch_valid_archive_name (maybe_archive) && arch_valid_package_name (maybe_revision, arch_no_archive, arch_req_patch_level, 0))
            {
              rel_table sig_table = 0;
              int y;

              arch_read_inode_sig (&sig_table, 0, tree_root, maybe_archive, maybe_revision);

              for (y = 0; y < rel_n_records (sig_table); ++y)
                {
                  t_uchar * this_tag = sig_table[y][0];
                  t_uchar * this_sig = sig_table[y][1];

                  {
                    t_uchar * already_have = assoc_ref (answer, this_sig);

                    if (already_have)
                      {
                        invariant (!str_cmp (this_tag, already_have));
                      }
                    else
                      {
                        assoc_set (&answer, this_sig, this_tag);
                      }
                  }
                  lim_free (0, this_tag);
                  lim_free (0, this_sig);
                }
            }

          lim_free (0, maybe_archive);
          lim_free (0, maybe_revision);
        }

      lim_free (0, path);
    }

  lim_free (0, inode_sig_dir);
  rel_free_table (dir_listing);
  *tags_shortcut = answer;
}




static void
inode_sig_callback (t_uchar * path, struct stat * stat_buf,
                    enum arch_inventory_category category,
                    t_uchar * tag, int has_source_name,
                    void * closure)
{
  rel_table * answer = (rel_table *)closure;
  t_uchar * signature = arch_statb_inode_sig (stat_buf);

  if (!S_ISDIR (stat_buf->st_mode) && !S_ISLNK (stat_buf->st_mode))
    rel_add_records (answer, rel_make_record (tag, signature, 0), 0);
}


static t_uchar *
arch_inode_sig_dir (t_uchar * tree_root)
{
  return file_name_in_vicinity (0, tree_root, "{arch}/,,inode-sigs");
}

static t_uchar *
arch_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision)
{
  t_uchar * inode_sig_dir = arch_inode_sig_dir (tree_root);
  t_uchar * basename = str_alloc_cat_many (0, archive, "%", revision, str_end);
  t_uchar * answer = file_name_in_vicinity (0, inode_sig_dir, basename);

  lim_free (0, inode_sig_dir);
  lim_free (0, basename);
  return answer;
}


static t_uchar *
arch_inode_sig_tmp_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision)
{
  t_uchar * inode_sig_dir = arch_inode_sig_dir (tree_root);
  t_uchar * basename = str_alloc_cat_many (0, ",,", archive, "%", revision, str_end);
  t_uchar * answer = file_name_in_vicinity (0, inode_sig_dir, basename);

  lim_free (0, inode_sig_dir);
  lim_free (0, basename);
  return answer;
}


static int
arch_creat_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision)
{
  t_uchar * inode_sig_dir = arch_inode_sig_dir (tree_root);
  t_uchar * inode_sig_tmp_file = arch_inode_sig_tmp_file (tree_root, archive, revision);
  int ign;
  int answer;

  ensure_directory_exists (inode_sig_dir);
  vu_unlink (&ign, inode_sig_tmp_file);
  answer = safe_open (inode_sig_tmp_file, (O_WRONLY | O_CREAT | O_EXCL), 0444);

  lim_free (0, inode_sig_dir);
  lim_free (0, inode_sig_tmp_file);

  return answer;
}

static void
arch_finish_inode_sig_file (t_uchar * tree_root, t_uchar * archive, t_uchar * revision, int fd)
{
  t_uchar * inode_sig_tmp_file = arch_inode_sig_tmp_file (tree_root, archive, revision);
  t_uchar * inode_sig_file = arch_inode_sig_file (tree_root, archive, revision);

  safe_close (fd);
  safe_rename (inode_sig_tmp_file, inode_sig_file);
  arch_prune_inode_sig_dir (tree_root);

  lim_free (0, inode_sig_tmp_file);
  lim_free (0, inode_sig_file);
}


static void
arch_prune_inode_sig_dir (t_uchar * tree_root)
{
  t_uchar * inode_sig_dir = arch_inode_sig_dir (tree_root);
  rel_table dir_listing = maybe_directory_files (inode_sig_dir);
  int dead_one = -1;
  time_t oldest_time;
  int n_inode_sig_files;
  int x;


  do
    {
      n_inode_sig_files = 0;

      for (x = 0; x < rel_n_records (dir_listing); ++x)
        {
          t_uchar * path = file_name_in_vicinity (0, inode_sig_dir, dir_listing[x][0]);
          t_uchar * pct = str_chr_index (dir_listing[x][0], '%');

          if (pct && str_cmp (".", dir_listing[x][0]) && str_cmp ("..", dir_listing[x][0]))
            {
              t_uchar * maybe_archive = str_save_n (0, dir_listing[x][0], pct - dir_listing[x][0]);
              t_uchar * maybe_revision = str_save (0, pct + 1);

              if (arch_valid_archive_name (maybe_archive) && arch_valid_package_name (maybe_revision, arch_no_archive, arch_req_patch_level, 0))
                {
                  struct stat statb;

                  ++n_inode_sig_files;

                  safe_lstat (path, &statb);

                  if ((dead_one < 0) || (statb.st_ctime < oldest_time))
                    {
                      dead_one = x;
                      oldest_time = statb.st_ctime;
                    }
                }

              lim_free (0, maybe_archive);
              lim_free (0, maybe_revision);
            }

          lim_free (0, path);
        }

      if (n_inode_sig_files > MAX_INODE_SIG_FILES)
        {
          t_uchar * dead_path = file_name_in_vicinity (0, inode_sig_dir, dir_listing[dead_one][0]);

          safe_unlink (dead_path);
          --n_inode_sig_files;

          lim_free (0, dead_path);
        }


    } while (n_inode_sig_files > MAX_INODE_SIG_FILES); /* usually only one iteration expected */

  lim_free (0, inode_sig_dir);
  rel_free_table (dir_listing);
}






/* tag: Tom Lord Fri Sep 12 09:56:44 2003 (inode-sig.c)
 */
