/* cmd-star-merge.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/star-merge.h"
#include "tla/libarch/cmd-star-merge.h"



static t_uchar * usage = "[options] FROM TREE [VERSION]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_archive, "A", "archive", 1, \
      "Override `my-default-archive'") \
  OP (opt_changes, "c", "changes OUTPUT", 1, \
      "Generate but don't apply the changeset.") \


t_uchar arch_cmd_star_merge_help[] = ("merge mutually merged branches\n"
                                      "Merge changes from FROM into TREE, considering common ancestry\n"
                                      "in VERSION (which defaults to the default tree version of TREE).\n"
                                      "\n"
                                      "The purpose of this command is to merge changes between development\n"
                                      "lines which may have merged in both directions.   It uses merge history\n"
                                      "to attempt to minimize the spurious conflicts that can arise from `mutual\n"
                                      "merging'.\n"
                                      "\n"
                                      "FROM indicates a revision (it may be specified as a version name,\n"
                                      "indicating the latest revision in that version).\n"
                                      "\n"
                                      "TREE is the project tree into which the merge will occur.\n"
                                      "\n"
                                      "Star-merge works by computing the most recent ANCESTOR revision of TREE and\n"
                                      "FROM (with respect to VERSION) and then applying the changeset:\n"
                                      "\n"
                                      "         delta (ANCESTOR, FROM)\n"
                                      "\n"
                                      "to TREE.\n"
                                      "\n"
                                      "The \"most recent ancestor\" is defined as follows:\n"
                                      "\n"
                                      "MAYBE_ANCESTOR_1 is defined as the highest patch level in the version of FROM\n"
                                      "for which both TREE and FROM have a patch log.   In other words, it is the\n"
                                      "latest revision of FROM's version already merged into TREE.\n"
                                      "\n"
                                      "MAYBE_ANCESTOR_2 is defined as the highest patch level in VERSION for which both FROM\n"
                                      "and TREE have a patch log.   In other words, it is the latest revision of VERSION\n"
                                      "already merged into FROM.\n"
                                      "\n"
                                      "MAYBE_ANCESTOR_2, if it is not \"nil\", was merged into FROM at some revision\n"
                                      "of FROM's version, which we can call LAST_MERGE_INTO_FROM.\n"
                                      "\n"
                                      "If both MAYBE_ANCESTOR_1 or MAYBE_ANCESTOR_2 are nil, star-merge can do nothing.\n"
                                      "\n"
                                      "If just one of MAYBE_ANCESTOR_1 is MAYBE_ANCESTOR_2 is not nil, then that\n"
                                      "non-nil value is ANCESTOR.\n"
                                      "\n"
                                      "If both MAYBE_ANCESTOR_1 and MAYBE_ANCESTOR_2 are not nil, then LAST_MERGE_INTO_FROM\n"
                                      "and and MAYBE_ANCESTOR_1 are compared (both are revisions in FROM's version).  If\n"
                                      "MAYBE_ANCESTOR_1 is the later revision, then MAYBE_ANCESTOR_1 is ANCESTOR, otherwise,\n"
                                      "MAYBE_ANCESTOR_2 is ANCESTOR.\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_star_merge (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  int exit_status = 2;
  t_uchar * changeset = 0;

  default_archive = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_star_merge_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_changes:
          {
            lim_free (0, changeset);
            changeset = str_save (0, option->arg_string);
            break;
          }
        }
    }

  if ((argc < 3) || (argc > 4))
    goto usage_error;

  {
    t_uchar * from_spec = 0;
    t_uchar * from_archive = 0;
    struct arch_archive * from_arch = 0;
    t_uchar * from_revision = 0;
    t_uchar * tree_spec = 0;
    t_uchar * tree_root = 0;
    t_uchar * to_version_spec = 0;
    t_uchar * to_archive = 0;
    t_uchar * to_version = 0;
    t_uchar * cache_dir = 0;


    from_spec = str_save (0, argv[1]);

    if (!arch_valid_package_name (from_spec, arch_maybe_archive, arch_req_version, 1))
      {
        safe_printfmt (2, "star-merge: invalid FROM revision spec (%s)\n", from_spec);
        exit (2);
      }

    from_archive = arch_parse_package_name (arch_ret_archive, default_archive, from_spec);
    from_arch = arch_archive_connect (from_archive, 0);

    if (arch_valid_package_name (from_spec, arch_maybe_archive, arch_req_patch_level, 0))
      {
        from_revision = arch_parse_package_name (arch_ret_non_archive, 0, from_spec);
      }
    else
      {
        t_uchar * from_version = 0;
        rel_table revisions = 0;

        from_version = arch_parse_package_name (arch_ret_non_archive, 0, from_spec);

        revisions = arch_archive_revisions (from_arch, from_version, 2);

        if (!revisions)
          {
            safe_printfmt (2, "star-merge: FROM version has no revisions (%s/%s)\n", from_archive, from_version);
            exit (2);
          }

        from_revision = str_save (0, revisions[rel_n_records (revisions) - 1][0]);

        lim_free (0, from_version);
        rel_free_table (revisions);
      }


    tree_spec = str_save (0, argv[2]);
    tree_root = arch_tree_root (0, tree_spec, 0);

    if (!tree_root)
      {
        safe_printfmt (2, "star-merge: TREE is not within a project tree (%s)\n", tree_spec);
        exit (2);
      }

    if (argc == 4)
      {
        to_version_spec = str_save (0, argv[3]);
      }
    else
      {
        to_version_spec = arch_tree_version (tree_root);

        if (!to_version_spec)
          {
            safe_printfmt (2, "star-merge: TREE has no default version (%s)\n", tree_root);
            exit (2);
          }
      }

    if (!arch_valid_package_name (to_version_spec, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "star-merge: invalid VERSION spec (%s)\n", to_version_spec);
        exit (2);
      }

    to_archive = arch_parse_package_name (arch_ret_archive, default_archive, to_version_spec);
    to_version = arch_parse_package_name (arch_ret_non_archive, 0, to_version_spec);

    cache_dir = file_name_directory_file (0, tree_root);

    exit_status = arch_star_merge (1, from_arch, from_archive, from_revision, tree_root, 0, to_archive, to_version, cache_dir, changeset);

    lim_free (0, from_spec);
    lim_free (0, from_archive);
    arch_archive_close (from_arch);
    lim_free (0, from_revision);

    lim_free (0, tree_spec);
    lim_free (0, tree_root);

    lim_free (0, to_version_spec);
    lim_free (0, to_archive);
    lim_free (0, to_version);

    lim_free (0, cache_dir);
  }



  lim_free (0, default_archive);

  return exit_status;
}




/* tag: Tom Lord Sat Jun 28 20:22:10 2003 (cmd-star-merge.c)
 */
