/* cmd-deltapatch.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */




#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "tla/libfsutils/tmp-files.h"
#include "tla/libfsutils/rmrf.h"
#include "tla/libfsutils/dir-as-cwd.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/my.h"
#include "tla/libarch/archive.h"
#include "tla/libarch/pristines.h"
#include "tla/libarch/local-cache.h"
#include "tla/libarch/make-changeset.h"
#include "tla/libarch/copy-project-tree.h"
#include "tla/libarch/apply-changeset.h"
#include "tla/libarch/cmd.h"
#include "tla/libarch/cmd-getrev.h"
#include "tla/libarch/cmd-revdelta.h"
#include "tla/libarch/cmd-deltapatch.h"


/* __STDC__ prototypes for static functions */
static void deltapatch_callback (void * vfd, char * fmt, va_list ap);



static t_uchar * usage = "[options] FROM(REVISION|DIR) TO(REVISION|DIR) ORIGDIR DESTDIR\n";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_archive, "A", "archive", 1, \
      "Override `my-default-archive'") \
  OP (opt_cache, 0, "cache DIR", 1, \
      "specify a cache root for pristine copies") \
  OP (opt_in_place, 0, "in-place", 0, \
      "modify UPON directly")


t_uchar arch_cmd_deltapatch_help[] = ("Compute a changeset between any two trees or revisions\n"
                                      "and apply them to a project tree\n"
                                      "\n"
                                      "A delta between A and B (both of which may be either a\n"
                                      "full revision or a project tree) is computed. The delta\n"
                                      "is then applied to ORIGDIR, the results of which are\n"
                                      "stored in DESTDIR\n"
                                      "\n"
                                      "DESTDIR is optional if --in-place is specified, resulting\n"
                                      "in ORIGDIR being patched instead\n"
                                      "\nExit Status Codes:"
                                      "                     0 No conflict during patch\n"
                                      "                     1 Conflicts occurred during patch\n"
                                      "                     3 Internal Error\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_deltapatch (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * cache_dir = 0;
  int in_place = 0;
  int exit_status = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_deltapatch_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_cache:
          {
            lim_free (0, cache_dir);
            cache_dir = str_save (0, option->arg_string);
            break;
          }

        case opt_in_place:
          {
            in_place = 1;
            break;
          }
        }
    }

  if (in_place ? (argc != 4) : (argc != 5))
    goto usage_error;

  if (default_archive && !arch_valid_archive_name (default_archive))
    {
      safe_printfmt (2, "deltapatch: invalid archive name (%s)\n", default_archive);
      exit (1);
    }

  default_archive = arch_my_default_archive (default_archive);

  {
    t_uchar * from_spec;
    t_uchar * to_spec;
    t_uchar * upon_spec;
    t_uchar * dest;
    t_uchar * dest_dir = 0;
    t_uchar * changeset = 0;

    from_spec = argv[1];
    to_spec = argv[2];
    upon_spec = argv[3];

    if (in_place)
      {
        dest = upon_spec;
      }
    else
      {
        dest = argv[4];
      }

    dest_dir = file_name_directory_file (0, dest);
    if (!dest_dir)
      dest_dir = str_save (0, ".");

    if (in_place)
      {
        changeset = tmp_file_name (dest, ",,deltapatch-changeset");
      }
    else
      {
        changeset = tmp_file_name (dest_dir, ",,deltapatch-changeset");

        if (arch_valid_package_name (upon_spec, arch_maybe_archive, arch_req_version, 1))
          {
            if (cache_dir)
              arch_call_cmd (arch_cmd_getrev, "deltapatch", "--cache", cache_dir, upon_spec, dest, 0);
            else
              arch_call_cmd (arch_cmd_getrev, "deltapatch", upon_spec, dest, 0);
          }
        else
          {
            safe_printfmt (1, "* copying %s to %s\n", upon_spec, dest);
            arch_copy_project_tree (upon_spec, dest, 1, 1);
          }
      }

    if (cache_dir)
      arch_call_cmd (arch_cmd_revdelta, "deltapatch", "--output", changeset, "--cache", cache_dir, from_spec, to_spec, 0);
    else
      arch_call_cmd (arch_cmd_revdelta, "deltapatch", "--output", changeset, from_spec, to_spec, 0);

    {
      struct arch_apply_changeset_report r = {0, };

      r.callback = deltapatch_callback;
      r.thunk = (void *)1;

      safe_printfmt (1, "* applying changeset\n");
      arch_apply_changeset (&r, changeset, dest, arch_unspecified_tagging, 0);

      exit_status = arch_conflicts_occured (&r);

      arch_free_apply_changeset_report_data (&r);
    }

    rmrf_file (changeset);

    lim_free (0, dest_dir);
    lim_free (0, changeset);
  }

  lim_free (0, default_archive);
  lim_free (0, cache_dir);
  return exit_status;
}




static void
deltapatch_callback (void * vfd, char * fmt, va_list ap)
{
  int fd;

  fd = (int)(t_ulong)vfd;
  safe_printfmt_va_list (fd, fmt, ap);
  safe_flush (fd);
}




/* tag: Tom Lord Wed Jun  4 15:40:44 2003 (cmd-deltapatch.c)
 */
