/* glpsol.c (stand-alone LP/MIP solver) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include "glpk.h"
#include "glplib.h"

/*----------------------------------------------------------------------
-- This program is a stand-alone LP/MIP solver. For pure LP problems
-- either the simplex method or the primal-dual interior point method
-- can be used. For MIP problems the branch-and-bound procedure based
-- on the simplex method is used.
----------------------------------------------------------------------*/

static char *version = "GLPSOL -- GLPK LP/MIP Solver, Version 3.2";
/* the version string */

static char *in_fname = NULL;
/* name of the input text file */

static int format = 'M';
/* format of the input text file:
   'M' - MPS
   'L' - GLPK/L */

static int check = 0;
/* if this flag is set, only input data checking is required */

static char *fn_gener = NULL;
/* name of the output text file, to which generated LP/MIP problem
   should be written in plain text format; NULL means no output */

static char *fn_write = NULL;
/* name of the output text file, to which LP/MIP problem should be
   written using MPS format; NULL means no output */

static int obj_dir = 0;
/* optimization direction flag:
   0       - not specified
   LPX_MIN - minimization
   LPX_MAX - maximization */

static int method = 'S';
/* the method which should be used for solving the problem:
   'S' - simplex method
   'T' - interior point method */

static char *out_fname = NULL;
/* name of the output text file to which the final solution should be
   sent using printable format (optional) */

static int scale = 1;
/* if this flag is set, the solver performs automatic scaling the
   problem before solving; otherwise scaling is not used */

static int basis = 1;
/* this parameter specifies initial basis option:
   0 - use standard initial basis
   1 - use advanced initial basis */

static int price = 1;
/* this parameter specifies pricing option:
   0 - use textbook pricing
   1 - use steepest edge pricing */

static int relax = 1;
/* if this flag is set, the solver uses two-pass ratio test (for both
   primal and dual simplex) proposed by P.Harris; otherwise the solver
   uses the standard "textbook" ratio test */

static int nomip = 0;
/* if this flag is set, the solver considers all integer variables as
   continuous (this allows solving MIP problem as pure LP) */

static int branch = 2;
/* this parameter specifies what branching technique should be used by
   the solver:
   0 - branch on the first integer variable
   1 - branch on the last integer variable
   2 - branch using heuristic by Driebeek and Tomlin */

static int btrack = 2;
/* this parameter specifies what backtracking technique should be used
   by the solver:
   0 - backtrack using depth first search
   1 - backtrack using breadth first search
   2 - backtrack using the best projection heuristic */

/*----------------------------------------------------------------------
-- display_help - display help.
--
-- This routine displays help information about the program as required
-- by the GNU Coding Standards. */

static void display_help(char *my_name)
{     print("Usage: %s [options...] filename", my_name);
      print("");
      print("General options:");
      print("   --mps             read LP/MIP data using MPS format (de"
         "fault)");
      print("   --lpm             read LP/MIP model written in GLPK/L m"
         "odeling");
      print("                     language");
      print("   --check           do not solve the problem, check input"
         " data only");
      print("   --gener filename  write generated LP/MIP problem to fil"
         "ename in");
      print("                     plain text format (has sense for --lp"
         "m only)");
      print("   --write filename  write LP/MIP problem to filename in M"
         "PS format");
      print("   --min             minimize the objective function");
      print("   --max             maximize the objective function");
      print("   --simplex         use simplex method (default)");
      print("   --interior        use interior point method (for pure L"
         "P only)");
      print("   -o filename, --output filename");
      print("                     write solution to filename in plain t"
         "ext format");
      print("   --scale           scale the problem (default)");
      print("   --noscale         do not scale the problem");
      print("   -h, --help        display this help information and exi"
         "t");
      print("   -v, --version     display program version and exit");
      print("");
      print("Options specific to simplex method:");
      print("   --std             use standard initial basis");
      print("   --adv             use advanced initial basis (default)")
         ;
      print("   --steep           use steepest edge technique (default)"
         );
      print("   --nosteep         use standard \"textbook\" pricing");
      print("   --relax           use Harris' two-pass ratio test (defa"
         "ult)");
      print("   --norelax         use standard \"textbook\" ratio test")
         ;
      print("");
      print("Options specific to MIP:");
      print("   --nomip           consider all integer variables as con"
         "tinuous");
      print("                     (this allows solving MIP problem as p"
         "ure LP)");
      print("   --first           branch on first integer variable");
      print("   --last            branch on last integer variable");
      print("   --drtom           branch using heuristic by Driebeck an"
         "d Tomlin");
      print("                     (default)");
      print("   --dfs             backtrack using depth first search");
      print("   --bfs             backtrack using breadth first search")
         ;
      print("   --bestp           backtrack using the best projection h"
         "euristic");
      print("                     (default)");
      print("");
      print("See GLPK webpage at <http://www.gnu.org/software/glpk/glpk"
         ".html>");
      print("");
      print("Report bugs to <bug-glpk@gnu.org>");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- display_version - display version.
--
-- This routine displays version information for the program as required
-- by the GNU Coding Standards. */

static void display_version(void)
{     print("%s", version);
      print("Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@gnu.or"
         "g>");
      print("This program is free software; you may redistribute it und"
         "er the terms of");
      print("the GNU General Public License. This program has absolutel"
         "y no warranty.");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- process_cmdline - process command line parameters.
--
-- This routine processes parameters specified in the command line. */

#define p(str) (strcmp(argv[k], str) == 0)

static void process_cmdline(int argc, char *argv[])
{     int k;
      for (k = 1; k < argc; k++)
      {  if (p("--mps"))
            format = 'M';
         else if (p("--lpm"))
            format = 'L';
         else if (p("--check"))
            check = 1;
         else if (p("--gener"))
         {  k++;
            if (k == argc || argv[k][0] == '\0' || argv[k][0] == '-')
            {  print("No output gener filename specified");
               exit(EXIT_FAILURE);
            }
            fn_gener = argv[k];
         }
         else if (p("--write"))
         {  k++;
            if (k == argc || argv[k][0] == '\0' || argv[k][0] == '-')
            {  print("No output MPS filename specified");
               exit(EXIT_FAILURE);
            }
            fn_write = argv[k];
         }
         else if (p("--min"))
            obj_dir = LPX_MIN;
         else if (p("--max"))
            obj_dir = LPX_MAX;
         else if (p("--simplex"))
            method = 'S';
         else if (p("--interior"))
            method = 'T';
         else if (p("-o") || p("--output"))
         {  k++;
            if (k == argc || argv[k][0] == '\0' || argv[k][0] == '-')
            {  print("No output solution filename specified");
               exit(EXIT_FAILURE);
            }
            out_fname = argv[k];
         }
         else if (p("--scale"))
            scale = 1;
         else if (p("--noscale"))
            scale = 0;
         else if (p("-h") || p("--help"))
            display_help(argv[0]);
         else if (p("-v") || p("--version"))
            display_version();
         else if (p("--std"))
            basis = 0;
         else if (p("--adv"))
            basis = 1;
         else if (p("--steep"))
            price = 1;
         else if (p("--nosteep"))
            price = 0;
         else if (p("--relax"))
            relax = 1;
         else if (p("--norelax"))
            relax = 0;
         else if (p("--nomip"))
            nomip = 1;
         else if (p("--first"))
            branch = 0;
         else if (p("--last"))
            branch = 1;
         else if (p("--drtom"))
            branch = 2;
         else if (p("--dfs"))
            btrack = 0;
         else if (p("--lifo"))
            btrack = 1;
         else if (p("--bestp"))
            btrack = 2;
         else if (argv[k][0] == '-' ||
                 (argv[k][0] == '-' && argv[k][1] == '-'))
         {  print("Invalid option `%s'; try %s --help",
               argv[k], argv[0]);
            exit(EXIT_FAILURE);
         }
         else
         {  if (in_fname != NULL)
            {  print("Only one input file allowed");
               exit(EXIT_FAILURE);
            }
            in_fname = argv[k];
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- main - main routine.
--
-- This main routine is called by the control program and manages the
-- solving process. */

int main(int argc, char *argv[])
{     LPX *lp;
      int ret;
      double start;
      /* process command line parameters */
      process_cmdline(argc, argv);
      /* read problem data from the input file */
      if (in_fname == NULL)
      {  print("No input file name specified; try %s --help", argv[0]);
         exit(EXIT_FAILURE);
      }
      switch (format)
      {  case 'M':
            lp = lpx_read_mps(in_fname);
            if (lp == NULL)
            {  print("MPS file processing error");
               exit(EXIT_FAILURE);
            }
            break;
         case 'L':
            lp = lpx_read_lpm(in_fname, fn_gener);
            if (lp == NULL)
            {  print("Model processing error");
               exit(EXIT_FAILURE);
            }
            if (lpx_get_num_rows(lp) == 0)
            {  print("Problem has no rows");
               exit(EXIT_FAILURE);
            }
            if (lpx_get_num_cols(lp) == 0)
            {  print("Problem has no columns");
               exit(EXIT_FAILURE);
            }
            break;
         default:
            insist(format != format);
      }
      /* write problem data in MPS format (if required) */
      if (fn_write != NULL)
      {  ret = lpx_write_mps(lp, fn_write);
         if (ret != 0)
         {  print("Unable to write MPS file");
            exit(EXIT_FAILURE);
         }
      }
      /* if only data check is required, skip computations */
      if (check) goto skip;
      /* change optimization direction (if required) */
      if (obj_dir != 0) lpx_set_obj_dir(lp, obj_dir);
      /* scale the problem data (if required) */
      if (scale) lpx_scale_prob(lp);
      /* build an advanced initial basis (if required) */
      if (method == 'S' && basis) lpx_adv_basis(lp);
      /* set some control parameters, which might be changed in the
         command line */
      lpx_set_int_parm(lp, LPX_K_PRICE, price);
      if (!relax) lpx_set_real_parm(lp, LPX_K_RELAX, 0.0);
      lpx_set_int_parm(lp, LPX_K_BRANCH, branch);
      lpx_set_int_parm(lp, LPX_K_BTRACK, btrack);
      /* solve the problem */
      start = utime();
      switch (method)
      {  case 'S':
            if (nomip || lpx_get_class(lp) == LPX_LP)
               lpx_simplex(lp);
            else
            {  method = 'I';
               lpx_simplex(lp);
               lpx_integer(lp);
            }
            break;
         case 'T':
            if (nomip || lpx_get_class(lp) == LPX_LP)
               lpx_interior(lp);
            else
            {  print("Interior point solver is not able to solve MIP pr"
                  "oblem; use --simplex");
               exit(EXIT_FAILURE);
            }
            break;
         default:
            insist(method != method);
      }
      /* display statistics (FYI: one megabyte is 1,000,000 bytes) */
      print("Time used:   %.1f secs", utime() - start);
      print("Memory used: %.1fM (%d bytes)",
         (double)lib_env_ptr()->mem_tpeak / (double)1000000,
         lib_env_ptr()->mem_tpeak);
      /* write problem solution found by the solver (if required) */
      if (out_fname != NULL)
      {  switch (method)
         {  case 'S':
               ret = lpx_print_sol(lp, out_fname);
               break;
            case 'I':
               ret = lpx_print_mip(lp, out_fname);
               break;
            case 'T':
               ret = lpx_print_ips(lp, out_fname);
               break;
            default:
               insist(method != method);
         }
         if (ret != 0)
         {  print("Unable to write problem solution");
            exit(EXIT_FAILURE);
         }
      }
skip: /* delete the problem object */
      lpx_delete_prob(lp);
#if 0
      print("%d %d",
         lib_env_ptr()->mem_total, lib_env_ptr()->mem_count);
#endif
      /* exit to the control program */
      return 0;
}

/* eof */
