/* glplpx8c.c */

/* (reserved for copyright notice) */

#include <float.h>
#include <stddef.h>
#include "glplib.h"
#include "glplpt.h"
#include "glplpx.h"

/*----------------------------------------------------------------------
-- lpx_read_lpt - read LP/MIP problem using CPLEX LP format.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- LPX *lpx_read_lpt(char *fname);
--
-- *Description*
--
-- The routine lpx_read_lpt reads LP/MIP problem data using CPLEX LP
-- format from a text file whose name is the character string fname.
--
-- *Returns*
--
-- If no error occurred, the routine returns a pointer to the created
-- problem object. Otherwise the routine returns NULL. */

struct lpt_info
{     /* transit information passed to the routine lpt_mat */
      LPT *lpt;
      /* LP/MIP problem data block */
      LPTROW *row;
      /* pointer to the current row (constraint) */
      LPTLFE *lfe;
      /* pointer to the next term of the corresponding linear form */
};

static double lpt_mat(void *_info, int *i, int *j)
{     /* "read" a next element of the constraint matrix */
      struct lpt_info *info = _info;
      double aij;
read: if (info->row == NULL || info->lfe == NULL)
      {  /* either it is the first call or the current row has been
            completely read; choose the first/next row */
         if (info->row == NULL)
            info->row = info->lpt->first_row;
         else
            info->row = info->row->next;
         if (info->row == NULL)
         {  /* the entire matrix has been completely read */
            *i = *j = 0; /* end of file */
            return 0.0;
         }
         /* set the pointer to the first element of the row */
         info->lfe = info->row->ptr;
         /* the row may be empty, so repeat the check */
         goto read;
      }
      /* obtain the next element from the current row */
      *i = info->row->i;
      *j = info->lfe->col->j;
      aij = info->lfe->coef;
      /* advance the pointer */
      info->lfe = info->lfe->next;
      /* skip the element if it has zero value */
      if (aij == 0.0) goto read;
      /* bring the next element to the calling program */
      return aij;
}

LPX *lpx_read_lpt(char *fname)
{     LPX *lpx;
      LPT *lpt; LPTROW *row; LPTCOL *col; LPTLFE *lfe;
      int i, j;
      /* read LP/MIP problem in CPLEX LP format */
      lpt = lpt_read_prob(fname);
      if (lpt == NULL) return NULL;
      /* create problem object */
      lpx = lpx_create_prob();
      lpx_set_prob_name(lpx, "PROBLEM");
      lpx_add_rows(lpx, lpt->m);
      lpx_add_cols(lpx, lpt->n);
      /* process the objective function */
      lpx_set_obj_name(lpx, lpt->name);
      switch (lpt->sense)
      {  case LPT_MIN:
            lpx_set_obj_dir(lpx, LPX_MIN);
            break;
         case LPT_MAX:
            lpx_set_obj_dir(lpx, LPX_MAX);
            break;
         default:
            insist(lpt->sense != lpt->sense);
      }
      for (lfe = lpt->obj; lfe != NULL; lfe = lfe->next)
         lpx_set_col_coef(lpx, lfe->col->j, lfe->coef);
      /* process rows (constraints) */
      i = 0;
      for (row = lpt->first_row; row != NULL; row = row->next)
      {  i++;
         lpx_set_row_name(lpx, i, row->name);
         insist(row->i == i);
         switch (row->sense)
         {  case LPT_LE:
               lpx_set_row_bnds(lpx, i, LPX_UP, 0.0, row->rhs);
               break;
            case LPT_GE:
               lpx_set_row_bnds(lpx, i, LPX_LO, row->rhs, 0.0);
               break;
            case LPT_EQ:
               lpx_set_row_bnds(lpx, i, LPX_FX, row->rhs, row->rhs);
               break;
            default:
               insist(row->sense != row->sense);
         }
      }
      insist(i == lpt->m);
      /* process columns (variables) */
      j = 0;
      for (col = lpt->first_col; col != NULL; col = col->next)
      {  j++;
         lpx_set_col_name(lpx, j, col->name);
         insist(col->j == j);
         switch (col->kind)
         {  case LPT_CON:
               break;
            case LPT_INT:
            case LPT_BIN:
               lpx_set_class(lpx, LPX_MIP);
               lpx_set_col_kind(lpx, j, LPX_IV);
               break;
            default:
               insist(col->kind != col->kind);
         }
         if (col->lb == -DBL_MAX && col->ub == +DBL_MAX)
            lpx_set_col_bnds(lpx, j, LPX_FR, 0.0, 0.0);
         else if (col->ub == +DBL_MAX)
            lpx_set_col_bnds(lpx, j, LPX_LO, col->lb, 0.0);
         else if (col->lb == -DBL_MAX)
            lpx_set_col_bnds(lpx, j, LPX_UP, 0.0, col->ub);
         else if (col->lb != col->ub)
            lpx_set_col_bnds(lpx, j, LPX_DB, col->lb, col->ub);
         else
            lpx_set_col_bnds(lpx, j, LPX_FX, col->lb, col->ub);
      }
      insist(j == lpt->n);
      /* load the constraint matrix */
      {  struct lpt_info info;
         info.lpt = lpt;
         info.row = NULL;
         info.lfe = NULL;
         lpx_load_mat(lpx, &info, lpt_mat);
      }
      /* deallocate the problem block */
      lpt_free_prob(lpt);
      /* return to the calling program */
      return lpx;
}

/* eof */
