/* glplib3.c */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "glplib.h"

/*----------------------------------------------------------------------
-- print - print informative message.
--
-- *Synopsis*
--
-- #include "glplib.h"
-- void print(char *fmt, ...);
--
-- *Description*
--
-- The routine print prints an informative message specified by the
-- format control string fmt and optional parameter list. */

void print(char *fmt, ...)
{     va_list arg;
      /* print an informative message */
      va_start(arg, fmt);
      vfprintf(stdout, fmt, arg);
      va_end(arg);
      fputc('\n', stdout);
      /* return to the calling program */
      return;
}

/*----------------------------------------------------------------------
-- fault - print error message and terminate program execution.
--
-- *Synopsis*
--
-- #include "glplib.h"
-- void fault(char *fmt, ...);
--
-- *Description*
--
-- The routine fault prints an error message specified by the format
-- control string fmt and optional parameter list, and then abnormally
-- terminates execution of the program.
--
-- *Returns*
--
-- The routine fault never returns. */

void fault(char *fmt, ...)
{     va_list arg;
      /* print an error message */
      va_start(arg, fmt);
      vfprintf(stdout, fmt, arg);
      va_end(arg);
      fputc('\n', stdout);
      /* deinitialize library environment */
      free_lib_env();
      /* terminate program execution */
#if 0
      abort();
#else
      exit(3);
#endif
      /* no return */
}

/*----------------------------------------------------------------------
-- insist - check for logical condition.
--
-- *Synopsis*
--
-- #include "glplib.h"
-- void insist(int expr);
--
-- *Description*
--
-- The routine insist (implemented as a macro) checks for a logical
-- condition specified by the parameter expr. If the condition is false
-- (i.e. expr is zero), the routine prints an appropriate error message
-- and abnormally terminates the program.
--
-- This routine is a replacement of the standard function assert. */

void _insist(char *expr, char *file, int line)
{     /* print an error message */
      fputc('\n', stdout);
      fprintf(stdout, "Assertion failed: %s, file %s, line %d\n",
         expr, file, line);
      /* deinitialize library environment */
      free_lib_env();
      /* terminate program execution */
#if 0
      abort();
#else
      exit(3);
#endif
      /* no return */
}

/*----------------------------------------------------------------------
-- jday - convert calendar date to Julian day.
--
-- This procedure converts a calendar date, Gregorian calendar, to the
-- corresponding Julian day number j. From the given day d, month m, and
-- year y, the Julian day number j is computed without using tables. The
-- procedure is valid for any valid Gregorian calendar date. */

static int jday(int d, int m, int y)
{     int c, ya, j;
      if (m > 2) m -= 3; else m += 9, y--;
      c = y / 100;
      ya = y - 100 * c;
      j = (146097 * c) / 4 + (1461 * ya) / 4 + (153 * m + 2) / 5 + d +
         1721119;
      return j;
}

/*----------------------------------------------------------------------
-- gmt2000 - determine the current time.
--
-- *Synopsis*
--
-- #include "glplib.h"
-- double gmt2000(void);
--
-- *Returns*
--
-- The routine gmt2000 returns the current time, in seconds, elapsed
-- since 12:00:00 GMT January 1, 2000. */

double gmt2000(void)
{     /* this is a platform independent version */
      time_t timer;
      struct tm *tm;
      int j2000, j;
      double secs;
      timer = time(NULL);
      tm = gmtime(&timer);
      j2000 = 2451545 /* = jday(1, 1, 2000) */;
      j = jday(tm->tm_mday, tm->tm_mon + 1, 1900 + tm->tm_year);
      secs = (((double)(j - j2000) * 24.0 + (double)tm->tm_hour) * 60.0
         + (double)tm->tm_min) * 60.0 + (double)tm->tm_sec - 43200.0;
      return secs;
}

#if 0
double gmt2000(void)
{     /* this is a version for Win32 */
      SYSTEMTIME st;
      FILETIME ft0, ft;
      double secs;
      /* ft0 = 12:00:00 GMT January 1, 2000 */
      ft0.dwLowDateTime  = 0xBAA22000;
      ft0.dwHighDateTime = 0x01BF544F;
      GetSystemTime(&st);
      SystemTimeToFileTime(&st, &ft);
      /* ft = ft - ft0 */
      if (ft.dwLowDateTime >= ft0.dwLowDateTime)
      {  ft.dwLowDateTime  -= ft0.dwLowDateTime;
         ft.dwHighDateTime -= ft0.dwHighDateTime;
      }
      else
      {  ft.dwLowDateTime  += (0xFFFFFFFF - ft0.dwLowDateTime) + 1;
         ft.dwHighDateTime -= ft0.dwHighDateTime + 1;
      }
      secs = (4294967296.0 * (double)(LONG)ft.dwHighDateTime +
         (double)ft.dwLowDateTime) / 10000000.0;
      return secs;
}
#endif

/*----------------------------------------------------------------------
-- watch - take reading of stop-watch.
--
-- *Synopsis*
--
-- #include "glplib.h"
-- double watch(void);
--
-- *Returns*
--
-- The routine watch returns the time, in seconds, elapsed since the
-- program started. */

double watch(void)
{     ENV *env = get_env_ptr();
      return gmt2000() - env->start;
}

/* eof */
