/* glprsm/dual_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- dual_row - choose basic variable (dual).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int dual_row(RSM *rsm, double bbar[], double dvec[], int *tagp,
--    double tol);
--
-- *Description*
--
-- The dual_row routine chooses basic variable (xB)p (i.e. pivot row of
-- the simplex table) which should leave the basis on the next iteration
-- of the dual simplex method.
--
-- The array bbar should contain current values of basic variables xB in
-- locations bbar[1], ..., bbar[m]. This array is not changed on exit.
--
-- The array dvec should contain elements of the vector delta in
-- locations dvec[1], ..., dvec[m]. This array is not changed on exit.
-- It is allowed to specify NULL instead the array dvec; in this case
-- the routine assumes that all elements of the vector delta are equal
-- to one.
--
-- If the routine chooses some basic variable (xB)p, it stores into
-- location tagp a tag that specifies to what subset xL or xU the basic
-- variable should be attributed after it has left the basis. This tag
-- has the same meaning as the field rsm.tagn (see 'glprsm.h'). Thus,
-- the subset xL means that the basic variable (xB)p violates its lower
-- bound and therefore it should be set on its lower bound. Analogously,
-- the subset xU means that the basic variable (xB)p violates its upper
-- bound and therefore it should be set on its upper bound. Note that if
-- the basic variable (xB)p is of fixed type, it is considered as
-- double-bounded variable (with lower bound equal to upper bound),
-- therefore its tag should be corrected before changing the basis by
-- means of the change_b routine. This exception is used in order to let
-- other routines know what to do with such fixed basic variable:
-- increase (in case of xL) or decrease (in case of xU).
--
-- The parameter tol is a relative tolerance (see below).
--
-- The dual_row routine considers only those basic variables which
-- violate their bounds:
--
--    bbar[i] < (lB)i - eps  or
--    bbar[i] > (uB)i + eps
--
-- where
--
--    eps = tol * max(1, |(lB)i|) in case of lower bound
--    eps = tol * max(1, |(uB)i|) in case of upper bound
--
-- The routine chooses the basic variable (xB)p which has the largest
-- (in absolute value) *scaled* residual. Thus, if the vector delta is
-- not used, the choice made by the routine corresponds to the textbook
-- technique. Otherwise the choice corresponds to the steepest edge
-- technique.
--
-- *Returns*
--
-- If the choice has been made, the dual_row routine returns p which is
-- a number of the chosen basic variable (xB)p, 1 <= p <= m. Otherwise,
-- if the current basis solution is primal feasible and therefore the
-- choice is impossible, the routine returns zero. */

int dual_row(RSM *rsm, double bbar[], double dvec[], int *_tagp,
      double tol)
{     int m = rsm->m, i, k, p, tagp;
      double big, temp;
      p = 0, tagp = -1, big = 0.0;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'L' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xB[i] has lower bound */
            if (check_rr(bbar[i], rsm->lb[k], tol) < -1)
            {  temp = rsm->lb[k] - bbar[i];
               if (dvec != NULL) temp /= sqrt(dvec[i]);
               if (big < temp) p = i, tagp = 'L', big = temp;
            }
         }
         if (rsm->type[k] == 'U' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xB[i] has upper bound */
            if (check_rr(bbar[i], rsm->ub[k], tol) > +1)
            {  temp = bbar[i] - rsm->ub[k];
               if (dvec != NULL) temp /= sqrt(dvec[i]);
               if (big < temp) p = i, tagp = 'U', big = temp;
            }
         }
      }
      *_tagp = tagp;
      return p;
}

/* eof */
