/* glpapi/old_api.c */

/* This file contains obsolete GLPK API routines, which will be removed
   from the package in the future. */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define GLP_OLD_API
#include "glpapi.h"
#include "glpbbm.h"
#include "glpipm.h"
#include "glplang.h"
#include "glpmps.h"

/*----------------------------------------------------------------------
-- glp_create_item - create new row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_create_item(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_create_item routine creates new row with
-- the given name. Initially the created row has GLP_FX type (i.e. the
-- auxiliary variable is fixed) with both lower and upper bounds equal
-- to zero (such row corresponds to the linear equality constraint with
-- zero right-hand side). Being created new row becomes the current row.
--
-- If what = GLP_COL, the glp_create_item routine creates new column
-- with the given name. Initially the created column has GLP_LO type
-- with lower bound equal to zero (such column corresponds to the
-- non-negative structural variable). Being created new column becomes
-- the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - row/column with the given name already exists;
-- 2 - the parameter what or the given name is invalid. */

int glp_create_item(int what, char *name)
{     if (what == GLP_ROW)
      {  /* create new row */
         STR *row_name;
         GLPITEM *row;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->row_tab, row_name) != NULL)
         {  /* row already exists */
            delete_str(row_name);
            return 1;
         }
         row = get_atom(glp->item_pool);
         row->name = row_name;
         row->kind = 0;
         row->type = GLP_FX;
         row->lb = row->ub = 0.0;
         row->ptr = NULL;
         row->seqn = 0;
         row->tagx = GLP_BS;
         row->valx = row->dx = 0.0;
         row->next = NULL;
         /* add to the symbol table */
         node = insert_by_key(glp->row_tab, row->name);
         node->link = row;
         /* add to the end of the linked list */
         if (glp->first_row == NULL)
            glp->first_row = row;
         else
            glp->last_row->next = row;
         glp->last_row = row;
         /* new row becomes the current row */
         glp->this_row = row;
      }
      else if (what == GLP_COL)
      {  /* create new column */
         STR *col_name;
         GLPITEM *col;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->col_tab, col_name) != NULL)
         {  /* column already exists */
            delete_str(col_name);
            return 1;
         }
         col = get_atom(glp->item_pool);
         col->name = col_name;
         col->kind = 0;
         col->type = GLP_LO;
         col->lb = col->ub = 0.0;
         col->ptr = NULL;
         col->seqn = 0;
         col->tagx = GLP_BS;
         col->valx = col->dx = 0.0;
         col->next = NULL;
         /* add to the symbol table */
         node = insert_by_key(glp->col_tab, col->name);
         node->link = col;
         /* add to the end of the linked list */
         if (glp->first_col == NULL)
            glp->first_col = col;
         else
            glp->last_col->next = col;
         glp->last_col = col;
         /* new column becomes the current column */
         glp->this_col = col;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_dump_ws - dump GLPK API workspace.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_dump_ws(char *fname, int what);
--
-- *Description*
--
-- The glp_dump_ws routine dumps information from GLPK API workspace to
-- the text file, whose name is the character string fname, in readable
-- format. This operation is intended for application debugging.
--
-- The parameter what specifies what information should be dumped. It
-- can be a combination of the following options:
--
-- GLP_D_PARS  dump control parameters;
-- GLP_D_ROWS  dump rows (auxiliary variables);
-- GLP_D_RMAT  dump constraint coefficients in row-wise format;
-- GLP_D_COLS  dump columns (structural variables);
-- GLP_D_CMAT  dump constraint coefficients in column-wise format;
-- GLP_D_ALL   dump all information (assumes all options above).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

static void dump_parameters(FILE *fp)
{     int iii;
      double rrr;
      char sss[GLP_MAX_NAME+1];
      glp_get_rpar("c0", &rrr);
      fprintf(fp, "c0             = %.8g\n", rrr);
      glp_get_cpar("fn_gener", sss);
      fprintf(fp, "fn_gener       = %s\n", sss);
      glp_get_ipar("mip_branch", &iii);
      fprintf(fp, "mip_branch     = %s\n",
         iii == GLP_FIRST ? "FIRST" : iii == GLP_LAST ? "LAST" :
         iii == GLP_DRTOM ? "DRTOM" : "???");
      glp_get_ipar("mip_btrack", &iii);
      fprintf(fp, "mip_btrack     = %s\n",
         iii == GLP_FIFO  ? "FIFO"  : iii == GLP_LIFO ? "LIFO" :
         iii == GLP_BESTP ? "BESTP" : "???");
      glp_get_cpar("mps_bnd_name", sss);
      fprintf(fp, "mps_bnd_name   = %s\n", sss);
      glp_get_cpar("mps_obj_name", sss);
      fprintf(fp, "mps_obj_name   = %s\n", sss);
      glp_get_ipar("mps_one_entry", &iii);
      fprintf(fp, "mps_one_entry  = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("mps_pedantic", &iii);
      fprintf(fp, "mps_pedantic   = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_cpar("mps_rhs_name", sss);
      fprintf(fp, "mps_rhs_name   = %s\n", sss);
      glp_get_cpar("mps_rng_name", sss);
      fprintf(fp, "mps_rng_name   = %s\n", sss);
      glp_get_ipar("mps_skip_empty", &iii);
      fprintf(fp, "mps_skip_empty = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("mps_use_names", &iii);
      fprintf(fp, "mps_use_names  = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("nc", &iii);
      fprintf(fp, "nc             = %d\n", iii);
      glp_get_ipar("nc_bin", &iii);
      fprintf(fp, "nc_bin         = %d\n", iii);
      glp_get_ipar("nc_int", &iii);
      fprintf(fp, "nc_int         = %d\n", iii);
      glp_get_ipar("nr", &iii);
      fprintf(fp, "nr             = %d\n", iii);
      glp_get_ipar("nz", &iii);
      fprintf(fp, "nz             = %d\n", iii);
      glp_get_ipar("obj_dir", &iii);
      fprintf(fp, "obj_dir        = %s\n",
         iii == GLP_MIN ? "MIN" : iii == GLP_MAX ? "MAX" : "???");
      glp_get_cpar("obj_row", sss);
      fprintf(fp, "obj_row        = %s\n", sss);
      glp_get_ipar("option", &iii);
      fprintf(fp, "option         = %s\n",
         iii == GLP_INI ? "INI" : iii == GLP_ANY ? "ANY" :
         iii == GLP_FIN ? "FIN" : "???");
      glp_get_cpar("problem", sss);
      fprintf(fp, "problem        = %s\n", sss);
      glp_get_ipar("round", &iii);
      fprintf(fp, "round          = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("scale", &iii);
      fprintf(fp, "scale          = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_form", &iii);
      fprintf(fp, "spx_form       = %s\n",
         iii == GLP_EFI    ? "EFI"    :
         iii == GLP_RFI_BG ? "RFI_BG" :
         iii == GLP_RFI_FT ? "RFI_FT" : "???");
      glp_get_ipar("spx_relax", &iii);
      fprintf(fp, "spx_relax      = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_steep", &iii);
      fprintf(fp, "spx_steep      = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("spx_use_dual", &iii);
      fprintf(fp, "spx_use_dual   = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_ipar("status", &iii);
      fprintf(fp, "status         = %s\n",
         iii == GLP_UNDEF  ? "UNDEF"  :
         iii == GLP_OPT    ? "OPT"    :
         iii == GLP_FEAS   ? "FEAS"   :
         iii == GLP_INFEAS ? "INFEAS" :
         iii == GLP_NOFEAS ? "NOFEAS" :
         iii == GLP_UNBND  ? "UNBND"  : "???");
      glp_get_ipar("sum_aij", &iii);
      fprintf(fp, "sum_aij        = %s\n",
         iii == GLP_NO ? "NO" : iii == GLP_YES ? "YES" : "???");
      glp_get_rpar("tol_aij", &rrr);
      fprintf(fp, "tol_aij        = %.8g\n", rrr);
      glp_get_rpar("tol_bnd", &rrr);
      fprintf(fp, "tol_bnd        = %.8g\n", rrr);
      glp_get_rpar("tol_dj", &rrr);
      fprintf(fp, "tol_dj         = %.8g\n", rrr);
      glp_get_rpar("tol_int", &rrr);
      fprintf(fp, "tol_int        = %.8g\n", rrr);
      glp_get_rpar("tol_obj", &rrr);
      fprintf(fp, "tol_obj        = %.8g\n", rrr);
      glp_get_rpar("tol_piv", &rrr);
      fprintf(fp, "tol_piv        = %.8g\n", rrr);
      return;
}

/*--------------------------------------------------------------------*/

static void dump_items(FILE *fp, int what, int mat)
{     int ret;
      for (ret = glp_first_item(what); ret == 0;
           ret = glp_next_item(what))
      {  char name[GLP_MAX_NAME+1];
         int kind, type, tagx;
         double lb, ub, valx, dx;
         glp_get_name(what, name);
         glp_get_kind(what, &kind);
         glp_get_bounds(what, &type, &lb, &ub);
         glp_get_activity(what, &tagx, &valx, &dx);
         glp_get_name(what, name);
         if (strlen(name) <= 15)
            fprintf(fp, "%-15s ", name);
         else
            fprintf(fp, "%s\n%16s", name, "");
         /* row/column kind */
         fprintf(fp, "%s ",
            kind == GLP_NO ? " " : kind == GLP_YES ? "I" : "?");
         /* row/column type, lower bound, and upper bound */
         switch (type)
         {  case GLP_FR:
               fprintf(fp, "FR");
               break;
            case GLP_LO:
               fprintf(fp, "LO %15.8g", lb);
               break;
            case GLP_UP:
               fprintf(fp, "UP %15s %15.8g", "", ub);
               break;
            case GLP_DB:
               fprintf(fp, "DB %15.8g %15.8g", lb, ub);
               break;
            case GLP_FX:
               fprintf(fp, "FX %15.8g %15s", lb, "=");
               break;
            default:
               fprintf(fp, "?? %15.8g %15.8g", lb, ub);
               break;
         }
         fprintf(fp, "\n%18s", "");
         /* row/column status, primal value, and dual value */
         fprintf(fp, "%s %15.8g %15.8g",
            tagx == GLP_BS ? "BS" : tagx == GLP_NL ? "NL" :
            tagx == GLP_NU ? "NU" : tagx == GLP_NF ? "NF" :
            tagx == GLP_NS ? "NS" : "??", valx, dx);
         fprintf(fp, "\n");
         /* constraint coefficients */
         if (mat)
         {  int ret;
            for (ret = glp_first_coef(what); ret == 0;
                 ret = glp_next_coef(what))
            {  double coef;
               glp_get_name(what == GLP_ROW ? GLP_COL : GLP_ROW, name);
               glp_get_coef(&coef);
               fprintf(fp, "%15.8g (%s)\n", coef, name);
            }
         }
      }
      return;
}

/*--------------------------------------------------------------------*/

int glp_dump_ws(char *fname, int what)
{     FILE *fp;
      print("glp_dump_ws: dumping workspace to `%s'...", fname);
      /* open the output text file */
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_dump_ws: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* print header */
      fprintf(fp, "*** DUMP OF GLPK API WORKSPACE ***\n");
      /* dump control parameters */
      if (what & GLP_D_PARS)
      {  fprintf(fp, "\n\n*** Control Parameters ***\n\n");
         dump_parameters(fp);
      }
      /* dump rows and constraint coefficients for each row */
      if (what & GLP_D_ROWS)
      {  fprintf(fp, "\n\n*** Rows (Auxiliary Variables) ***\n\n");
         dump_items(fp, GLP_ROW, what & GLP_D_RMAT);
      }
      /* dump columns and constraint coefficients for each column */
      if (what & GLP_D_COLS)
      {  fprintf(fp, "\n\n*** Columns (Structural Variables) ***\n\n");
         dump_items(fp, GLP_COL, what & GLP_D_CMAT);
      }
      /* end of dump */
      fprintf(fp, "\n\n*** END OF DUMP ***\n");
      /* close the output text file */
      fflush(fp);
      if (ferror(fp))
      {  error("glp_dump_ws: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      /* return to the calling program */
      return 0;
fail: /* the operation failed */
      if (fp != NULL) fclose(fp);
      return 1;
}

/*----------------------------------------------------------------------
-- glp_find_item - find row or column by name.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_find_item(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_find_item routine finds the row with the
-- given name and makes it be the current row.
--
-- If what = GLP_COL, the glp_find_item routine finds the column with
-- the given name and makes it be the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - row/column with the given name doesn't exist;
-- 2 - the parameter what or the given name is invalid. */

int glp_find_item(int what, char *name)
{     if (what == GLP_ROW)
      {  /* find row */
         STR *row_name;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         node = find_by_key(glp->row_tab, row_name);
         delete_str(row_name);
         if (node == NULL) return 1;
         glp->this_row = node->link;
      }
      else if (what == GLP_COL)
      {  /* find column */
         STR *col_name;
         AVLNODE *node;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         node = find_by_key(glp->col_tab, col_name);
         delete_str(col_name);
         if (node == NULL) return 1;
         glp->this_col = node->link;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_first_coef - find the first constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_first_coef(int what);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_first_coef routine finds the first
-- constraint coefficient placed in the current row and makes it be the
-- current coefficient. At the same time the routine also makes the
-- column, which corresponds to the found coefficient, be the current
-- column (the current row remains unchanged).
--
-- If what = GLP_COL, the glp_first_coef routine finds the first
-- constraint coefficient placed in the current column and makes it be
-- the current coefficient. At the same time the routine also makes the
-- row, which corresponds to the found coefficient, be the current row
-- (the current column remains unchanged).
--
-- The ordering of constraint coefficients can be changed by GLPK API
-- routines, therefore the application program shouldn't rely on some
-- particular ordering (except cases, when the particular ordering is
-- explictly declared).
--
-- *Complexity*
--
-- This operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row is undefined or empty (in case of GLP_ROW), or
--     the current column is undefined or empty (in case of GLP_COL);
-- 2 - the parameter what is invalid. */

int glp_first_coef(int what)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         glp->this_coef = glp->this_row->ptr;
         if (glp->this_coef == NULL) return 1;
         glp->this_col = glp->this_coef->col;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         glp->this_coef = glp->this_col->ptr;
         if (glp->this_coef == NULL) return 1;
         glp->this_row = glp->this_coef->row;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_first_item - find the first row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_first_item(int what);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_first_item routine finds the first row of
-- the problem and makes it be the current row.
--
-- If what = GLP_COL, the glp_first_item routine finds the first column
-- of the problem and makes it be the current column.
--
-- It's assumed that one row (column) precedes other row (column) if the
-- former was created before the latter. Hence, the first row (column)
-- is that one, which was created before any other rows (columns).
--
-- *Complexity*
--
-- This operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - there is no rows or no columns in the problem;
-- 2 - the parameter what is invalid. */

int glp_first_item(int what)
{     if (what == GLP_ROW)
      {  glp->this_row = glp->first_row;
         if (glp->this_row == NULL) return 1;
      }
      else if (what == GLP_COL)
      {  glp->this_col = glp->first_col;
         if (glp->this_col == NULL) return 1;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_activity - get activity of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_activity(int what, int *tagx, double *valx, double *dx);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_activity routine stores the status,
-- the primal activity, and the dual activity of the current row (i.e.
-- of the corresponding auxiliary variable) to locations, which
-- parameters tagx, valx, and dx point to, respectively.
--
-- If what = GLP_COL, the glp_get_activity routine stores the same
-- information, but for the current column (i.e. for the corresponding
-- structural variable).
--
-- Information reported by the glp_get_activity routine is determined
-- by the final simplex table found by the solver. This information has
-- the meaning only if the problem has been successfully solved.
--
-- The status of the variable may be one of the following:
--
-- GLP_BS - the variable is basis;
-- GLP_NL - the variable is non-basis and placed on its lower bound
--          (only for variables of GLP_LO and GLP_DB types);
-- GLP_NU - the variable is non-basis and placed on its upper bound
--          (only for variables of GLP_UP and GLP_DB types);
-- GLP_NF - the free variable is non-basis (only for variables of
--          GLP_FR type);
-- GLP_NS - the fixed variable is non-basis (only for variables of
--          GLP_FX type).
--
-- The primal activity is the computed value of the variable. (Note that
-- value of any free non-basis variable is always equal to zero.)
--
-- The dual activity is the computed reduced cost (marginal value) of
-- the variable.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_activity(int what, int *tagx, double *valx, double *dx)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      if (tagx != NULL) *tagx = item->tagx;
      if (valx != NULL) *valx = item->valx;
      if (dx != NULL) *dx = item->dx;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_bounds - get bounds of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_bounds(int what, int *type, double *lb, double *ub);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_bounds routine stores the type, the
-- lower bound and the upper bound of the current row (i.e. of the
-- corresponding auxiliary variable) to locations, which parameters
-- type, lb, and ub point to, respectively.
--
-- If what = GLP_COL, the glp_get_bounds routine stores the same
-- information, but for the current column (i.e. for the corresponding
-- structural variable).
--
-- If some of pointers type, lb, or ub is NULL, the corresponding value
-- is not stored.
--
-- Types and the corresponding bounds of rows and columns are shown in
-- the following table:
--
--     Type          Bounds              Note
--    ------    -----------------    -------------
--    GLP_FR    -inf <  x <  +inf    free variable
--    GLP_LO      lb <= x <  +inf    lower bound
--    GLP_UP    -inf <  x <=  ub     upper bound
--    GLP_DB      lb <= x <=  ub     double bound
--    GLP_FX            x  =  lb     fixed variable
--
-- where x is the auxiliary (in case of row) or structural (in case of
-- column) variable.
--
-- If the current row/column has no lower or/and upper bound, the
-- corresponding values of lb or/and ub will be set to zero. If the
-- current row is og GLP_FX type, both values lb and ub will be set to
-- the fixed value of the row/column.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_bounds(int what, int *type, double *lb, double *ub)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      if (type != NULL) *type = item->type;
      if (lb != NULL) *lb = item->lb;
      if (ub != NULL) *ub = item->ub;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_coef - get value of the current constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_coef(double *val);
--
-- *Description*
--
-- The glp_get_coef routine stores the value of the current constraint
-- coefficient to the location, which the parameter val points to.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current coefficient is undefined. */

int glp_get_coef(double *val)
{     if (glp->this_coef == NULL) return 1;
      *val = glp->this_coef->val;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_cpar - get value of text control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_cpar(char *name, char *val);
--
-- *Description*
--
-- The glp_get_cpar routine copies the value of the text control
-- parameter with the given name to the character string val.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid. */

int glp_get_cpar(char *name, char *val)
{     if (strcmp(name, "fn_gener") == 0)
      {  /* file name to output generated LP/MIP problem */
         if (glp->fn_gener == NULL)
            val[0] = '\0';
         else
            get_str(val, glp->fn_gener);
      }
      else if (strcmp(name, "mps_bnd_name") == 0)
      {  /* the name of bound vector */
         strcpy(val, glp->mps_bnd_name);
      }
      else if (strcmp(name, "mps_obj_name") == 0)
      {  /* the name of the objective function */
         strcpy(val, glp->mps_obj_name);
      }
      else if (strcmp(name, "mps_rhs_name") == 0)
      {  /* the name of the right-hand side (RHS) vector */
         strcpy(val, glp->mps_rhs_name);
      }
      else if (strcmp(name, "mps_rng_name") == 0)
      {  /* the name of the range vector */
         strcpy(val, glp->mps_rng_name);
      }
      else if (strcmp(name, "obj_row") == 0)
      {  /* the name of the objective function */
         if (glp->obj_row == NULL)
            val[0] = '\0';
         else
            get_str(val, glp->obj_row->name);
      }
      else if (strcmp(name, "problem") == 0)
      {  /* the name of problem */
         get_str(val, glp->problem);
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_ipar - get value of integer control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_ipar(char *name, int *val);
--
-- *Description*
--
-- The glp_get_ipar routine stores the value of the integer control
-- parameter with the given name to the location, which the parameter
-- val points to.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid. */

int glp_get_ipar(char *name, int *val)
{     if (strcmp(name, "mip_branch") == 0)
      {  /* branching technique flag */
         *val = glp->mip_branch;
      }
      else if (strcmp(name, "mip_btrack") == 0)
      {  /* backtracking technique flag */
         *val = glp->mip_btrack;
      }
      else if (strcmp(name, "mps_one_entry") == 0)
      {  /* write MPS file not using fields 5 and 6 */
         *val = glp->mps_one_entry;
      }
      else if (strcmp(name, "mps_pedantic") == 0)
      {  /* write MPS file using pedantic style */
         *val = glp->mps_pedantic;
      }
      else if (strcmp(name, "mps_skip_empty") == 0)
      {  /* don't write empty columns to MPS file */
         *val = glp->mps_skip_empty;
      }
      else if (strcmp(name, "mps_use_names") == 0)
      {  /* use original names as templates to generate MPS names */
         *val = glp->mps_use_names;
      }
      else if (strcmp(name, "nc") == 0)
      {  /* total number of columns */
         *val = glp->col_tab->size;
      }
      else if (strcmp(name, "nc_bin") == 0)
      {  /* number of binary columns */
         GLPITEM *col;
         int nc_bin = 0;
         for (col = glp->first_col; col != NULL; col = col->next)
            if (col->kind && col->type == GLP_DB &&
                col->lb == 0.0 && col->ub == 1.0) nc_bin++;
         *val = nc_bin;
      }
      else if (strcmp(name, "nc_int") == 0)
      {  /* number of integer columns */
         GLPITEM *col;
         int nc_int = 0;
         for (col = glp->first_col; col != NULL; col = col->next)
            if (col->kind) nc_int++;
         *val = nc_int;
      }
      else if (strcmp(name, "nr") == 0)
      {  /* total number of rows */
         *val = glp->row_tab->size;
      }
      else if (strcmp(name, "nz") == 0)
      {  /* total number of non-zeros in the constraint matrix */
         *val = glp->coef_pool->count;
      }
      else if (strcmp(name, "obj_dir") == 0)
      {  /* optimization direction flag */
         *val = glp->obj_dir;
      }
      else if (strcmp(name, "option") == 0)
      {  /* what solution should be found by the solver */
         *val = glp->option;
      }
      else if (strcmp(name, "round") == 0)
      {  /* round the computed values of basis variable */
         *val = glp->round;
      }
      else if (strcmp(name, "scale") == 0)
      {  /* scale the problem before solving */
         *val = glp->scale;
      }
      else if (strcmp(name, "spx_form") == 0)
      {  /* the form of the basis matrix used by the solver */
         *val = glp->spx_form;
      }
      else if (strcmp(name, "spx_relax") == 0)
      {  /* use the ratio test proposed by P.Harris */
         *val = glp->spx_relax;
      }
      else if (strcmp(name, "spx_steep") == 0)
      {  /* use the steepest edge technique */
         *val = glp->spx_steep;
      }
      else if (strcmp(name, "spx_use_dual") == 0)
      {  /* use dual simplex to search for feasible solution */
         *val = glp->spx_use_dual;
      }
      else if (strcmp(name, "status") == 0)
      {  /* status of the computed solution (set bythe solver) */
         *val = glp->status;
      }
      else if (strcmp(name, "sum_aij") == 0)
      {  /* sum multiplets of the constraint matrix */
         *val = glp->sum_aij;
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_kind - get kind of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_kind(int what, int *kind);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_kind routine stores the kind of the
-- current row to the location which kind points to.
--
-- If what = GLP_COL, the glp_get_kind routine stores the kind of the
-- current column to the location which kind points to.
--
-- GLP_NO means that the corresponding variable is continuous, GLP_YES
-- means that the corresponding variable is integer (discrete).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_kind(int what, int *kind)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         *kind = (glp->this_row->kind == 0 ? GLP_NO : GLP_YES);
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         *kind = (glp->this_col->kind == 0 ? GLP_NO : GLP_YES);
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_name - get name of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_name(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_name routine copies the symbolic name
-- of the current row to the character string name.
--
-- If what = GLP_COL, the glp_get_name routine copies the symbolic name
-- of the current column to the character string name.
--
-- Maximal length of symbolic names is 255 characters, so the array name
-- should contain at least 256 locations.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row or column is undefined;
-- 2 - parameter what is invalid. */

int glp_get_name(int what, char *name)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      get_str(name, item->name);
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_rpar - get value of real control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_rpar(char *name, double *val);
--
-- *Description*
--
-- The glp_get_rpar routine stores the value of the real control
-- parameter with the given name to the location, which the parameter
-- val points to.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid. */

int glp_get_rpar(char *name, double *val)
{     if (strcmp(name, "c0") == 0)
      {  /* the constant term of the objective function */
         *val = glp->c0;
      }
      else if (strcmp(name, "tol_aij") == 0)
      {  /* tolerance used by the solver to drop small elements of the
            constraint matrix */
         *val = glp->tol_aij;
      }
      else if (strcmp(name, "tol_bnd") == 0)
      {  /* tolerance used by the solver to check primal feasibility of
            the current basis solution */
         *val = glp->tol_bnd;
      }
      else if (strcmp(name, "tol_dj") == 0)
      {  /* tolerance used by the solver to check dual feasibility of
            the current basis solution */
         *val = glp->tol_bnd;
      }
      else if (strcmp(name, "tol_int") == 0)
      {  /* tolerance used by MIP solver to check integer feasibility
            of the current basis solution */
         *val = glp->tol_int;
      }
      else if (strcmp(name, "tol_obj") == 0)
      {  /* tolerance used by MIP solver to compare values of objective
            function */
         *val = glp->tol_obj;
      }
      else if (strcmp(name, "tol_piv") == 0)
      {  /* tolerance used by the solver to check pivoting element */
         *val = glp->tol_piv;
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_get_seqn - get seqn attribute of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_seqn(int what, int *seqn);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_seqn routine stores the seqn attribute
-- of the current row to the location pointed by the parameter seqn.
--
-- If what = GLP_COL, the glp_get_seqn routine stores the seqn attribute
-- of the current column to the location pointed by the parameter seqn.
--
-- The seqn attribute is an additional attribute of rows and columns.
-- It is just an integer value, which may be used for any purposes.
-- This attribute is intended mainly for needs of GLPK routines, however
-- it may be used by the application program.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_seqn(int what, int *seqn)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         *seqn = glp->this_row->seqn;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         *seqn = glp->this_col->seqn;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_initialize - initialize GLPK application program interface.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_initialize(void);
--
-- *Description*
--
-- The glp_initialize routine initializes the GLPK application program
-- interface (API).
--
-- This routine should be called once before calls any other GLPK API
-- routines.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - GLPK API already initialized. */

GLPWKSP *glp = NULL;
/* pointer to the active workspace */

int glp_initialize(void)
{     if (glp != NULL) return 1;
      glp = umalloc(sizeof(GLPWKSP));
      /* --- memory management segment --- */
      glp->str_pool = create_str_pool();
      glp->item_pool = create_pool(sizeof(GLPITEM));
      glp->coef_pool = create_pool(sizeof(GLPCOEF));
      /* --- problem data segment --- */
      glp->row_tab = create_avl((int (*)(void *, void *))compare_str);
      glp->first_row = glp->last_row = glp->this_row = NULL;
      glp->col_tab = create_avl((int (*)(void *, void *))compare_str);
      glp->first_col = glp->last_col = glp->this_col = NULL;
      glp->this_coef = NULL;
      /* --- control parameters segment --- */
      glp->c0 = 0.0;
      glp->fn_gener = NULL;
      glp->mip_branch = GLP_DRTOM;
      glp->mip_btrack = GLP_BESTP;
      glp->mps_bnd_name[0] = '\0';
      glp->mps_obj_name[0] = '\0';
      glp->mps_one_entry = GLP_NO;
      glp->mps_pedantic = GLP_YES;
      glp->mps_rhs_name[0] = '\0';
      glp->mps_rng_name[0] = '\0';
      glp->mps_skip_empty = GLP_NO;
      glp->mps_use_names = GLP_YES;
      glp->obj_dir = GLP_MIN;
      glp->obj_row = NULL;
      glp->option = GLP_FIN;
      glp->problem = set_str(create_str(glp->str_pool), "UNKNOWN");
      glp->round = GLP_YES;
      glp->scale = GLP_YES;
      glp->spx_form = GLP_RFI_BG;
      glp->spx_relax = GLP_YES;
      glp->spx_steep = GLP_YES;
      glp->spx_use_dual = GLP_NO;
      glp->status = GLP_UNDEF;
      glp->sum_aij = GLP_NO;
      glp->tol_aij = 1e-15;
      glp->tol_bnd = 1e-8;
      glp->tol_dj = 1e-7;
      glp->tol_int = 1e-6;
      glp->tol_obj = 1e-7;
      glp->tol_piv = 1e-10;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_integer - solve MIP problem using branch-and-bound procedure.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_integer(void);
--
-- *Description*
--
-- The glp_integer is a MIP problem solver based on the branch-and-bound
-- procedure and the revised dual simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Since many MIP problems may take a long time, the glp_integer routine
-- reports some visual information about current status of the search.
-- The information is sent to stdout approximately once per second and
-- has the following format:
--
--    +nnn: mip = xxx; lp = yyy (aaa; sss)
--
-- where nnn is total number of simplex iteration, xxx is a value of
-- the objective function that corresponds to the best MIP solution
-- (if this solution has not been found yet, xxx is the text "not found
-- yet"), yyy is a value of the objective function that corresponds to
-- the initial relaxed optimal solution (it is not changed during all
-- the optimization process), aaa is number of subroblems in the active
-- list, sss is number of subproblems which have been solved yet.
--
-- Note that currently this solver uses easy heuristics for branching
-- and backtracking and therefore it is not perfect. Most probably this
-- solver fits for MIP problems that have not so many integer variables
-- (several tens, not more). Of course, hard or very large MIP problems
-- can't be solved by this routine.
--
-- *Control parameters*
--
-- The behavior of the glp_integer routine depends on a set of control
-- parameters which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note, for example, if the problem
--     has no integer feasible solution, the solver returns zero code);
-- 1 - it's not possible to start solving the problem due to incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_integer(void)
{     LP *lp;
      LPSOL *sol = NULL;
      struct rsm1_cp rsm_cp;
      int m, n, option, form, branch, btrack, ret;
      struct bbm1_cp bbm_cp;
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL)
      {  ret = 1;
         goto done;
      }
      if (lp->kind == NULL)
      {  error("glp_integer: problem has no integer variables");
         ret = 1;
         goto done;
      }
      m = lp->m, n = lp->n;
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* STAGE 1: FIND OPTIMAL SOLUTION OF RELAXED LP PROBLEM */
      /* set control parameters */
      rsm_cp.what = 2; /* optimal solution is required */
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     rsm_cp.form = 0; break;
         case GLP_RFI_BG:  rsm_cp.form = 1; break;
         case GLP_RFI_FT:  rsm_cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("scale", &rsm_cp.scale);
      glp_get_ipar("spx_use_dual", &rsm_cp.dual);
      glp_get_ipar("spx_steep", &rsm_cp.steep);
      glp_get_ipar("spx_relax", &rsm_cp.relax);
      glp_get_rpar("tol_bnd", &rsm_cp.tol_bnd);
      glp_get_rpar("tol_dj", &rsm_cp.tol_dj), rsm_cp.tol_dj *= 0.15;
      glp_get_rpar("tol_piv", &rsm_cp.tol_piv);
      rsm_cp.iter_max = 0;
      glp_get_ipar("round", &rsm_cp.round);
      /* solve LP problem by means of the revised simplex method */
      ret = rsm1_driver(lp, sol, &rsm_cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* check if the relaxed problem has been solved successfully */
      if (!(ret == 0 && sol->status == 'O'))
      {  error("glp_integer: integer optimization not possible");
         if (ret != 0) ret = 2;
         goto done;
      }
      /* STAGE 2: SOLVE MIP PROBLEM USING BRANCH-AND-BOUND METHOD */
      /* set control parameters */
      glp_get_ipar("option", &option);
      switch (option)
      {  case GLP_INI:     bbm_cp.what = 0; break;
         case GLP_ANY:     bbm_cp.what = 1; break;
         case GLP_FIN:     bbm_cp.what = 2; break;
         default:          insist(option != option);
      }
      glp_get_ipar("mip_branch", &branch);
      switch (branch)
      {  case GLP_FIRST:   bbm_cp.branch = BB_FIRST; break;
         case GLP_LAST:    bbm_cp.branch = BB_LAST;  break;
         case GLP_DRTOM:   bbm_cp.branch = BB_DRTOM; break;
         default:          insist(branch != branch);
      }
      glp_get_ipar("mip_btrack", &btrack);
      switch (btrack)
      {  case GLP_FIFO:    bbm_cp.btrack = BB_FIFO;  break;
         case GLP_LIFO:    bbm_cp.btrack = BB_LIFO;  break;
         case GLP_BESTP:   bbm_cp.btrack = BB_BESTP; break;
         default:          insist(btrack != btrack);
      }
      glp_get_rpar("tol_int", &bbm_cp.tol_int);
      glp_get_rpar("tol_obj", &bbm_cp.tol_obj);
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     bbm_cp.form = 0; break;
         case GLP_RFI_BG:  bbm_cp.form = 1; break;
         case GLP_RFI_FT:  bbm_cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("spx_steep", &bbm_cp.steep);
      glp_get_ipar("spx_relax", &bbm_cp.relax);
      glp_get_rpar("tol_bnd", &bbm_cp.tol_bnd);
      glp_get_rpar("tol_dj", &bbm_cp.tol_dj);
      glp_get_rpar("tol_piv", &bbm_cp.tol_piv);
      bbm_cp.iter_max = 0;
      glp_get_ipar("round", &bbm_cp.round);
      /* solve MIP problem by means of branch-and-bound method */
      ret = bbm1_driver(lp, sol, &bbm_cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* check if the integer problem has been solved successfully */
      if (ret != 0) ret = 2;
done: /* free working data structures */
      if (lp != NULL) delete_lp(lp);
      if (sol != NULL) delete_lpsol(sol);
      /* return to the application program */
      return ret;
}

/*----------------------------------------------------------------------
-- glp_interior - solve problem by means of the interior point method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_interior(void);
--
-- *Description*
--
-- The glp_interior routine is a *tentative* LP problem solver based on
-- the primal-dual interior point method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_interior routine performs the following actions:
--
-- preparing internal data structures;
-- transforming original LP problem to the standard form;
-- searching for optimal solution;
-- recovering found solution for the original LP problem;
-- storing computed solution back to the workspace.
--
-- On each iteration the glp_interior routine reports some information
-- about the current point. This information is sent to stdout and has
-- the following format:
--
--    nnn: F = fff; rpi = ppp; rdi = ddd; gap = ggg
--
-- where nnn is number of iteration (0 means the starting point), fff
-- is current value of the objective function (in case of maximization
-- this value has opposite sign), ppp is relative primal infeasibility,
-- ddd is relative dual infeasibility, ggg is relative primal-dual gap
-- (the difference between values of primal and dual objectives).
--
-- Note that the glp_interior solver is *tentative* and doesn't include
-- many important features (no dense column handling, no iterative
-- refinement of the solution, no optimal basis identification, etc.).
--
-- *Returns*
--
-- 0 - optimal solution found;
-- 1 - the solver can't start solving the problem because of incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_interior(void)
{     LP *lp;
      LPSOL *sol;
      int m, n, option, status, kase, ret;
      /* determine kind of the desired solution */
      glp_get_ipar("option", &option);
      if (option == GLP_INI)
      {  error("glp_interior: the interior point solver is not intended"
            " for computing initial basis solution");
         return 1;
      }
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL) return 1;
      m = lp->m, n = lp->n;
      /* if only primal feasible solution is needed, nullify objective
         function coefficients */
      if (option == GLP_ANY)
      {  int j;
         for (j = 0; j <= n; j++) lp->c[j] = 0.0;
      }
      /* warn about dense columns */
      if (m > 200)
      {  int j;
         for (j = 1; j <= n; j++)
         {  if (count_nz(lp->A, -j) > m / 10)
            {  print("*WARNING* THE PROBLEM HAS DENSE COLUMN(S)");
               break;
            }
         }
      }
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* solve LP problem using primal-dual interior point method */
      ret = ipm1_driver(lp, sol);
      if (ret == 0)
         status = (option == GLP_ANY ? GLP_FEAS : GLP_OPT);
      else
         status = GLP_UNDEF;
      /* store solution back to the workspace (incomplete) */
      glp_set_ipar("status", status);
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int k, seqn;
            glp_get_seqn(what, &seqn);
            k = (kase == 0 ? 0 : m) + seqn;
            /* all variables are superbasis */
            glp_set_activity(what, GLP_BS, sol->valx[k], sol->dx[k]);
         }
      }
      /* delete LP solution block */
      delete_lpsol(sol);
      /* delete LP data block */
      delete_lp(lp);
      /* returns to the application program */
      return (status != GLP_UNDEF ? 0 : 2);
}

/*----------------------------------------------------------------------
-- glp_new_coef - create new constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_new_coef(double val);
--
-- *Description*
--
-- The glp_new_coef creates new constraint coefficient, which is placed
-- in the current row and in the current column and has the value val.
-- Being created new coefficient becomes the current coefficient.
--
-- The application program needn't specify zero coefficients, because if
-- a coefficient was not specified, it considered as zero.
--
-- *Complexity*
--
-- This operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row or/and column are undefined. */

int glp_new_coef(double val)
{     GLPCOEF *coef;
      if (glp->this_row == NULL || glp->this_col == NULL) return 1;
      /* create new constraint coefficient */
      coef = get_atom(glp->coef_pool);
      coef->row = glp->this_row;
      coef->col = glp->this_col;
      coef->val = val;
      coef->next1 = glp->this_row->ptr, glp->this_row->ptr = coef;
      coef->next2 = glp->this_col->ptr, glp->this_col->ptr = coef;
      /* new coefficient becomes the current coefficient */
      glp->this_coef = coef;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_next_coef - find the next constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_next_coef(int what);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_next_coef routine finds the next
-- constraint coefficient placed in the same row as the current
-- coefficient and makes it be the current coefficient. At the same time
-- the routine also makes the column, which corresponds to the found
-- coefficient, be the current column (the current row remains unchanged
-- even if it doesn't correspond to the found coefficient).
--
-- If what = GLP_COL, the glp_next_coef routine finds the next
-- constraint coefficient placed in the same column as the current
-- coefficient and makes it be the current coefficient. At the same time
-- the routine also makes the row, which corresponds to the found
-- coefficient, be the current row (the current column remains unchanged
-- even if it doesn't correspond to the found coefficient).
--
-- The ordering of constraint coefficients can be changed by GLPK API
-- routines, therefore the application program shouldn't rely on some
-- particular ordering (except cases, when the particular ordering is
-- explictly declared).
--
-- *Complexity*
--
-- This operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current coefficient is undefined or there is no the next
--     coefficient in the same row or column;
-- 2 - the parameter what is invalid. */

int glp_next_coef(int what)
{     if (what == GLP_ROW)
      {  if (glp->this_coef == NULL) return 1;
         glp->this_coef = glp->this_coef->next1;
         if (glp->this_coef == NULL) return 1;
         glp->this_col = glp->this_coef->col;
      }
      else if (what == GLP_COL)
      {  if (glp->this_coef == NULL) return 1;
         glp->this_coef = glp->this_coef->next2;
         if (glp->this_coef == NULL) return 1;
         glp->this_row = glp->this_coef->row;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_next_item - find the next row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_next_item(int what);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_next_item routine finds the next row of
-- the problem, which follows the current row, and makes it be the
-- current row.
--
-- If what = GLP_COL, the glp_next_item routine finds the next column of
-- the problem, which follows the current column, and makes it be the
-- current column.
--
-- It's assumed that one row (column) precedes other row (column) if the
-- former was created before the latter. Hence, the next row (column) is
-- that one, which was created immediately after the current row
-- (column).
--
-- *Complexity*
--
-- The operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row (column) is undefined or there is no the next row
--     (column) in the problem;
-- 2 - the parameter what is invalid. */

int glp_next_item(int what)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         glp->this_row = glp->this_row->next;
         if (glp->this_row == NULL) return 1;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         glp->this_col = glp->this_col->next;
         if (glp->this_col == NULL) return 1;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_print_sol - write problem solution using printable format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_print_sol(char *fname);
--
-- *Description*
--
-- The glp_print_sol routine writes the problem solution using printable
-- format to the text file, whose name is the character string fname.
--
-- Information reported by the routine corresponds to the final simplex
-- table found by the solver. This information is intended mainly for
-- the visual analysis and has the meaning only if the problem has been
-- successfully solved.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_print_sol(char *fname)
{     FILE *fp;
      int kase;
      print("glp_print_sol: writing problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_print_sol: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char name[GLP_MAX_NAME+1];
         glp_get_cpar("problem", name);
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         glp_get_ipar("nr", &nr);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc, nc_int, nc_bin;
         glp_get_ipar("nc", &nc);
         glp_get_ipar("nc_int", &nc_int);
         glp_get_ipar("nc_bin", &nc_bin);
         fprintf(fp, "%-12s%d", "Columns:", nc);
         if (nc_int)
            fprintf(fp, " (%d integer%s, %d binar%s)",
               nc_int, nc_int == 1 ? "" : "s",
               nc_bin, nc_bin == 1 ? "y" : "ies");
         fprintf(fp, "\n");
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         glp_get_ipar("nz", &nz);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* objective function */
      {  char name[GLP_MAX_NAME+1];
         fprintf(fp, "%-12s", "Objective:");
         glp_get_cpar("obj_row", name);
         if (name[0] == '\0')
            fprintf(fp, "UNDEFINED\n");
         else
         {  int dir;
            double val;
            glp_get_ipar("obj_dir", &dir);
            glp_find_item(GLP_ROW, name);
            glp_get_activity(GLP_ROW, NULL, &val, NULL);
            fprintf(fp, "%s = %.6g %s\n", name, val,
               dir == GLP_MIN  ? "(MINimization)" :
               dir == GLP_MAX  ? "(MAXimization)" : "(???)");
         }
      }
      /* problem status */
      {  int status;
         glp_get_ipar("status", &status);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == GLP_UNDEF  ? "UNDEFINED" :
            status == GLP_OPT    ? "OPTIMAL" :
            status == GLP_FEAS   ? "FEASIBLE" :
            status == GLP_INFEAS ? "INFEASIBLE (INTERMEDIATE)" :
            status == GLP_NOFEAS ? "INFEASIBLE (FINAL)" :
            status == GLP_UNBND  ? "UNBOUNDED" :
            status == GLP_INTOPT ? "INTEGER OPTIMAL" :
            status == GLP_INTSOL ? "INTEGER FEASIBLE" :
            status == GLP_DISINT ? "INTEGER INFEASIBLE (INTERMEDIATE)" :
            status == GLP_NOINT  ? "INTEGER INFEASIBLE (FINAL)" : "???")
            ;
      }
      /* main output */
      for (kase = 0; kase <= 1; kase++)
      {  int what = kase == 0 ? GLP_ROW : GLP_COL, numb = 0, ret;
         char name[GLP_MAX_NAME+1];
         fprintf(fp, "\n");
         fprintf(fp, "  No. %-12s   St   Activity     Lower bound   Upp"
            "er bound    Marginal\n",
            what == GLP_ROW ? "  Row name" : "Column name");
         fprintf(fp, "----- ------------   -- ------------- -----------"
            "-- ------------- -------------\n");
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int kind, type, tagx;
            double lb, ub, valx, dx;
            /* row/column sequential number */
            numb++;
            fprintf(fp, "%5d ", numb);
            /* row column/name */
            glp_get_name(what, name);
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%19s", name, "");
            /* get row/column information */
            glp_get_kind(what, &kind);
            glp_get_bounds(what, &type, &lb, &ub);
            glp_get_activity(what, &tagx, &valx, &dx);
            /* row/column kind */
            fprintf(fp, "%s ", kind ? "*" : " ");
            /* row/column status */
            if (tagx == GLP_BS)
            {  if (type == GLP_LO && valx < lb ||
                   type == GLP_DB && valx < lb ||
                   type == GLP_FX && valx < lb)
                  fprintf(fp, "B- ");
               else if (type == GLP_UP && valx > ub ||
                        type == GLP_DB && valx > ub ||
                        type == GLP_FX && valx > ub)
                  fprintf(fp, "B+ ");
               else if (kind && valx != floor(valx + 0.5))
                  fprintf(fp, "B* ");
               else
                  fprintf(fp, "B  ");
            }
            else if (tagx == GLP_NL)
               fprintf(fp, "NL ");
            else if (tagx == GLP_NU)
               fprintf(fp, "NU ");
            else if (tagx == GLP_NF)
               fprintf(fp, "NF ");
            else if (tagx == GLP_NS)
               fprintf(fp, "NS ");
            else
               fprintf(fp, "??");
            /* row/column primal activity */
            fprintf(fp, "%13.6g ", valx);
            /* row/column lower bound */
            if (type == GLP_LO || type == GLP_DB || type == GLP_FX)
               fprintf(fp, "%13.6g ", lb);
            else
               fprintf(fp, "%13s ", "");
            /* row/column upper bound */
            if (type == GLP_UP || type == GLP_DB)
               fprintf(fp, "%13.6g ", ub);
            else if (type == GLP_FX)
               fprintf(fp, "%13s ", "=");
            else
               fprintf(fp, "%13s ", "");
            /* row/column dual activity */
            if (tagx != GLP_BS)
            {  if (dx == 0.0)
                  fprintf(fp, "%13s", "< eps");
               else
                  fprintf(fp, "%13.6g", dx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  error("glp_print_sol: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/*----------------------------------------------------------------------
-- glp_read_lpm - read linear programming model written in GLPK/L.
--
-- *Synopsis*
--
-- #include "glpk.h:
-- int glp_read_lpm(char *fname);
--
-- *Description*
--
-- The glp_read_lpm routine reads an LP model written in the modeling
-- language GLPK/L from the text file whose name is the character string
-- fname into the workspace.
--
-- As a rule the workspace should be empty before a call to the
-- glp_read_lpm routine, i.e. the workspace should contain no rows and
-- no columns.
--
-- For detailed description of GLPK/L modeling language see the program
-- documentation.
--
-- *Control parameters*
--
-- The behavior of the glp_read_lpm routine depends on the following
-- control parameters:
--
-- fn_gener (name of the file to output generated LP/MIP problem).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_read_lpm(char *fname)
{     struct prob *prob = NULL;
      int m, n, i, j;
      char fn_gener[MAX_NAME+1];
      /* initialize the language processor environment */
      if (initialize(fname) != 0) goto fail;
      /* set error handling */
      pdb->flag = 1;
      if (setjmp(pdb->jump)) goto fail;
      /* parse model description */
      load_model();
      /* output generated LP/MIP problem to the specified file in plain
         text format */
      glp_get_cpar("fn_gener", fn_gener);
      if (fn_gener[0] != '\0')
      {  int ret;
         ret = gener_lp(fn_gener);
         if (ret != 0) goto fail;
      }
      /* create data structure for generating LP/MIP */
      prob = create_prob();
      m = prob->m;
      n = prob->n;
      /* set problem name */
      glp_set_cpar("problem", pdb->model_name);
      /* create columns that correspond model variables */
      for (j = 1; j <= n; j++)
      {  VAR *var = prob->memb[m+j]->link;
         char *name = gener_name(prob, m+j);
         int type;
         if (glp_create_item(GLP_COL, name))
         {  error("glp_read_lpm: error on creating column `%s'", name);
            goto fail;
         }
         if (var->kind) glp_set_kind(GLP_COL, GLP_YES);
         switch (var->type)
         {  case 'F': type = GLP_FR; break;
            case 'L': type = GLP_LO; break;
            case 'U': type = GLP_UP; break;
            case 'D': type = GLP_DB; break;
            case 'S': type = GLP_FX; break;
            default: insist(var->type != var->type);
         }
         glp_set_bounds(GLP_COL, type, var->lb, var->ub);
      }
      /* create rows that correspond model constraints; build the
         constraint matrix */
      for (i = 1; i <= m; i++)
      {  CONS *cons = prob->memb[i]->link;
         char *name = gener_name(prob, i);
         struct elem *form, *e;
         int type;
         if (glp_create_item(GLP_ROW, name))
         {  error("glp_read_lpm: error on creating row `%s'", name);
            goto fail;
         }
         form = build_form(prob, i);
         if (form == NULL) goto fail;
         for (e = form; e != NULL; e = e->next)
         {  if (e->j == 0)
            {  if (cons->type == 'F')
               {  error("glp_read_lpm: free row `%s' has constant term",
                     name);
                  goto fail;
               }
               cons->lb -= e->val, cons->ub -= e->val;
            }
            else
            {  glp_find_item(GLP_COL, gener_name(prob, m+e->j));
               glp_new_coef(e->val);
            }
         }
         erase_form(prob, form);
         switch (cons->type)
         {  case 'F': type = GLP_FR; break;
            case 'L': type = GLP_LO; break;
            case 'U': type = GLP_UP; break;
            case 'D': type = GLP_DB; break;
            case 'S': type = GLP_FX; break;
            default: insist(cons->type != cons->type);
         }
         glp_set_bounds(GLP_ROW, type, cons->lb, cons->ub);
      }
      /* set the objective function */
      switch (prob->obj_dir)
      {  case '-':
            insist(glp_set_ipar("obj_dir", GLP_MIN) == 0);
            break;
         case '+':
            insist(glp_set_ipar("obj_dir", GLP_MAX) == 0);
            break;
         default:
            insist(prob->obj_dir != prob->obj_dir);
      }
      glp_set_rpar("c0", 0.0);
      if (prob->obj_row == 0)
      {  glp_set_ipar("option", GLP_ANY);
         glp_set_cpar("obj_row", "");
      }
      else
      {  glp_set_ipar("option", GLP_FIN);
         glp_set_cpar("obj_row", gener_name(prob, prob->obj_row));
      }
      /* free auxiliary data structure */
      delete_prob(prob);
      /* terminate the language processor environment */
      terminate();
      /* model has been read successfully */
      return 0;
fail: /* the operation failed */
      error("glp_read_lpm: processing terminated due to errors");
      if (prob != NULL) delete_prob(prob);
      if (pdb != NULL) terminate();
      return 1;
}

/*----------------------------------------------------------------------
-- glp_read_mps - read problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_read_mps(char *fname)
--
-- *Description*
--
-- The glp_read_mps routine reads LP problem data using MPS format from
-- the text file named fname into the workspace.
--
-- As a rule the workspace should be empty before a call to the
-- glp_read_mps routine, i.e. the workspace should contain no rows and
-- no columns.
--
-- Detailed description of the MPS format can be found, for example,
-- in the following book:
--
-- B.A.Murtagh. Advanced Linear Programming: Computation and Practice.
-- McGraw-Hill, 1981.
--
-- *Control parameters*
--
-- The behavior of the glp_read_mps routine depends on the following
-- control parameters:
--
-- mps_obj_name (the name of row that specifies the objective function);
-- mps_rhs_name (the name of right-hand side vector);
-- mps_rng_name (the name of range vector);
-- mps_bnd_name (the name of bound vector).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_read_mps(char *fname)
{     MPS *mps;
      MPSROW *row; MPSCOL *col; MPSCQE *cqe; MPSBQE *bqe;
      int m, n, i, j, v;
      char obj[8+1], rhs[8+1], rng[8+1], bnd[8+1];
      glp_get_cpar("mps_obj_name", obj);
      glp_get_cpar("mps_rhs_name", rhs);
      glp_get_cpar("mps_rng_name", rng);
      glp_get_cpar("mps_bnd_name", bnd);
      /* load MPS data block */
      mps = load_mps(fname);
      if (mps == NULL) goto fail;
      m = mps->n_row;
      n = mps->n_col;
      /* set problem name */
      glp_set_cpar("problem", mps->name);
      /* process ROWS section */
      for (i = 1; i <= m; i++)
      {  int type;
         row = mps->row[i];
         if (glp_create_item(GLP_ROW, row->name))
         {  error("glp_read_mps: error on creating row `%s'",
               row->name);
            goto fail;
         }
         if (strcmp(row->type, "N") == 0)
            type = GLP_FR;
         else if (strcmp(row->type, "L") == 0)
            type = GLP_UP;
         else if (strcmp(row->type, "G") == 0)
            type = GLP_LO;
         else if (strcmp(row->type, "E") == 0)
            type = GLP_FX;
         else
         {  error("glp_read_mps: row `%s' has unknown type `%s'",
               row->name, row->type);
            goto fail;
         }
         glp_set_bounds(GLP_ROW, type, 0.0, 0.0);
      }
      /* process COLUMNS section */
      for (j = 1; j <= n; j++)
      {  col = mps->col[j];
         if (glp_create_item(GLP_COL, col->name))
         {  error("glp_read_mps: error on creating column `%s'",
               col->name);
            goto fail;
         }
         glp_set_bounds(GLP_COL, GLP_LO, 0.0, 0.0);
         glp_set_kind(GLP_COL, col->flag ? GLP_YES : GLP_NO);
         for (cqe = col->ptr; cqe != NULL; cqe = cqe->next)
         {  glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_new_coef(cqe->val);
         }
      }
      /* set the objective function */
      glp_set_ipar("option", GLP_ANY);
      glp_set_cpar("obj_row", "");
      for (i = 1; i <= m; i++)
      {  row = mps->row[i];
         if (obj[0] == '\0' && strcmp(row->type, "N") == 0 ||
             obj[0] != '\0' && strcmp(row->name, obj) == 0)
         {  strcpy(obj, row->name);
            glp_set_ipar("option", GLP_FIN);
            glp_set_cpar("obj_row", obj);
            break;
         }
      }
      if (obj[0] != '\0' && i > m)
      {  error("glp_read_mps: objective function row `%s' not found",
            obj);
         goto fail;
      }
      /* process RHS section */
      if (rhs == NULL || rhs[0] == '\0')
         v = (mps->n_rhs == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rhs; v++)
            if (strcmp(mps->rhs[v]->name, rhs) == 0) break;
         if (v > mps->n_rhs)
         {  error("glp_read_mps: right-hand side vector `%s' not found",
               rhs);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rhs[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type = GLP_FR;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_get_bounds(GLP_ROW, &type, NULL, NULL);
            switch (type)
            {  case GLP_FR:
                  /* if the current row is the objective function row,
                     specified right-hand side is considered as the
                     constant term of the objective function with
                     opposite sign; in other cases specified right-hand
                     side is ignored */
                  if (strcmp(mps->row[cqe->ind]->name, obj) == 0)
                     glp_set_rpar("c0", - cqe->val);
                  break;
               case GLP_LO:
                  lb = cqe->val;
                  break;
               case GLP_UP:
                  ub = cqe->val;
                  break;
               case GLP_FX:
                  lb = ub = cqe->val;
                  break;
               default:
                  insist(type != type);
            }
            glp_set_bounds(GLP_ROW, type, lb, ub);
         }
      }
      /* process RANGES section */
      if (rng == NULL || rng[0] == '\0')
         v = (mps->n_rng == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rng; v++)
            if (strcmp(mps->rng[v]->name, rng) == 0) break;
         if (v > mps->n_rng)
         {  error("glp_read_mps: range vector `%s' not found", rng);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rng[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type = GLP_FR;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            switch (type)
            {  case GLP_FR:
                  error("glp_read_mps: range vector entry refers to row"
                     " `%s' of N type", mps->row[cqe->ind]->name);
                  goto fail;
               case GLP_LO:
                  ub = lb + fabs(cqe->val);
                  break;
               case GLP_UP:
                  lb = ub - fabs(cqe->val);
                  break;
               case GLP_FX:
                  if (cqe->val >= 0.0)
                     ub += fabs(cqe->val);
                  else
                     lb -= fabs(cqe->val);
                  break;
               default:
                  insist(type != type);
            }
            glp_set_bounds(GLP_ROW, lb == ub ? GLP_FX : GLP_DB, lb, ub);
         }
      }
      /* process BOUNDS section */
      if (bnd == NULL || bnd[0] == '\0')
         v = (mps->n_bnd == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_bnd; v++)
            if (strcmp(mps->bnd[v]->name, bnd) == 0) break;
         if (v > mps->n_bnd)
         {  error("glp_read_mps: bound vector `%s' not found", bnd);
            goto fail;
         }
      }
      if (v > 0)
      {  for (bqe = mps->bnd[v]->ptr; bqe != NULL; bqe = bqe->next)
         {  int type = GLP_FX;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_COL, mps->col[bqe->ind]->name);
            glp_get_bounds(GLP_COL, &type, &lb, &ub);
            if (type == GLP_FR || type == GLP_UP) lb = -DBL_MAX;
            if (type == GLP_FR || type == GLP_LO) ub = +DBL_MAX;
            if (strcmp(bqe->type, "LO") == 0)
               lb = bqe->val;
            else if (strcmp(bqe->type, "UP") == 0)
               ub = bqe->val;
            else if (strcmp(bqe->type, "FX") == 0)
               lb = ub = bqe->val;
            else if (strcmp(bqe->type, "FR") == 0)
               lb = -DBL_MAX, ub = +DBL_MAX;
            else if (strcmp(bqe->type, "MI") == 0)
               lb = -DBL_MAX;
            else if (strcmp(bqe->type, "PL") == 0)
               ub = +DBL_MAX;
            else if (strcmp(bqe->type, "UI") == 0)
            {  /* integer structural variable with upper bound */
               glp_set_kind(GLP_COL, GLP_YES);
               ub = bqe->val;
            }
            else if (strcmp(bqe->type, "BV") == 0)
            {  /* binary structural variable */
               glp_set_kind(GLP_COL, GLP_YES);
               lb = 0.0, ub = 1.0;
            }
            else
            {  error("glp_read_mps: bound vector entry for column `%s' "
                  "has unknown type `%s'",
                  mps->col[bqe->ind]->name, bqe->type);
               goto fail;
            }
            if (lb == -DBL_MAX && ub == +DBL_MAX)
               type = GLP_FR;
            else if (ub == +DBL_MAX)
               type = GLP_LO;
            else if (lb == -DBL_MAX)
               type = GLP_UP;
            else if (lb != ub)
               type = GLP_DB;
            else
               type = GLP_FX;
            glp_set_bounds(GLP_COL, type, lb, ub);
         }
      }
      /* free MPS data block */
      free_mps(mps);
      /* return to the application program */
      return 0;
fail: /* the operation failed */
      if (mps != NULL) free_mps(mps);
      return 1;
}

/*----------------------------------------------------------------------
-- glp_set_activity - set activity of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_activity(int what, int tagx, double valx, double dx);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_activity routine assigns the computed
-- solution to the current row.
--
-- If what = GLP_COL, the glp_set_activity routine assigns the computed
-- solution to the current column.
--
-- The parameter tagx specifies the status of row/column:
--
-- GLP_BS - the variable is basis;
-- GLP_NL - the variable is non-basis and placed on its lower bound
--          (only for variables of GLP_LO and GLP_DB types);
-- GLP_NU - the variable is non-basis and placed on its upper bound
--          (only for variables of GLP_UP and GLP_DB types);
-- GLP_NF - the free variable is non-basis (only for variables of
--          GLP_FR type);
-- GLP_NS - the fixed variable is non-basis (only for variables of
--          GLP_FX type).
--
-- The parameter valx specifies the primal activity (the computed value
-- of the variable).
--
-- The parameter dx specifies the dual activity (the computed marginal
-- value of the variable).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid, or the parameter tagx specifies
--     invalid status. */

int glp_set_activity(int what, int tagx, double valx, double dx)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      if (!(tagx == GLP_BS || tagx == GLP_NL || tagx == GLP_NU ||
            tagx == GLP_NF || tagx == GLP_NS)) return 2;
      item->tagx = tagx;
      item->valx = valx;
      item->dx = dx;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_bounds - set bounds of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_bounds(int what, int type, double lb, double ub);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_bounds routine sets (changes) the
-- type and bounds of the current row.
--
-- If what = GLP_COL, the glp_set_bounds routine sets (changes) the
-- type and bounds of the current column.
--
-- Parameters type, lb, and ub specify respectively the type, the lower
-- bound, and the upper bound, which should be set for the current row
-- or column:
--
--     Type          Bounds              Note
--    ------    -----------------    -------------
--    GLP_FR    -inf <  x <  +inf    free variable
--    GLP_LO      lb <= x <  +inf    lower bound
--    GLP_UP    -inf <  x <=  ub     upper bound
--    GLP_DB      lb <= x <=  ub     double bound
--    GLP_FX            x  =  lb     fixed variable
--
-- where x is the auxiliary (in case of row) or structural (in case of
-- column) variable.
--
-- If the current row/column has no lower or/and upper bound, the
-- corresponding values of lb or/and ub are ignored. If the current
-- row/column is of GLP_FX type, the fixed value should be specified by
-- the parameter lb, and the parameter ub is ignored.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid, or the parameter type is invalid,
--     or lb >= ub (in case of type = GLP_DB only). */

int glp_set_bounds(int what, int type, double lb, double ub)
{     GLPITEM *item;
      switch (what)
      {  case GLP_ROW:
            item = glp->this_row;
            break;
         case GLP_COL:
            item = glp->this_col;
            break;
         default:
            return 2;
      }
      if (item == NULL) return 1;
      switch (type)
      {  case GLP_FR:
            item->type = GLP_FR;
            item->lb = item->ub = 0.0;
            break;
         case GLP_LO:
            item->type = GLP_LO;
            item->lb = lb, item->ub = 0.0;
            break;
         case GLP_UP:
            item->type = GLP_UP;
            item->lb = 0.0, item->ub = ub;
            break;
         case GLP_DB:
            item->type = GLP_DB;
            item->lb = lb, item->ub = ub;
            break;
         case GLP_FX:
            item->type = GLP_FX;
            item->lb = item->ub = lb;
            break;
         default:
            return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_cpar - set value of text control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_cpar(char *name, char *val);
--
-- *Description*
--
-- The glp_set_cpar routine assigns the character string val to the text
-- control parameter with the given name.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid;
-- 2 - the given value (character string) is invalid. */

int glp_set_cpar(char *name, char *val)
{     if (strcmp(name, "fn_gener") == 0)
      {  /* file name to output generated LP/MIP problem */
         if (strlen(val) > GLP_MAX_NAME) return 2;
         if (glp->fn_gener == NULL)
            glp->fn_gener = create_str(glp->str_pool);
         if (val[0] == '\0')
         {  delete_str(glp->fn_gener);
            glp->fn_gener = NULL;
         }
         else
            set_str(glp->fn_gener, val);
      }
      else if (strcmp(name, "mps_bnd_name") == 0)
      {  /* the name of bound vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_bnd_name, val);
      }
      else if (strcmp(name, "mps_obj_name") == 0)
      {  /* the name of the objective function */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_obj_name, val);
      }
      else if (strcmp(name, "mps_rhs_name") == 0)
      {  /* the name of the right-hand side (RHS) vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_rhs_name, val);
      }
      else if (strcmp(name, "mps_rng_name") == 0)
      {  /* the name of the range vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_rng_name, val);
      }
      else if (strcmp(name, "obj_row") == 0)
      {  /* the name of the objective function */
         if (val[0] == '\0')
            glp->obj_row = NULL;
         else
         {  STR *row_name;
            AVLNODE *node;
            if (glp_check_name(val)) return 2;
            row_name = set_str(create_str(glp->str_pool), val);
            node = find_by_key(glp->row_tab, row_name);
            delete_str(row_name);
            if (node == NULL) return 2; /* row not found */
            glp->obj_row = node->link;
         }
         /* nullify the constant term, because the objective function
            has been changed */
         glp->c0 = 0.0;
      }
      else if (strcmp(name, "problem") == 0)
      {  /* the name of problem */
         if (glp_check_name(val)) return 2;
         set_str(glp->problem, val);
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_ipar - set value of integer control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_ipar(char *name, int val);
--
-- *Description*
--
-- The glp_set_ipar routine assigns the given value val to the integer
-- control parameter with the given name.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid;
-- 2 - the given value is invalid. */

int glp_set_ipar(char *name, int val)
{     if (strcmp(name, "mip_branch") == 0)
      {  /* branching technique flag */
         if (!(val == GLP_FIRST || val == GLP_LAST || val == GLP_DRTOM))
            return 2;
         glp->mip_branch = val;
      }
      else if (strcmp(name, "mip_btrack") == 0)
      {  /* backtracking technique flag */
         if (!(val == GLP_FIFO || val == GLP_LIFO || val == GLP_BESTP))
            return 2;
         glp->mip_btrack = val;
      }
      else if (strcmp(name, "mps_one_entry") == 0)
      {  /* write MPS file not using fields 5 and 6 */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->mps_one_entry = val;
      }
      else if (strcmp(name, "mps_pedantic") == 0)
      {  /* write MPS file using pedantic style */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->mps_pedantic = val;
      }
      else if (strcmp(name, "mps_skip_empty") == 0)
      {  /* don't write empty columns to MPS file */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->mps_skip_empty = val;
      }
      else if (strcmp(name, "mps_use_names") == 0)
      {  /* use original names as templates to generate MPS names */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->mps_use_names = val;
      }
      else if (strcmp(name, "obj_dir") == 0)
      {  /* optimization direction flag */
         if (!(val == GLP_MIN || val == GLP_MAX)) return 2;
         glp->obj_dir = val;
      }
      else if (strcmp(name, "option") == 0)
      {  /* what solution should be found by the solver */
         if (!(val == GLP_INI || val == GLP_ANY || val == GLP_FIN))
            return 2;
         glp->option = val;
      }
      else if (strcmp(name, "round") == 0)
      {  /* round the computed values of basis variable */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->round = val;
      }
      else if (strcmp(name, "scale") == 0)
      {  /* scale the problem before solving */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->scale = val;
      }
      else if (strcmp(name, "spx_form") == 0)
      {  /* the form of the basis matrix used by the solver */
         if (!(val == GLP_EFI ||
               val == GLP_RFI_BG || val == GLP_RFI_FT)) return 2;
         glp->spx_form = val;
      }
      else if (strcmp(name, "spx_relax") == 0)
      {  /* use the ratio test proposed by P.Harris */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->spx_relax = val;
      }
      else if (strcmp(name, "spx_steep") == 0)
      {  /* use the steepest edge technique */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->spx_steep = val;
      }
      else if (strcmp(name, "spx_use_dual") == 0)
      {  /* use dual simplex to search for feasible solution */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->spx_use_dual = val;
      }
      else if (strcmp(name, "status") == 0)
      {  /* status of the computed solution (set bythe solver) */
         if (!(val == GLP_UNDEF  || val == GLP_OPT    ||
               val == GLP_FEAS   || val == GLP_INFEAS ||
               val == GLP_NOFEAS || val == GLP_UNBND  ||
               val == GLP_INTOPT || val == GLP_INTSOL ||
               val == GLP_DISINT || val == GLP_NOINT)) return 2;
         glp->status = val;
      }
      else if (strcmp(name, "sum_aij") == 0)
      {  /* sum multiplets of the constraint matrix */
         if (!(val == GLP_NO || val == GLP_YES)) return 2;
         glp->sum_aij = val;
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_kind - set kind of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_kind(int what, int kind);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_kind routine sets (changes) the kind
-- of the current row.
--
-- If what = GLP_COL, the glp_set_kind routine sets (changes) the kind
-- of the current column.
--
-- The parameter kind specifies the kind of the current row or column
-- which has to be set. GLP_NO means that the variable is continuous,
-- GLP_YES means that the variable is integer (discrete).
--
-- Note that only columns (structural variables) can be of integer kind.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid or the parameter kind is invalid
--     or kind = GLP_YES is specified for row. */

int glp_set_kind(int what, int kind)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         if (kind == GLP_NO)
            glp->this_row->kind = 0;
         else
            return 2;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         if (kind == GLP_NO)
            glp->this_col->kind = 0;
         else if (kind == GLP_YES)
            glp->this_col->kind = 1;
         else
            return 2;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_name - change name of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_name(int what, char *name);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_name routine assigns the new name to
-- the current row.
--
-- If what = GLP_COL, the glp_set_name routine assigns the new name to
-- the current column.
--
-- *Complexity*
--
-- This operation has time complexity O(log n), where n is the current
-- number of rows or columns in the workspace.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined or row/column with the given
--     name already exists;
-- 2 - the parameter what or the given name is invalid. */

int glp_set_name(int what, char *name)
{     if (what == GLP_ROW)
      {  STR *row_name;
         AVLNODE *node;
         if (glp->this_row == NULL) return 1;
         if (glp_check_name(name)) return 2;
         row_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->row_tab, row_name) != NULL)
         {  /* row already exists */
            delete_str(row_name);
            return 1;
         }
         node = find_by_key(glp->row_tab, glp->this_row->name);
         insist(node != NULL);
         delete_node(glp->row_tab, node);
         delete_str(glp->this_row->name);
         glp->this_row->name = row_name;
         node = insert_by_key(glp->row_tab, glp->this_row->name);
         node->link = glp->this_row;
      }
      else if (what == GLP_COL)
      {  STR *col_name;
         AVLNODE *node;
         if (glp->this_col == NULL) return 1;
         if (glp_check_name(name)) return 2;
         col_name = set_str(create_str(glp->str_pool), name);
         if (find_by_key(glp->col_tab, col_name) != NULL)
         {  /* column already exists */
            delete_str(col_name);
            return 1;
         }
         node = find_by_key(glp->col_tab, glp->this_col->name);
         insist(node != NULL);
         delete_node(glp->col_tab, node);
         delete_str(glp->this_col->name);
         glp->this_col->name = col_name;
         node = insert_by_key(glp->col_tab, glp->this_col->name);
         node->link = glp->this_col;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_rpar - set value of real control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_rpar(char *name, double val);
--
-- *Description*
--
-- The glp_set_rpar routine assigns the given value val to the real
-- control parameter with the given name.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid;
-- 2 - the given value is invalid. */

int glp_set_rpar(char *name, double val)
{     if (strcmp(name, "c0") == 0)
      {  /* the constant term of the objective function */
         glp->c0 = val;
      }
      else if (strcmp(name, "tol_aij") == 0)
      {  /* tolerance used by the solver to drop small elements of the
            constraint matrix */
         if (!(0.0 <= val && val <= 1.0)) return 2;
         glp->tol_aij = val;
      }
      else if (strcmp(name, "tol_bnd") == 0)
      {  /* tolerance used by the solver to check primal feasibility of
            the current basis solution */
         if (!(0.0 <= val <= 1.0)) return 2;
         glp->tol_bnd = val;
      }
      else if (strcmp(name, "tol_dj") == 0)
      {  /* tolerance used by the solver to check dual feasibility of
            the current basis solution */
         if (!(0.0 <= val <= 1.0)) return 2;
         glp->tol_bnd = val;
      }
      else if (strcmp(name, "tol_int") == 0)
      {  /* tolerance used by MIP solver to check integer feasibility
            of the current basis solution */
         if (!(0.0 <= val <= 1.0)) return 2;
         glp->tol_int = val;
      }
      else if (strcmp(name, "tol_obj") == 0)
      {  /* tolerance used by MIP solver to compare values of objective
            function */
         if (!(0.0 <= val <= 1.0)) return 2;
         glp->tol_obj = val;
      }
      else if (strcmp(name, "tol_piv") == 0)
      {  /* tolerance used by the solver to check pivoting element */
         if (!(0.0 <= val && val <= 1.0)) return 2;
         glp->tol_piv = val;
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_set_seqn - set seqn attribute of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_seqn(int what, int seqn);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_seqn routine assigns the given value
-- seqn to the seqn attribute of the current row.
--
-- If what = GLP_COL, the glp_set_seqn routine assigns the fiven value
-- seqn to the seqn attribute of the current column.
--
-- The seqn attribute is an additional attribute of rows and columns.
-- It is just an integer value, which may be used for any purposes.
-- This attribute is intended mainly for needs of GLPK routines, however
-- it may be used by the application program.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_set_seqn(int what, int seqn)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         glp->this_row->seqn = seqn;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         glp->this_col->seqn = seqn;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- glp_simplex - solve problem by means of the revised simplex method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_simplex(void);
--
-- *Description*
--
-- The glp_simplex routine is a LP problem solver based on the two-phase
-- revised simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_simplex routine performs the following:
--
-- preparing internal data structures;
-- searching for feasible basis solution (pahse I);
-- searching for optimal basis solution (phase II);
-- storing the computed solution to the workspace.
--
-- Since large-scale problems may take a long time, the glp_simplex
-- routine reports some visual information about current status of the
-- search. This information is sent to stdout approximately once per
-- second and has the following format:
--
--    *nnn:   objval = xxx   infsum = yyy (ddd)
--
-- where nnn is an iteration count, xxx is the current value of the
-- objective function (which is unscaled and has correct sign), yyy is
-- the sum of infeasibilities (which is scaled and therefore it may be
-- used only for visual estimating), ddd is the current number of basis
-- fixed variables. If the asterisk (*) precedes to nnn, the solver is
-- searching for optimal solution (i.e. the feasible solution has been
-- found yet), otherwise the solver is searching for some feasible
-- solution.
--
-- Please note that this solver is not perfect. Although it has been
-- successfully tested on a wide set of LP problems, there are so called
-- hard problems, which can't be solved by this solver.
--
-- *Control parameters*
--
-- The behavior of the glp_simplex routine depends on a set of control
-- parameters which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note, for example, if the problem
--     has no feasible solution, the solver returns zero code);
-- 1 - it's not possible to start solving the problem due to incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_simplex(void)
{     LP *lp;
      LPSOL *sol;
      struct rsm1_cp cp;
      int m, n, option, form, ret;
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL) return 1;
      m = lp->m, n = lp->n;
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* set control parameters */
      glp_get_ipar("option", &option);
      switch (option)
      {  case GLP_INI:     cp.what = 0; break;
         case GLP_ANY:     cp.what = 1; break;
         case GLP_FIN:     cp.what = 2; break;
         default:          insist(option != option);
      }
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     cp.form = 0; break;
         case GLP_RFI_BG:  cp.form = 1; break;
         case GLP_RFI_FT:  cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("scale", &cp.scale);
      glp_get_ipar("spx_use_dual", &cp.dual);
      glp_get_ipar("spx_steep", &cp.steep);
      glp_get_ipar("spx_relax", &cp.relax);
      glp_get_rpar("tol_bnd", &cp.tol_bnd);
      glp_get_rpar("tol_dj", &cp.tol_dj);
      glp_get_rpar("tol_piv", &cp.tol_piv);
      cp.iter_max = 0;
      glp_get_ipar("round", &cp.round);
      /* solve the problem by means of the revised simplex method */
      ret = rsm1_driver(lp, sol, &cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* free working data structures */
      delete_lp(lp);
      delete_lpsol(sol);
      /* return to the application program */
      return ret == 0 ? 0 : 2;
}

/*----------------------------------------------------------------------
-- glp_terminate - terminate GLPK application program interface.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_terminate(void);
--
-- *Description*
--
-- The glp_terminate routine terminates the GLPK application program
-- interface (API) and frees all memory allocated to the workspace. All
-- data in the workspace are destroyed.
--
-- As a rule this routine is used before termination of the application
-- program. However, it may be used also for cleaning the workspace --
-- in this case the application program should call the glp_initialize
-- routine again for re-initializing API.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - GLPK API already terminated. */

int glp_terminate(void)
{     if (glp == NULL) return 1;
      delete_pool(glp->str_pool);
      delete_pool(glp->item_pool);
      delete_pool(glp->coef_pool);
      delete_avl(glp->row_tab);
      delete_avl(glp->col_tab);
      ufree(glp);
      glp = NULL;
      return 0;
}

/*----------------------------------------------------------------------
-- glp_write_mps - write problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_write_mps(char *fname);
--
-- *Description*
--
-- The glp_write_routine writes LP problem data using MPS format from
-- the workspace to the output text file, whose name is the character
-- string fname.
--
-- Should note that the MPS format allows row and column names up to 8
-- characters. So, if some rows or columns presented in the workspace
-- contain more 8 characters, the routine will reduce such names.
-- However, the routine make all reduced names be unique.
--
-- Detailed description of the MPS format can be found, for example,
-- in the following book:
--
-- B.A.Murtagh. Advanced Linear Programming: Computation and Practice.
-- McGraw-Hill, 1981.
--
-- *Control parameters*
--
-- The behavior of the glp_write_mps routine depends on the following
-- control parameters:
--
-- mps_use_names (reduce long names using original names);
-- mps_one_entry (don't use fields 5 and 6);
-- mps_pedantic (always put column and vector names to the field 2);
-- mps_skip_empty (skip empty columns).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

typedef char mps_name[8+1];
/* standard MPS names (can contain up to 8 characters) */

static void make_names(int what, mps_name alias[]);
/* generate standard MPS names using original names */

static char *mps_number(double val);
/* convert number to standard 12-character MPS format */

int glp_write_mps(char *fname)
{     FILE *fp;
      mps_name *row_name = NULL, *col_name = NULL;
      int use_names, one_entry, pedantic, skip_empty;
      int nrows, ncols, ret, flag, obj_seqn;
      int marker = 0; /* intorg/intend marker count */
      print("glp_write_mps: writing problem data to `%s'...", fname);
      glp_get_ipar("mps_use_names", &use_names);
      glp_get_ipar("mps_one_entry", &one_entry);
      glp_get_ipar("mps_pedantic", &pedantic);
      glp_get_ipar("mps_skip_empty", &skip_empty);
      /* open the output text file */
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_write_mps: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* count and renumber rows and columns (their sequential numbers
         are placed in the seqn fields) */
      nrows = 0;
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
         glp_set_seqn(GLP_ROW, ++nrows);
      ncols = 0;
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL))
         glp_set_seqn(GLP_COL, ++ncols);
      /* the problem should contain at least one row and one column */
      if (nrows == 0)
      {  error("glp_write_mps: problem has no rows");
         goto fail;
      }
      if (ncols == 0)
      {  error("glp_write_mps: problem has no columns");
         goto fail;
      }
      /* allocate arrays for 8-character row and column names */
      row_name = ucalloc(1+nrows, sizeof(mps_name));
      col_name = ucalloc(1+ncols, sizeof(mps_name));
      /* generate 8-character row and column names (the 8-character
         name of the row or the column having number seqn is placed to
         row_name[seqn] or col_name[seqn] respectively) */
      if (use_names)
      {  /* use original row and column names as templates */
         make_names(GLP_ROW, row_name);
         make_names(GLP_COL, col_name);
      }
      else
      {  /* make plain names based on sequential numbers */
         int k;
         char *t;
         for (k = 1; k <= nrows; k++)
         {  sprintf(row_name[k], "R%7d", k);
            for (t = row_name[k]; *t; t++) if (*t == ' ') *t = '_';
         }
         for (k = 1; k <= ncols; k++)
         {  sprintf(col_name[k], "C%7d", k);
            for (t = col_name[k]; *t; t++) if (*t == ' ') *t = '_';
         }
      }
      /* write NAME indicator card */
      {  char name[GLP_MAX_NAME+1];
         glp_get_cpar("problem", name), name[8] = '\0';
         fprintf(fp, "NAME          %s\n", name);
      }
      /* write ROWS section */
      fprintf(fp, "ROWS\n");
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
      {  int seqn, type;
         glp_get_seqn(GLP_ROW, &seqn);
         glp_get_bounds(GLP_ROW, &type, NULL, NULL);
         switch (type)
         {  case GLP_FR: type = 'N'; break;
            case GLP_LO: type = 'G'; break;
            case GLP_UP: type = 'L'; break;
            case GLP_DB: type = 'E'; break;
            case GLP_FX: type = 'E'; break;
            default: insist(type != type);
         }
         fprintf(fp, " %c  %s\n", type, row_name[seqn]);
      }
      /* write COLUMNS section */
      fprintf(fp, "COLUMNS\n");
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL))
      {  int seqn, nl = 1, ret;
         char *name;
         glp_get_seqn(GLP_COL, &seqn);
         name = col_name[seqn];
         glp_get_kind(GLP_COL, &flag);
         if (flag && marker % 2 == 0)
         {  /* open new intorg/intend group */
            marker++;
            fprintf(fp, "    M%07d  'MARKER'                 'INTORG'\n"
               , marker);
         }
         else if (!flag && marker % 2 == 1)
         {  /* close the current intorg/intend group */
            marker++;
            fprintf(fp, "    M%07d  'MARKER'                 'INTEND'\n"
               , marker);
         }
         if (glp_first_coef(GLP_COL) != 0 && !skip_empty)
            fprintf(fp, "    %-8s  %-8s  %12s   $ empty column\n",
               name, row_name[1], mps_number(0.0));
         for (ret = glp_first_coef(GLP_COL); ret == 0;
              ret = glp_next_coef(GLP_COL))
         {  int seqn;
            double coef;
            if (nl)
               fprintf(fp, "    %-8s  ", name);
            else
               fprintf(fp, "   ");
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_coef(&coef);
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(coef));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
            if (!pedantic) name = "";
         }
         if (!nl) fprintf(fp, "\n");
      }
      if (marker % 2 == 1)
      {  /* close the last intorg/intend group (if not closed) */
         marker++;
         fprintf(fp, "    M%07d  'MARKER'                 'INTEND'\n",
            marker);
      }
      /* determine seqn of the objective function row */
      {  char obj_row[GLP_MAX_NAME+1];
         glp_get_cpar("obj_row", obj_row);
         if (obj_row[0] == '\0')
            obj_seqn = 0;
         else
         {  glp_find_item(GLP_ROW, obj_row);
            glp_get_seqn(GLP_ROW, &obj_seqn);
         }
      }
      /* write RHS section */
      flag = 0;
      {  int nl = 1, ret;
         char *name = (pedantic ? "RHS1" : "");
         for (ret = glp_first_item(GLP_ROW); ret == 0;
              ret = glp_next_item(GLP_ROW))
         {  int seqn, type;
            double lb, ub, rhs;
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            switch (type)
            {  case GLP_FR:
                  /* if the current row is the objective function row,
                     right-hand side is set to the constant term of the
                     objective function with opposite sign; in other
                     cases right-hand side is not used */
                  if (seqn == obj_seqn)
                     glp_get_rpar("c0", &rhs), rhs = - rhs;
                  else
                     rhs = 0.0;
                  break;
               case GLP_LO:
                  rhs = lb;
                  break;
               case GLP_UP:
                  rhs = ub;
                  break;
               case GLP_DB:
                  rhs = (ub > 0.0 ? lb : ub);
                  break;
               case GLP_FX:
                  rhs = lb;
                  break;
               default:
                  insist(type != type);
            }
            if (rhs == 0.0) continue;
            if (!flag) fprintf(fp, "RHS\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(rhs));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write RANGES section */
      flag = 0;
      {  int nl = 1, ret;
         char *name = (pedantic ? "RNG1" : "");
         for (ret = glp_first_item(GLP_ROW); ret == 0;
              ret = glp_next_item(GLP_ROW))
         {  int seqn, type;
            double lb, ub, rng;
            glp_get_seqn(GLP_ROW, &seqn);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            if (type != GLP_DB) continue;
            if (!flag) fprintf(fp, "RANGES\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            rng = (ub > 0.0 ? ub - lb : lb - ub);
            fprintf(fp, "%-8s  %12s", row_name[seqn], mps_number(rng));
            if (!one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write BOUNDS section */
      flag = 0;
      {  int ret;
         char *name = (pedantic ? "BND1" : "");
         for (ret = glp_first_item(GLP_COL); ret == 0;
              ret = glp_next_item(GLP_COL))
         {  int seqn, type;
            double lb, ub;
            glp_get_seqn(GLP_COL, &seqn);
            glp_get_bounds(GLP_COL, &type, &lb, &ub);
            if (type == GLP_LO && lb == 0.0) continue;
            if (!flag) fprintf(fp, "BOUNDS\n"), flag = 1;
            switch (type)
            {  case GLP_FR:
                  fprintf(fp, " FR %-8s  %-8s\n", name, col_name[seqn]);
                  break;
               case GLP_LO:
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  break;
               case GLP_UP:
                  fprintf(fp, " MI %-8s  %-8s\n", name, col_name[seqn]);
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(ub));
                  break;
               case GLP_DB:
                  if (lb != 0.0)
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(ub));
                  break;
               case GLP_FX:
                  fprintf(fp, " FX %-8s  %-8s  %12s\n", name,
                     col_name[seqn], mps_number(lb));
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* write ENDATA indicator card */
      fprintf(fp, "ENDATA\n");
      /* free working arrays */
      ufree(row_name), row_name = NULL;
      ufree(col_name), col_name = NULL;
      /* close the output text file */
      fflush(fp);
      if (ferror(fp))
      {  error("glp_write_mps: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      /* returns to the calling program */
      return 0;
fail: /* the operation failed */
      if (row_name != NULL) ufree(row_name);
      if (col_name != NULL) ufree(col_name);
      if (fp != NULL) fclose(fp);
      return 1;
}

/*----------------------------------------------------------------------
-- make_names - generate standard MPS names using original names.
--
-- This routine tries to make names of rows (if what = GLP_ROW) or
-- columns (if what = GLP_COL), whose length doesn't exceed 8 chars,
-- using original row and column names as templates. The result names
-- are placed in alias[1], ..., alias[n], where n is the total number
-- of rows/columns. */

static void make_names(int what, mps_name alias[])
{     AVLTREE *tree;
      AVLNODE *node;
      int k = 0, ret;
      tree = create_avl((int (*)(void *, void *))strcmp);
      for (ret = glp_first_item(what); ret == 0;
           ret = glp_next_item(what))
      {  char name[GLP_MAX_NAME+1];
         int len;
         k++;
         glp_get_name(what, name);
         if (name[0] == '$') goto alas;
         len = strlen(name);
         if (len <= 8)
         {  strcpy(alias[k], name);
            if (find_by_key(tree, alias[k]) == NULL) goto fini;
            goto alas;
         }
         /* the first try: abc~wxyz */
         memcpy(alias[k]+0, name+0, 3);
         memcpy(alias[k]+3, "~", 1);
         memcpy(alias[k]+4, name+(len-4), 4);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the second try: abcd~xyz */
         memcpy(alias[k]+0, name+0, 4);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+5, name+(len-3), 3);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the third try: abcde~yz */
         memcpy(alias[k]+0, name+0, 5);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+6, name+(len-2), 2);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
         /* the fourth try: abcdef~z */
         memcpy(alias[k]+0, name+0, 6);
         memcpy(alias[k]+4, "~", 1);
         memcpy(alias[k]+7, name+(len-1), 1);
         if (find_by_key(tree, alias[k]) == NULL) goto fini;
alas:    /* nothing came of it */
         {  char *t;
            sprintf(alias[k], "%c%7d", what == GLP_ROW ? 'R' : 'C', k);
            for (t = alias[k]; *t; t++) if (*t == ' ') *t = '~';
            insist(find_by_key(tree, alias[k]) == NULL);
         }
fini:    /* enter the generated name to the symbol table */
         insist(strlen(alias[k]) <= 8);
         node = insert_by_key(tree, alias[k]);
         node->type = k;
      }
      delete_avl(tree);
      return;
}

/*----------------------------------------------------------------------
-- mps_number - convert number to standard 12-character MPS format.
--
-- This routine converts the given floating point value val to the
-- standard 12-character MPS format. It tries to provide maximal number
-- of significan digits. */

static char *mps_number(double val)
{     static char numb[255+1]; int n; char *e;
      for (n = 12; n >= 6; n--)
      {  if (val != 0.0 && fabs(val) < 1.0)
            sprintf(numb, "%.*E", n, val);
         else
            sprintf(numb, "%.*G", n, val);
         insist(strlen(numb) <= 255);
         e = strrchr(numb, 'E');
         if (e != NULL) sprintf(e+1, "%d", atoi(e+1));
         if (strlen(numb) <= 12) return numb;
      }
      fault("glp_write_mps: can't convert floating point number '%g' to"
         " character string", val);
      return NULL; /* to relax compiler */
}

/* eof */
