/* glpapi/glp_new_aij.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_new_aij - create new constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_new_aij(LPI *lp, int i, int j, double aij);
--
-- *Description*
--
-- The routine glp_new_aij creates new constraint coefficient that is
-- placed in the i-th row and the j-th column of the constraint matrix
-- and has the value aij.
--
-- Note that multiplets, i.e. elements with identical row and column
-- indices, are not allowed. However, the routine check for multiplets
-- only if the data consistency checking flag is on.
--
-- *Complexity*
--
-- If the data consistency checking flag is off, the time complexity is
-- O(1). Otherwise, the time complexity is O(nz'), where nz' is number
-- of existing non-zero elements in the j-th column. */

void glp_new_aij(LPI *lp, int i, int j, double aij)
{     ELEM *e;
      if (!(1 <= i && i <= lp->m))
         fault("glp_new_elem: i = %d; invalid row number", i);
      if (!(1 <= j && j <= lp->n))
         fault("glp_new_elem: j = %d; invalid column number", j);
      if (lp->check)
      {  for (e = lp->col[j]->ptr; e != NULL; e = e->col)
         {  if (e->i == i)
               fault("glp_new_elem: i = %d, j = %d; duplicate element",
                  i, j);
         }
      }
      e = get_atom(lp->elem_pool);
      e->i = i;
      e->j = j;
      e->val = aij;
      e->row = lp->row[i]->ptr, lp->row[i]->ptr = e;
      e->col = lp->col[j]->ptr, lp->col[j]->ptr = e;
      lp->basis = '?', lp->status = GLP_UNDEF;
      return;
}

/* eof */
