/* glplpx2.c (problem querying routines) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include "glplpx.h"

/*----------------------------------------------------------------------
-- lpx_get_num_rows - determine number of rows.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_num_rows(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_num_rows returns current number of rows in an LP
-- problem object, which the parameter lp points to. */

int lpx_get_num_rows(LPX *lp)
{     int m = lp->m;
      return m;
}

/*----------------------------------------------------------------------
-- lpx_get_num_cols - determine number of columns.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_num_cols(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_num_cols returns current number of columns in an
-- LP problem object, which the parameter lp points to. */

int lpx_get_num_cols(LPX *lp)
{     int n = lp->n;
      return n;
}

/*----------------------------------------------------------------------
-- lpx_get_num_nz - determine number of constraint coefficients.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_num_nz(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_num_nz returns current number non-zero elements
-- in the constraint matrix that belongs to an LP problem object, which
-- the parameter lp points to. */

int lpx_get_num_nz(LPX *lp)
{     int m = lp->m;
      int *aa_len = lp->A->len;
      int i, count = 0;
      for (i = 1; i <= m; i++) count += aa_len[i];
      return count;
}

/*----------------------------------------------------------------------
-- lpx_get_prob_name - obtain problem name.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- char *lpx_get_prob_name(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_prob_name returns a pointer to a static buffer
-- that contains symbolic name of the problem. However, if the problem
-- has no assigned name, the routine returns NULL. */

char *lpx_get_prob_name(LPX *lp)
{     return
         lp->name[0] == NULL ? NULL : get_str(lp->buf, lp->name[0]);
}

/*----------------------------------------------------------------------
-- lpx_get_row_name - obtain row name.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- char *lpx_get_row_name(LPX *lp, int i);
--
-- *Returns*
--
-- The routine lpx_get_row_name returns a pointer to a static buffer
-- that contains symbolic name of the i-th row. However, if the i-th
-- row has no assigned name, the routine returns NULL. */

char *lpx_get_row_name(LPX *lp, int i)
{     if (!(1 <= i && i <= lp->m))
         fault("lpx_get_row_name: i = %d; row number out of range", i);
      return
         lp->name[i] == NULL ? NULL : get_str(lp->buf, lp->name[i]);
}

/*----------------------------------------------------------------------
-- lpx_get_col_name - obtain column name.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- char *lpx_get_col_name(LPX *lp, int j);
--
-- *Returns*
--
-- The routine lpx_get_col_name returns a pointer to a static buffer
-- that contains symbolic name of the j-th column. However, if the j-th
-- column has no assigned name, the routine returns NULL. */

char *lpx_get_col_name(LPX *lp, int j)
{     if (!(1 <= j && j <= lp->n))
         fault("lpx_get_col_name: j = %d; column number out of range",
            j);
      j += lp->m;
      return
         lp->name[j] == NULL ? NULL : get_str(lp->buf, lp->name[j]);
}

/*----------------------------------------------------------------------
-- glp_get_row_bnds - obtain row bounds.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- void lpx_get_row_bnds(LPX *lp, int i, int *typx, double *lb,
--    double *ub);
--
-- *Description*
--
-- The routine lpx_get_row_bnds stores the type, lower bound and upper
-- bound of the i-th row to locations, which the parameters typx, lb,
-- and ub point to, respectively.
--
-- If some of the parameters typx, lb, or ub is NULL, the corresponding
-- value is not stored.
--
-- Types and bounds have the following meaning:
--
--     Type          Bounds            Note
--    -------------------------------------------
--    LPX_FR   -inf <  x <  +inf   free variable
--    LPX_LO     lb <= x <  +inf   lower bound
--    LPX_UP   -inf <  x <=  ub    upper bound
--    LPX_DB     lb <= x <=  ub    double bound
--    LPX_FX           x  =  lb    fixed variable
--
-- where x is the corresponding auxiliary variable.
--
-- If the row has no lower bound, *lb is set to zero. If the row has no
-- upper bound, *ub is set to zero. If the row is of fixed type, *lb and
-- *ub are set to the same value. */

void lpx_get_row_bnds(LPX *lp, int i, int *typx, double *lb, double *ub)
{     if (!(1 <= i && i <= lp->m))
         fault("lpx_get_row_bnds: i = %d; row number out of range", i);
      if (typx != NULL) *typx = lp->typx[i];
      if (lb != NULL) *lb = lp->lb[i] / lp->rs[i];
      if (ub != NULL) *ub = lp->ub[i] / lp->rs[i];
      return;
}

/*----------------------------------------------------------------------
-- glp_get_col_bnds - obtain column bounds.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- void lpx_get_col_bnds(LPX *lp, int j, int *typx, double *lb,
--    double *ub);
--
-- *Description*
--
-- The routine lpx_get_col_bnds stores the type, lower bound and upper
-- bound of the j-th column to locations, which the parameters typx, lb,
-- and ub point to, respectively.
--
-- If some of the parameters typx, lb, or ub is NULL, the corresponding
-- value is not stored.
--
-- Types and bounds have the following meaning:
--
--     Type          Bounds            Note
--    -------------------------------------------
--    LPX_FR   -inf <  x <  +inf   free variable
--    LPX_LO     lb <= x <  +inf   lower bound
--    LPX_UP   -inf <  x <=  ub    upper bound
--    LPX_DB     lb <= x <=  ub    double bound
--    LPX_FX           x  =  lb    fixed variable
--
-- where x is the corresponding structural variable.
--
-- If the column has no lower bound, *lb is set to zero. If the column
-- has no upper bound, *ub is set to zero. If the column is of fixed
-- type, *lb and *ub are set to the same value. */

void lpx_get_col_bnds(LPX *lp, int j, int *typx, double *lb, double *ub)
{     if (!(1 <= j && j <= lp->n))
         fault("lpx_get_col_bnds: j = %d; column number out of range",
            j);
      j += lp->m;
      if (typx != NULL) *typx = lp->typx[j];
      if (lb != NULL) *lb = lp->lb[j] * lp->rs[j];
      if (ub != NULL) *ub = lp->ub[j] * lp->rs[j];
      return;
}

/*----------------------------------------------------------------------
-- lpx_get_col_kind - query column kind.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_col_kind(LPX *lp, int j);
--
-- *Returns*
--
-- The routine lpx_get_col_kind returns the kind of j-th structural
-- variable:
--
-- LPX_CV - continuous variable;
-- LPX_IV - integer variable. */

int lpx_get_col_kind(LPX *lp, int j)
{     if (!(1 <= j && j <= lp->n))
         fault("lpx_get_col_kind: j = %d; column number out of range",
            j);
      return (lp->mip ? lp->kind[j] : LPX_CV);
}

/*----------------------------------------------------------------------
-- lpx_get_obj_dir - determine optimization direction.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_obj_dir(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_obj_dir returns a flag that defines optimization
-- direction (i.e. sense of the objective function):
--
-- LPX_MIN - the objective function has to be minimized;
-- LPX_MAX - the objective function has to be maximized. */

int lpx_get_obj_dir(LPX *lp)
{     int dir = lp->dir;
      return dir;
}

/*----------------------------------------------------------------------
-- lpx_get_obj_c0 - obtain constant term of the obj. function.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- double lpx_get_obj_c0(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_obj_c0 return a constant term of the objective
-- function for an LP problem, which the parameter lp points to. */

double lpx_get_obj_c0(LPX *lp)
{     double c0 = lp->coef[0];
      return c0;
}

/*----------------------------------------------------------------------
-- lpx_get_row_coef - obtain row objective coefficient.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- double lpx_get_row_coef(LPX *lp, int i);
--
-- *Returns*
--
-- The routine lpx_get_row_coef returns a coefficient of the objective
-- function at the i-th auxiliary variable (row). */

double lpx_get_row_coef(LPX *lp, int i)
{     if (!(1 <= i && i <= lp->m))
         fault("lpx_get_row_coef: i = %d; row number out of range", i);
      return lp->coef[i] * lp->rs[i];
}

/*----------------------------------------------------------------------
-- lpx_get_col_coef - obtain column objective coefficient.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- double lpx_get_col_coef(LPX *lp, int j);
--
-- *Returns*
--
-- The routine lpx_get_col_coef returns a coefficient of the objective
-- function at the j-th structural variable (column). */

double lpx_get_col_coef(LPX *lp, int j)
{     if (!(1 <= j && j <= lp->n))
         fault("lpx_get_col_coef: j = %d; column number out of range",
            j);
      j += lp->m;
      return lp->coef[j] / lp->rs[j];
}

/*----------------------------------------------------------------------
-- lpx_get_mat_row - get row of the constraint matrix.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_mat_row(LPX *lp, int i, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_get_mat_row scans (non-zero) elements of the i-th
-- row of the constraint matrix and stores their column indices and
-- values to locations ndx[1], ..., ndx[len] and val[1], ..., val[len]
-- respectively, where 0 <= len <= n is number of elements in the i-th
-- row, n is number of columns. It is allowed to specify val as NULL,
-- in which case only column indices are stored.
--
-- *Returns*
--
-- The routine returns len, which is number of stored elements (length
-- of the i-th row). */

int lpx_get_mat_row(LPX *lp, int i, int ndx[], double val[])
{     int m = lp->m;
      double *rs = lp->rs;
      int *aa_ptr = lp->A->ptr;
      int *aa_len = lp->A->len;
      int *sv_ndx = lp->A->ndx;
      double *sv_val = lp->A->val;
      int beg, len, t;
      double rs_i;
      if (!(1 <= i && i <= m))
         fault("lpx_get_mat_row: i = %d; row number out of range", i);
      beg = aa_ptr[i];
      len = aa_len[i];
      memcpy(&ndx[1], &sv_ndx[beg], len * sizeof(int));
      if (val != NULL)
      {  memcpy(&val[1], &sv_val[beg], len * sizeof(double));
         rs_i = rs[i];
         for (t = 1; t <= len; t++)
            val[t] /= (rs_i * rs[m + ndx[t]]);
      }
      return len;
}

/*----------------------------------------------------------------------
-- lpx_get_mat_col - get column of the constraint matrix.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_mat_col(LPX *lp, int j, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_get_mat_col scans (non-zero) elements of the j-th
-- column of the constraint matrix and stores their row indices and
-- values to locations ndx[1], ..., ndx[len] and val[1], ..., val[len]
-- respectively, where 0 <= len <= m is number of elements in the j-th
-- column, m is number of rows. It is allowed to specify val as NULL,
-- in which case only row indices are stored.
--
-- *Returns*
--
-- The routine returns len, which is number of stored elements (length
-- of the j-th column). */

int lpx_get_mat_col(LPX *lp, int j, int ndx[], double val[])
{     int m = lp->m;
      int n = lp->n;
      double *rs = lp->rs;
      int *aa_ptr = lp->A->ptr;
      int *aa_len = lp->A->len;
      int *sv_ndx = lp->A->ndx;
      double *sv_val = lp->A->val;
      int beg, len, t;
      double rs_j;
      if (!(1 <= j && j <= n))
         fault("lpx_get_mat_col: j = %d; column number out of range",
            j);
      j += m;
      beg = aa_ptr[j];
      len = aa_len[j];
      memcpy(&ndx[1], &sv_ndx[beg], len * sizeof(int));
      if (val != NULL)
      {  memcpy(&val[1], &sv_val[beg], len * sizeof(double));
         rs_j = rs[j];
         for (t = 1; t <= len; t++)
            val[t] /= (rs[ndx[t]] * rs_j);
      }
      return len;
}

/*----------------------------------------------------------------------
-- lpx_get_status - query basic solution status.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_status(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_status reports the status of the current basic
-- solution obtained for an LP problem object, which the parameter lp
-- points to:
--
-- LPX_OPT      - solution is optimal;
-- LPX_FEAS     - solution is feasible;
-- LPX_INFEAS   - solution is infeasible;
-- LPX_NOFEAS   - problem has no feasible solution;
-- LPX_UNBND    - problem has unbounded solution;
-- LPX_UNDEF    - solution is undefined. */

int lpx_get_status(LPX *lp)
{     int p_stat = lp->p_stat;
      int d_stat = lp->d_stat;
      int status;
      switch (p_stat)
      {  case LPX_P_UNDEF:
            status = LPX_UNDEF;
            break;
         case LPX_P_FEAS:
            switch (d_stat)
            {  case LPX_D_UNDEF:
                  status = LPX_FEAS;
                  break;
               case LPX_D_FEAS:
                  status = LPX_OPT;
                  break;
               case LPX_D_INFEAS:
                  status = LPX_FEAS;
                  break;
               case LPX_D_NOFEAS:
                  status = LPX_UNBND;
                  break;
               default:
                  insist(d_stat != d_stat);
            }
            break;
         case LPX_P_INFEAS:
            status = LPX_INFEAS;
            break;
         case LPX_P_NOFEAS:
            status = LPX_NOFEAS;
            break;
         default:
            insist(p_stat != p_stat);
      }
      return status;
}

/*----------------------------------------------------------------------
-- lpx_get_prim_stat - query primal status of basic solution.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_prim_stat(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_prim_stat reports the primal status of the basic
-- solution obtained by the solver for an LP problem object, which the
-- parameter lp points to:
--
-- LPX_P_UNDEF  - the primal status is undefined;
-- LPX_P_FEAS   - the solution is primal feasible;
-- LPX_P_INFEAS - the solution is primal infeasible;
-- LPX_P_NOFEAS - no primal feasible solution exists. */

int lpx_get_prim_stat(LPX *lp)
{     int p_stat = lp->p_stat;
      return p_stat;
}

/*----------------------------------------------------------------------
-- lpx_get_dual_stat - query dual status of basic solution.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_get_dual_stat(LPX *lp);
--
-- *Returns*
--
-- The routine lpx_get_dual_stat reports the dual status of the basic
-- solution obtained by the solver for an LP problem object, which the
-- parameter lp points to:
--
-- LPX_D_UNDEF  - the dual status is undefined;
-- LPX_D_FEAS   - the solution is dual feasible;
-- LPX_D_INFEAS - the solution is dual infeasible;
-- LPX_D_NOFEAS - no dual feasible solution exists. */

int lpx_get_dual_stat(LPX *lp)
{     int d_stat = lp->d_stat;
      return d_stat;
}

/*----------------------------------------------------------------------
-- lpx_get_row_info - obtain row solution information.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- void lpx_get_row_info(LPX *lp, int i, int *tagx, double *valx,
--    double *dx);
--
-- *Description*
--
-- The routine lpx_get_row_info stores status, primal value and dual
-- value of the i-th auxiliary variable (row) to locations, which the
-- parameters tagx, valx, and dx point to, respectively.
--
-- The status code has the following meaning:
--
-- LPX_BS - basic variable;
-- LPX_NL - non-basic variable on its lower bound;
-- LPX_NU - non-basic variable on its upper bound;
-- LPX_NF - non-basic free (unbounded) variable;
-- LPX_NS - non-basic fixed variable.
--
-- If some of pointers tagx, valx, or dx is NULL, the corresponding
-- value is not stored. */

void lpx_get_row_info(LPX *lp, int i, int *tagx, double *valx,
      double *dx)
{     int m = lp->m;
      int n = lp->n;
      int typx_i, tagx_i, t;
      double lb_i, ub_i, valx_i, dx_i, temp;
      if (!(1 <= i && i <= m))
         fault("lpx_get_row_info: i = %d; row number out of range", i);
      /* obtain the status */
      tagx_i = lp->tagx[i];
      if (tagx != NULL) *tagx = tagx_i;
      /* obtain the primal value */
      if (valx != NULL)
      {  if (lp->p_stat == LPX_P_UNDEF)
         {  /* the primal value is undefined */
            valx_i = 0.0;
         }
         else
         {  if (tagx_i == LPX_BS)
            {  /* basic variable */
               t = lp->posx[i]; /* x[i] = xB[t] */
               insist(1 <= t && t <= m);
               valx_i = lp->bbar[t];
               /* round the primal value (if required) */
               if (lp->round)
               {  typx_i = lp->typx[i];
                  if (typx_i == LPX_LO || typx_i == LPX_DB ||
                      typx_i == LPX_FX)
                  {  /* x[i] has lower bound */
                     lb_i = lp->lb[i];
                     temp = fabs(valx_i - lb_i) / (1.0 + fabs(lb_i));
                     if (temp < lp->tol_bnd) valx_i = lb_i;
                  }
                  if (typx_i == LPX_UP || typx_i == LPX_DB ||
                      typx_i == LPX_FX)
                  {  /* x[i] has upper bound */
                     ub_i = lp->ub[i];
                     temp = fabs(valx_i - ub_i) / (1.0 + fabs(ub_i));
                     if (temp < lp->tol_bnd) valx_i = ub_i;
                  }
                  if (fabs(valx_i) < lp->tol_bnd) valx_i = 0.0;
               }
            }
            else
            {  /* non-basic variable */
               switch (tagx_i)
               {  case LPX_NL:
                     valx_i = lp->lb[i]; break;
                  case LPX_NU:
                     valx_i = lp->ub[i]; break;
                  case LPX_NF:
                     valx_i = 0.0; break;
                  case LPX_NS:
                     valx_i = lp->lb[i]; break;
                  default:
                     insist(tagx_i != tagx_i);
               }
            }
            /* unscale the primal value */
            valx_i /= lp->rs[i];
         }
         *valx = valx_i;
      }
      /* obtain the dual value */
      if (dx != NULL)
      {  if (lp->d_stat == LPX_D_UNDEF)
         {  /* the dual value is undefined */
            dx_i = 0.0;
         }
         else
         {  if (tagx_i == LPX_BS)
            {  /* basic variable */
               dx_i = 0.0;
            }
            else
            {  /* non-basic variable */
               t = lp->posx[i] - m; /* x[i] = xN[t] */
               insist(1 <= t && t <= n);
               dx_i = lp->cbar[t];
               /* round the dual value (if required) */
               if (lp->round && fabs(dx_i) < lp->tol_dj) dx_i = 0.0;
            }
            /* unscale the dual value */
            dx_i *= lp->rs[i];
         }
         *dx = dx_i;
      }
      return;
}

/*----------------------------------------------------------------------
-- lpx_get_col_info - obtain column solution information.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- void lpx_get_col_info(LPX *lp, int j, int *tagx, double *valx,
--    double *dx);
--
-- *Description*
--
-- The routine lpx_get_col_info stores status, primal value and dual
-- value of the j-th structural variable (column) to locations, which
-- the parameters tagx, valx, and dx point to, respectively.
--
-- The status code has the following meaning:
--
-- LPX_BS - basic variable;
-- LPX_NL - non-basic variable on its lower bound;
-- LPX_NU - non-basic variable on its upper bound;
-- LPX_NF - non-basic free (unbounded) variable;
-- LPX_NS - non-basic fixed variable.
--
-- If some of pointers tagx, valx, or dx is NULL, the corresponding
-- value is not stored. */

void lpx_get_col_info(LPX *lp, int j, int *tagx, double *valx,
      double *dx)
{     int m = lp->m;
      int n = lp->n;
      int typx_j, tagx_j, t;
      double lb_j, ub_j, valx_j, dx_j, temp;
      if (!(1 <= j && j <= n))
         fault("lpx_get_col_info: j = %d; column number out of range",
            j);
      j += m;
      /* obtain the status */
      tagx_j = lp->tagx[j];
      if (tagx != NULL) *tagx = tagx_j;
      /* obtain the primal value */
      if (valx != NULL)
      {  if (lp->p_stat == LPX_P_UNDEF)
         {  /* the primal value is undefined */
            valx_j = 0.0;
         }
         else
         {  if (tagx_j == LPX_BS)
            {  /* basic variable */
               t = lp->posx[j]; /* x[j] = xB[t] */
               insist(1 <= t && t <= m);
               valx_j = lp->bbar[t];
               /* round the primal value (if required) */
               if (lp->round)
               {  typx_j = lp->typx[j];
                  if (typx_j == LPX_LO || typx_j == LPX_DB ||
                      typx_j == LPX_FX)
                  {  /* x[j] has lower bound */
                     lb_j = lp->lb[j];
                     temp = fabs(valx_j - lb_j) / (1.0 + fabs(lb_j));
                     if (temp < lp->tol_bnd) valx_j = lb_j;
                  }
                  if (typx_j == LPX_UP || typx_j == LPX_DB ||
                      typx_j == LPX_FX)
                  {  /* x[j] has upper bound */
                     ub_j = lp->ub[j];
                     temp = fabs(valx_j - ub_j) / (1.0 + fabs(ub_j));
                     if (temp < lp->tol_bnd) valx_j = ub_j;
                  }
                  if (fabs(valx_j) < lp->tol_bnd) valx_j = 0.0;
               }
            }
            else
            {  /* non-basic variable */
               switch (tagx_j)
               {  case LPX_NL:
                     valx_j = lp->lb[j]; break;
                  case LPX_NU:
                     valx_j = lp->ub[j]; break;
                  case LPX_NF:
                     valx_j = 0.0; break;
                  case LPX_NS:
                     valx_j = lp->lb[j]; break;
                  default:
                     insist(tagx_j != tagx_j);
               }
            }
            /* unscale the primal value */
            valx_j *= lp->rs[j];
         }
         *valx = valx_j;
      }
      /* obtain the dual value */
      if (dx != NULL)
      {  if (lp->d_stat == LPX_D_UNDEF)
         {  /* the dual value is undefined */
            dx_j = 0.0;
         }
         else
         {  if (tagx_j == LPX_BS)
            {  /* basic variable */
               dx_j = 0.0;
            }
            else
            {  /* non-basic variable */
               t = lp->posx[j] - m; /* x[i] = xN[t] */
               insist(1 <= t && t <= n);
               dx_j = lp->cbar[t];
               /* round the dual value (if required) */
               if (lp->round && fabs(dx_j) < lp->tol_dj) dx_j = 0.0;
            }
            /* unscale the dual value */
            dx_j /= lp->rs[j];
         }
         *dx = dx_j;
      }
      return;
}

/*----------------------------------------------------------------------
-- lpx_get_obj_val - obtain value of the objective function.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- double lpx_get_obj_val(LPX *lp);
--
-- *Returns*
--
-- The routine returns the current value of the objective function for
-- an LP problem object, which the parameter lp points to. */

double lpx_get_obj_val(LPX *lp)
{     int m = lp->m;
      int n = lp->n;
      int i, j;
      double sum, coef, valx;
      sum = lpx_get_obj_c0(lp);
      for (i = 1; i <= m; i++)
      {  coef = lpx_get_row_coef(lp, i);
         if (coef != 0.0)
         {  lpx_get_row_info(lp, i, NULL, &valx, NULL);
            sum += coef * valx;
         }
      }
      for (j = 1; j <= n; j++)
      {  coef = lpx_get_col_coef(lp, j);
         if (coef != 0.0)
         {  lpx_get_col_info(lp, j, NULL, &valx, NULL);
            sum += coef * valx;
         }
      }
      return sum;
}

/*----------------------------------------------------------------------
-- lpx_print_sol - write problem solution in printable format.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_print_sol(LPX *lp, char *fname);
--
-- *Description*
--
-- The routine lpx_print_sol writes the current basic solution of an LP
-- problem, which is specified by the pointer lp, to a text file, whose
-- name is the character string fname, in printable format.
--
-- Information reported by the routine lpx_print_sol is intended mainly
-- for visual analysis.
--
-- *Returns*
--
-- If the operation was successful, the routine returns zero. Otherwise
-- the routine prints an error message and returns non-zero. */

int lpx_print_sol(LPX *lp, char *fname)
{     FILE *fp;
      int what, round;
      print("lpx_print_sol: writing problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  print("lpx_print_sol: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char *name;
         name = lpx_get_prob_name(lp);
         if (name == NULL) name = "";
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         nr = lpx_get_num_rows(lp);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc;
         nc = lpx_get_num_cols(lp);
         fprintf(fp, "%-12s%d\n", "Columns:", nc);
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         nz = lpx_get_num_nz(lp);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* solution status */
      {  int status;
         status = lpx_get_status(lp);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == LPX_OPT    ? "OPTIMAL" :
            status == LPX_FEAS   ? "FEASIBLE" :
            status == LPX_INFEAS ? "INFEASIBLE (INTERMEDIATE)" :
            status == LPX_NOFEAS ? "INFEASIBLE (FINAL)" :
            status == LPX_UNBND  ? "UNBOUNDED" :
            status == LPX_UNDEF  ? "UNDEFINED" : "???");
      }
      /* objective function */
      {  int dir;
         double obj;
         dir = lpx_get_obj_dir(lp);
         round = lp->round, lp->round = 1;
         obj = lpx_get_obj_val(lp);
         lp->round = round;
         fprintf(fp, "%-12s%.6g %s\n", "Objective:", obj,
            dir == LPX_MIN ? "(MINimization)" :
            dir == LPX_MAX ? "(MAXimization)" : "(???)");
      }
      /* main sheet */
      for (what = 1; what <= 2; what++)
      {  int mn, ij;
         fprintf(fp, "\n");
         fprintf(fp, "  No. %-12s   St   Activity     Lower bound   Upp"
            "er bound    Marginal\n",
            what == 1 ? "  Row name" : "Column name");
         fprintf(fp, "----- ------------   -- ------------- -----------"
            "-- ------------- -------------\n");
         mn = (what == 1 ? lpx_get_num_rows(lp) : lpx_get_num_cols(lp));
         for (ij = 1; ij <= mn; ij++)
         {  char *name;
            int typx, tagx;
            double lb, ub, valx, dx;
            if (what == 1)
            {  name = lpx_get_row_name(lp, ij);
               if (name == NULL) name = "";
               lpx_get_row_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               lpx_get_row_info(lp, ij, &tagx, &valx, &dx);
               lp->round = round;
            }
            else
            {  name = lpx_get_col_name(lp, ij);
               if (name == NULL) name = "";
               lpx_get_col_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               lpx_get_col_info(lp, ij, &tagx, &valx, &dx);
               lp->round = round;
            }
            /* row/column ordinal number */
            fprintf(fp, "%5d ", ij);
            /* row column/name */
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%19s", name, "");
            /* row/column status */
            fprintf(fp, "  %s ",
               tagx == LPX_BS ? "B " :
               tagx == LPX_NL ? "NL" :
               tagx == LPX_NU ? "NU" :
               tagx == LPX_NF ? "NF" :
               tagx == LPX_NS ? "NS" : "??");
            /* row/column primal activity */
            fprintf(fp, "%13.6g ", valx);
            /* row/column lower bound */
            if (typx == LPX_LO || typx == LPX_DB || typx == LPX_FX)
               fprintf(fp, "%13.6g ", lb);
            else
               fprintf(fp, "%13s ", "");
            /* row/column upper bound */
            if (typx == LPX_UP || typx == LPX_DB)
               fprintf(fp, "%13.6g ", ub);
            else if (typx == LPX_FX)
               fprintf(fp, "%13s ", "=");
            else
               fprintf(fp, "%13s ", "");
            /* row/column dual activity */
            if (tagx != LPX_BS)
            {  if (dx == 0.0)
                  fprintf(fp, "%13s", "< eps");
               else
                  fprintf(fp, "%13.6g", dx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  print("lpx_print_sol: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
