/* glpavl.h */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#ifndef _GLPAVL_H
#define _GLPAVL_H

#include "glplib.h"

#define create_avl            _glp_create_avl
#define insert_by_key         _glp_insert_by_key
#define next_node             _glp_next_node
#define prev_node             _glp_prev_node
#define find_by_key           _glp_find_by_key
#define next_by_key           _glp_next_by_key
#define insert_by_pos         _glp_insert_by_pos
#define find_by_pos           _glp_find_by_pos
#define delete_node           _glp_delete_node
#define rotate_avl            _glp_rotate_avl
#define delete_avl            _glp_delete_avl

typedef struct AVLTREE AVLTREE;
typedef struct AVLNODE AVLNODE;

struct AVLTREE
{     /* Adelson-Velsky & Landis binary tree (AVL-tree) */
      POOL *pool;
      /* memory pool holding tree nodes */
      int (*comp)(void *, void *);
      /* user-supplied routine to compare node keys; if this pointer is
         NULL, ordering by keys is not used */
      int size;
      /* the total number of nodes (tree size) */
      AVLNODE *root;
      /* pointer to the root node */
      int height;
      /* the total height of the tree */
};

struct  AVLNODE
{     /* node of AVL-tree */
      void *key;
      /* pointer to node key (data type to represent keys is supplied
         by the user) */
      int rank;
      /* node rank = the relative position of the node in its own
         subtree = the number of nodes in the left subtree plus one */
      int type;
      /* node type (supplied by the user) */
      void *link;
      /* pointer to node extension (supplied by the user) */
      AVLNODE *up;
      /* pointer to the parent node */
      short int flag;
      /* node flag:
         0 - this node is the left node for its parent (or this node
             is the root of entire tree and has no parent)
         1 - this node is the right node for its parent */
      short int bal;
      /* balance = the difference between heights of the right and of
         the left subtrees:
         -1 - the left subtree is higher than the right one;
          0 - the left and the right subtrees have equal heights;
         +1 - the left subtree is lower than the right one */
      AVLNODE *left;
      /* pointer to the root of the left subtree */
      AVLNODE *right;
      /* pointer to the root of the right subtree */
};

extern AVLTREE *create_avl(int (*comp)(void *, void *));
/* create AVL-tree */

extern AVLNODE *insert_by_key(AVLTREE *tree, void *key);
/* insert new node with given key to AVL-tree */

extern AVLNODE *next_node(AVLTREE *tree, AVLNODE *node);
/* search for the next node in AVL-tree */

extern AVLNODE *prev_node(AVLTREE *tree, AVLNODE *node);
/* search for the previous node in AVL-tree */

extern AVLNODE *find_by_key(AVLTREE *tree, void *key);
/* search for the first node with given key in AVL-tree */

extern AVLNODE *next_by_key(AVLTREE *tree, AVLNODE *node);
/* search for the next node having same key in AVL-tree */

extern AVLNODE *insert_by_pos(AVLTREE *tree, int pos);
/* insert new node to given position of AVL-tree */

extern AVLNODE *find_by_pos(AVLTREE *tree, int pos);
/* search for node placed in given position of AVL-tree */

extern void delete_node(AVLTREE *tree, AVLNODE *node);
/* delete given node from AVL-tree */

extern AVLNODE *rotate_avl(AVLTREE *tree, AVLNODE *node);
/* restore correct balance of AVL-subtree */

extern void delete_avl(AVLTREE *tree);
/* delete AVL-tree */

#endif

/* eof */
