/* glprsm/check_gvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- check_gvec - check accuracy of the vector gamma.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double check_gvec(RSM *rsm, double gvec[]);
--
-- *Description*
--
-- The check_gvec routine is intended for checking accuracy of the
-- vector gamma. It computes the absolute error
--
--    e = max |gamma'[j] - gamma[j]|
--
-- where gamma' is exact vector computed by means of the exact_gvec
-- routine, gamma is approximate vector given in the array gvec.
--
-- This operation is extremely inefficient and may be used only for
-- debugging purposes.
--
-- *Returns*
--
-- The check_gvec routine returns the absolute error e (see above). */

double check_gvec(RSM *rsm, double gvec[])
{     int n = rsm->n, j;
      double d, dmax = 0.0;
      for (j = 1; j <= n; j++)
      {  d = fabs(exact_gvec(rsm, j) - gvec[j]);
         if (dmax < d) dmax = d;
      }
      return dmax;
}

/* eof */
