/* glpmat/gm_scaling.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- gm_scaling - implicit geometric mean scaling.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- void gm_scaling(MAT *A, double R[], double S[], int ord, double eps,
--    int itmax);
--
-- *Description*
--
-- The gm_scaling routine performs implicit geometric mean scaling of
-- the matrix R*A*S, where A is the given sparse matrix (that remains
-- unchanged on exit), R and S are the given diagonal scaling matrices.
-- The result of scaling is the matrix R'*A*S', where R' and S' are new
-- scaling matrices computed by the routine, which are placed in the
-- same array. Diagonal elements of the matrix R should be placed in
-- locations R[1], R[2], ..., R[m], where m is number of rows of the
-- matrix A. Diagonal elements of the matrix S should be placed in
-- locations S[1], S[2], ..., S[n], where n is number of columns of the
-- matrix A. Diagonal elements of the matrices R' and S' will be placed
-- in the same manner.
--
-- To perform geometric mean scaling the gm_scaling routine divides all
-- elements of each row (column) by sqrt(beta/alfa), where alfa and beta
-- are, respectively, smallest and largest absolute values of non-zero
-- elements of the corresponding row (column). In order to improve the
-- scaling quality the routine performs row and columns scaling several
-- times.
--
-- Before a call the matrices R and S should be defined (if the matrix
-- A is unscaled, R and S should be untity matrices). As a result of
-- scaling the routine computes new matrices R' and S', that define the
-- scaled matrix R'*A*S'. (Thus scaling is implicit, because the matrix
-- A remains unchanged.)
--
-- The parameter ord defines the order of scaling:
--
-- if ord = 0, at first rows, then columns;
-- if ord = 1, at first columns, then rows.
--
-- The parameter eps > 0 is a criterion, that is used to decide when
-- the scaling process has to stop. The scaling process stops if the
-- condition t[k-1] - t[k] < eps * t[k-1] becomes true, where
-- t[k] = beta[k] / alfa[k] is the "quality" of scaling, alfa[k] and
-- beta[k] are, respectively, smallest and largest abolute values of
-- elements of the current matrix, k is number of iteration. The value
-- eps = 0.01-0.1 may be recommended for most cases.
--
-- The parameter itmax defines maximal number of scaling iterations.
-- Recommended value is itmax = 10-50.
--
-- On each iteration the gm_scaling routine prints the "quality" of
-- scaling (see above). */

static double ratio(MAT *A, double R[], double S[]);
static void scale_rows(MAT *A, double R[], double S[]);
static void scale_cols(MAT *A, double R[], double S[]);

void gm_scaling(MAT *A, double R[], double S[], int ord, double eps,
      int itmax)
{     int iter;
      double told, tnew;
      print("gm_scaling: scaling...");
      told = DBL_MAX;
      for (iter = 1; iter <= itmax; iter++)
      {  tnew = ratio(A, R, S);
         print("gm_scaling: max / min = %9.3e", tnew);
         if (told - tnew < eps * told) break;
         told = tnew;
         if (ord == 0)
         {  scale_rows(A, R, S);
            scale_cols(A, R, S);
         }
         else
         {  scale_cols(A, R, S);
            scale_rows(A, R, S);
         }
      }
      return;
}

static double ratio(MAT *A, double R[], double S[])
{     /* this routine computes the "quality" of scaling, which is
         beta/alfa, where alfa and beta are, respectively, smallest and
         largest absolute values of elements of the current matrix */
      ELEM *e;
      int i;
      double alfa, beta, temp;
      alfa = DBL_MAX; beta = 0.0;
      for (i = 1; i <= A->m; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
         {  temp = fabs(R[e->i] * e->val * S[e->j]);
            if (temp == 0.0) continue;
            if (alfa > temp) alfa = temp;
            if (beta < temp) beta = temp;
         }
      }
      temp = (beta == 0.0 ? 1.0 : beta / alfa);
      return temp;
}

static void scale_rows(MAT *A, double R[], double S[])
{     /* this routine performs geometric mean scaling of rows */
      ELEM *e;
      int i;
      double alfa, beta, temp;
      for (i = 1; i <= A->m; i++)
      {  alfa = DBL_MAX; beta = 0.0;
         for (e = A->row[i]; e != NULL; e = e->row)
         {  temp = fabs(R[e->i] * e->val * S[e->j]);
            if (temp == 0.0) continue;
            if (alfa > temp) alfa = temp;
            if (beta < temp) beta = temp;
         }
         if (beta != 0.0) R[i] /= sqrt(alfa * beta);
      }
      return;
}

static void scale_cols(MAT *A, double R[], double S[])
{     /* this routine performs geometric mean scaling of columns */
      ELEM *e;
      int j;
      double alfa, beta, temp;
      for (j = 1; j <= A->n; j++)
      {  alfa = DBL_MAX; beta = 0.0;
         for (e = A->col[j]; e != NULL; e = e->col)
         {  temp = fabs(R[e->i] * e->val * S[e->j]);
            if (temp == 0.0) continue;
            if (alfa > temp) alfa = temp;
            if (beta < temp) beta = temp;
         }
         if (beta != 0.0) S[j] /= sqrt(alfa * beta);
      }
      return;
}

/* eof */
