/* glpimg.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPIMG_H
#define _GLPIMG_H

#define create_img            glp_create_img
#define reset_rgb             glp_reset_rgb
#define get_imgval            glp_get_imgval
#define set_imgval            glp_set_imgval
#define delete_img            glp_delete_img
#define save_as_bmp           glp_save_as_bmp
#define save_as_tiff          glp_save_as_tiff
#define save_img              glp_save_img

typedef struct IMG IMG;
typedef struct RGB RGB;

struct IMG
{     /* raster image */
      int type;
      /* image type: */
#define IMG_2     0 /* 2 colors (1 bit per pixel) */
#define IMG_4     1 /* 4 colors (2 bits per pixel) */
#define IMG_16    2 /* 16 colors (4 bits per pixel) */
#define IMG_256   3 /* 256 colors (8 bits per pixel) */
      RGB *rgb; /* RGB rgb[2, 4, 16, or 256]; */
      /* color map table */
      int xres, yres;
      /* horizontal and vertical resolution (in pixels per meter) */
      int xsize, ysize;
      /* width and height of the image (in pixels) */
      int len;
      /* length of each horizontal scan line (in bytes) */
      unsigned char *scan; /* unsigned char scan[ysize*len]; */
      /* array of scan lines */
      int color;
      /* current color index */
      int x0, y0;
      /* absolute coordinates of the base point (it may be placed out
         of the image or the clipping region) */
      int x1, y1, x2, y2;
      /* absolute coordinates of the rectangular clipping region */
      int x, y;
      /* absolute coordinates of the current point (it may be placed
         out of the image or the clipping region) */
};

struct RGB
{     /* color map entry */
      unsigned char red;
      /* red intensity (0-255) */
      unsigned char green;
      /* green intensity (0-255) */
      unsigned char blue;
      /* blue intensity (0-255) */
};

/* default color indices: */
#define IMG_BLACK          0x00
#define IMG_BLUE           0x01
#define IMG_GREEN          0x02
#define IMG_CYAN           0x03
#define IMG_RED            0x04
#define IMG_MAGENTA        0x05
#define IMG_BROWN          0x06
#define IMG_LIGHTGRAY      0x07
#define IMG_DARKGRAY       0x08
#define IMG_LIGHTBLUE      0x09
#define IMG_LIGHTGREEN     0x0A
#define IMG_LIGHTCYAN      0x0B
#define IMG_LIGHTRED       0x0C
#define IMG_LIGHTMAGENTA   0x0D
#define IMG_YELLOW         0x0E
#define IMG_WHITE          0x0F

extern IMG *create_img(int type, int xsize, int ysize);
/* create raster image */

extern void reset_rgb(IMG *img);
/* initialize color map table by default spectrum */

extern int get_imgval(IMG *img, int x, int y);
/* get current value of pixel */

extern void set_imgval(IMG *img, int x, int y, int val);
/* set pixel to given value */

extern void delete_img(IMG *img);
/* delete raster image */

extern int save_as_bmp(IMG *img, char *fname);
/* write raster image using Windows Bitmap format */

extern int save_as_tiff(IMG *img, char *fname);
/* write raster image using TIFF format */

extern int (*save_img)(IMG *img, char *fname);
/* write raster image using default format */

#endif

/* eof */
