/* glprsm/pivot_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- pivot_row - choose basic variable (primal, standard technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int pivot_row(RSM *rsm, int q, int dir, double aq[], double bbar[],
--    int *tagp, double tol);
--
-- *Description*
--
-- The pivot_row routine chooses basic variable (xB)p (i.e. pivot row of
-- the simplex table) which should leave the basis on the next iteration
-- of the primal simplex method. The routine is based on the standard
-- (textbook) ratio test.
--
-- The parameter q specifies a number of the non-basic variable (xN)q
-- which has been chosen to enter the basis, 1 <= q <= n.
--
-- The parameter dir specifies in what direction the non-basic variable
-- (xN)q is changing:
--
-- 0, if (xN)q is increasing;
--
-- 1, if (xN)q is decreasing.
--
-- The array aq should contain q-th (pivot) column of the simplex table
-- (i.e. of the matrix A~) in locations aq[1], ..., aq[m].
--
-- The array bbar should contain current values of basic variables xB in
-- locations bbar[1], ..., bbar[m].
--
-- If the routine chooses some basic variable (xB)p, it stores into
-- location tagp a tag that specifies to what subset xL, xU, or xS the
-- basic variable should be attributed after it has left the basis. This
-- tag has the same meaning as the field rsm.tagn (see 'glprsm.h').
--
-- The parameter tol is a relative tolerance (see below).
--
-- The pivot_row routine implements the standard (textbook) ratio test
-- for choosing basic variable, i.e. the routine determines that basic
-- variable which reaches its (lower or upper) bound first when the
-- non-basic variable (xN)q is changing in the feasible direction.
-- Besides, the following additional rules are used:
--
-- if |aq[i]| < tol * max|aq[*]|, i.e. if the influence coefficient
-- aq[i] is relatively close to zero, it is assumed that the
-- corresponding basic variable (xB)i doesn't depend on the non-basic
-- variable (xN)q and therefore such basic variable is not considered to
-- be chosen;
--
-- if current value bbar[i] of some basic variable (xB)i violates its
-- bound, it is assumed that this happens dur to round-off errors and
-- actually the basic variable is exactly on its bound (because the
-- current basis solution should be primal feasible);
--
-- if several basic variables reach their bounds first at the same time,
-- the routine prefers that variable which has the largest (in absolute
-- value) influence coefficient.
--
-- For further details see the program documentation.
--
-- *Returns*
--
-- If the choice has been made, the pivot_row routine returns p which
-- is a number of the chosen basic variable (xB)p, 1 <= p <= m. In the
-- special case, if the non-basic variable (xN)q being double-bounded
-- variable reaches its opposite bound before any basic variable, the
-- routine returns a negative value. Otherwise, if the non-basic
-- variable (xN)q can unlimitedly change in the feasible direction and
-- therefore the choice is impossible, the routine returns zero. */

int pivot_row(RSM *rsm, int q, int dir, double aq[], double bbar[],
      int *_tagp, double tol)
{     int m = rsm->m, i, k, tag, p, tagp;
      double big, eps, temp, teta;
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (i = 1; i <= m; i++)
         if (big < fabs(aq[i])) big = fabs(aq[i]);
      eps = tol * big;
      /* turn to the case of increasing xN[q] in order to simplify
         program logic */
      if (dir) for (i = 1; i <= m; i++) aq[i] = - aq[i];
      /* initial settings */
      k = rsm->indn[q]; /* x[k] = xN[q] */
      if (rsm->type[k] == 'D')
      {  p = -1;
         tagp = -1;
         teta = rsm->ub[k] - rsm->lb[k];
         big = 1.0;
      }
      else
      {  p = 0;
         tagp = -1;
         teta = DBL_MAX;
         big = 0.0;
      }
      /* look through the list of basic variables */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (aq[i] == 0.0 || fabs(aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'F')
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == 'L')
         {  /* xB[i] has lower bound */
            if (aq[i] > 0.0) continue;
lo:         temp = (rsm->lb[k] - bbar[i]) / aq[i];
            tag = 'L';
         }
         else if (rsm->type[k] == 'U')
         {  /* xB[i] has upper bound */
            if (aq[i] < 0.0) continue;
up:         temp = (rsm->ub[k] - bbar[i]) / aq[i];
            tag = 'U';
         }
         else if (rsm->type[k] == 'D')
         {  /* xB[i] has both lower and upper bounds */
            if (aq[i] < 0.0) goto lo; else goto up;
         }
         else if (rsm->type[k] == 'S')
         {  /* xB[i] is fixed variable */
            temp = 0.0;
            tag = 'S';
         }
         else
            insist(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its bound, temp is negative;
            in this case it is assumed that xB[i] is exactly on its
            bound, therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply minimal ratio test */
         if (teta > temp || teta == temp && big < fabs(aq[i]))
         {  p = i;
            tagp = tag;
            teta = temp;
            big = fabs(aq[i]);
         }
      }
      /* restore original signs of the coefficients aq[i] */
      if (dir) for (i = 1; i <= m; i++) aq[i] = - aq[i];
      /* store a tag for xB[p] */
      *_tagp = tagp;
      /* return to the calling program */
      return p;
}

/* eof */
