/* glprsm/harris_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- harris_row - choose basic variable (primal, Harris technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int harris_row(RSM *rsm, int q, int dir, double aq[], double bbar[],
--    int *tagp, double tol, double tol1);
--
-- *Description*
--
-- The harris_row routine chooses basic variable (xB)p (i.e. pivot row
-- of the simplex table) which should leave the basis on the next
-- iteration of the primal simplex method. The routine is based on the
-- two-pass ratio test proposed by P.Harris.
--
-- The harris_row routine has the same program specification as the
-- pivit_row routine.
--
-- The only difference is the additional parameter tol1 which is a
-- relative tolerance used for relaxing bounds of basic variables on the
-- first pass. The routine replaces original bounds of basic variables
-- by their relaxed bounds:
--
--    (lB)i - eps <= (xB)i <= (uB)i + eps,
--
-- where
--
--    eps1 = tol * max(1, |(lB)i|) in case of lower bound
--    eps1 = tol * max(1, |(uB)i|) in case of upper bound
--
-- For futher details see the program documentation. */

int harris_row(RSM *rsm, int q, int dir, double aq[], double bbar[],
      int *_tagp, double tol, double tol1)
{     int m = rsm->m, i, k, tag, p, tagp;
      double *lb = rsm->lb, *ub = rsm->ub;
      double big, eps, temp, teta;
#     define gap(bnd) (tol1 * (fabs(bnd) > 1.0 ? fabs(bnd) : 1.0))
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (i = 1; i <= m; i++)
         if (big < fabs(aq[i])) big = fabs(aq[i]);
      eps = tol * big;
      /* turn to the case of increasing xN[q] in order to simplify
         program logic */
      if (dir) for (i = 1; i <= m; i++) aq[i] = - aq[i];
      /* initial settings for the first pass */
      k = rsm->indn[q]; /* x[k] = xN[q] */
      if (rsm->type[k] == 'D')
         teta = (ub[k] + gap(ub[k])) - (lb[k] - gap(lb[k]));
      else
         teta = DBL_MAX;
      /* the first look through the list of basis variables */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (aq[i] == 0.0 || fabs(aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'F')
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == 'L')
         {  /* xB[i] has lower bound */
            if (aq[i] > 0.0) continue;
lo_1:       temp = ((lb[k] - gap(lb[k])) - bbar[i]) / aq[i];
         }
         else if (rsm->type[k] == 'U')
         {  /* xB[i] has upper bound */
            if (aq[i] < 0.0) continue;
up_1:       temp = ((ub[k] + gap(ub[k])) - bbar[i]) / aq[i];
         }
         else if (rsm->type[k] == 'D')
         {  /* xB[i] has both lower and upper bounds */
            if (aq[i] < 0.0) goto lo_1; else goto up_1;
         }
         else if (rsm->type[k] == 'S')
         {  /* xB[i] is fixed variable */
            if (aq[i] < 0.0) goto lo_1; else goto up_1;
         }
         else
            insist(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its (relaxed!) bound, temp is
            negative; in this case it is assumes thst xB[i] is exactly
            on its (relaxed!) bound, so temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* compute maximal allowable change of xN[q] */
         if (teta > temp) teta = temp;
      }
      /* initial settings for the second pass */
      p = 0, tagp = -1, big = 0.0;
      k = rsm->indn[q]; /* x[k] = xN[q] */
      if (rsm->type[k] == 'D')
      {  temp = ub[k] - lb[k];
         if (temp <= teta) p = -1, tagp = -1, big = 1.0;
      }
      /* the second look through the list of the basis variable */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (aq[i] == 0.0 || fabs(aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'F')
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == 'L')
         {  /* xB[i] has lower bound */
            if (aq[i] > 0.0) continue;
lo_2:       temp = (lb[k] - bbar[i]) / aq[i];
            tag = 'L';
         }
         else if (rsm->type[k] == 'U')
         {  /* xB[i] has upper bound */
            if (aq[i] < 0.0) continue;
up_2:       temp = (ub[k] - bbar[i]) / aq[i];
            tag = 'U';
         }
         else if (rsm->type[k] == 'D')
         {  /* xB[i] has both lower and upper bounds */
            if (aq[i] < 0.0) goto lo_2; else goto up_2;
         }
         else if (rsm->type[k] == 'S')
         {  /* xB[i] is fixed variable */
            temp = 0.0;
            tag = 'S';
         }
         else
            insist(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its (original!) bound, temp is
            negative; in this case it is assumed that xB[i] is exactly
            on its (original!) bound, so temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the Harris' rule */
         if (temp <= teta && big < fabs(aq[i]))
            p = i, tagp = tag, big = fabs(aq[i]);
      }
      /* restore original signs of the coefficient aq[i] */
      if (dir) for (i = 1; i <= m; i++) aq[i] = - aq[i];
      /* store a tag for xB[p] */
      *_tagp = tagp;
      /* return to the calling program */
      return p;
#     undef gap
}

/* eof */
