/* glprsm/exact_dvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- exact_dvec - compute exact value of delta[i].
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double exact_dvec(RSM *rsm, int i);
--
-- *Description*
--
-- The exact_dvec routine computes exact value of delta[i] using its
-- definition:
--
--    delta[i] = 1 + alfa[i,1]^2 + ... + alfa[i,n]^2
--
-- where alfa[i,j] is the element of the current simplex table placed
-- in i-th row and j-th column.
--
-- This operation is extremely inefficient and may be used only for
-- debugging purposes.
--
-- *Returns*
--
-- The exact_dvec routine returns the computed value of delta[i]. */

double exact_dvec(RSM *rsm, int i)
{     int m = rsm->m, n = rsm->n, j;
      double *zeta, *ai, t;
      insist(1 <= i && i <= m);
      zeta = ucalloc(1+m, sizeof(double));
      ai = ucalloc(1+n, sizeof(double));
      eval_zeta(rsm, i, zeta);
      eval_row(rsm, zeta, ai);
      t = 1.0;
      for (j = 1; j <= n; j++) t += ai[j] * ai[j];
      ufree(zeta), ufree(ai);
      return t;
}

/* eof */
