/* glprsm/eval_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_col - compute column of simplex table.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_col(RSM *rsm, int j, double aj[], int save);
--
-- *Description*
--
-- The eval_col routine computes j-th column of the simplex table, i.e.
-- j-th column of the matrix A~ = -inv(B)*N, and stores its elements to
-- locations aj[1], ..., aj[m] respectively.
--
-- The parameter save is a flag. If this flag is set, it means that the
-- computed column is the column of non-basic variable (xN)q which has
-- been chosen to enter the basis (i.e. j = q). This flag is passed to
-- the ftran routine which is called by the eval_col routine in order to
-- perform forward transformation.
--
-- The j-th column of the simplex table is computed using the following
-- formula:
--
--    A~[j] = - inv(B) * N[j],
--
-- where B is the current basis matrix, N[j] is column of the expanded
-- matrix A, which corresponds to non-basic variable (xN)j. */

void eval_col(RSM *rsm, int j, double aj[], int save)
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, k;
      double *u = aj;
      insist(1 <= j && j <= n);
      k = rsm->indn[j]; /* x[k] = xN[j] */
      /* u = N[j] */
      for (i = 1; i <= m; i++) u[i] = 0.0;
      for (e = rsm->A->col[k]; e != NULL; e = e->col)
         u[e->i] = + e->val;
      /* aj = - inv(B) * u */
      ftran(rsm, u, save);
      for (i = 1; i <= m; i++) aj[i] = - u[i];
      return;
}

/* eof */
