/* glpmat/per_sym.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- per_sym - permute symmetric matrix (A := P*A*P').
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *per_sym(PER *P, MAT *A, void *work[]);
--
-- *Description*
--
-- The per_sym routine computes the product A := P*A*P', where P is a
-- permutation matrix, A is a symmetric matrix specified only by upper
-- triangular part; on exit the matrix A also contains upper triangle.
-- The matrix P remains unchanged on exit. The auxiliary array work
-- should have at least 1+n locations, where n is order of the matrix A
-- (if this parameter is NULL, the routine automatically allocates and
-- frees the working array).
--
-- *Returns*
--
-- The per_sym routine returns a pointer to the matrix A. */

MAT *per_sym(PER *P, MAT *A, void *work[])
{     ELEM *e;
      int i, j;
      if (!(P->n == A->m && P->n == A->n))
         fault("per_sym: product undefined");
      /* compute P*A*P' as if A would be upper triangular */
      per_mat(P, A, work);
      inv_per(P);
      mat_per(A, P, work);
      inv_per(P);
      /* now some elements of A may be in the lower triangular part, so
         they should be removed to the corresponding symmetric positions
         of the upper triangular part */
      for (j = 1; j <= A->n; j++) A->col[j] = NULL;
      for (i = 1; i <= A->m; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
         {  j = e->j;
            if (i > j) e->i = j, e->j = i;
            e->col = A->col[e->j], A->col[e->j] = e;
         }
      }
      /* restore correct row lists */
      for (i = 1; i <= A->m; i++) A->row[i] = NULL;
      for (j = 1; j <= A->n; j++)
      {  for (e = A->col[j]; e != NULL; e = e->col)
            e->row = A->row[e->i], A->row[e->i] = e;
      }
      return A;
}

/* eof */
