/* glpgel/gel_bg.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glpgel.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- gel_bg - sparse gaussian elimination (Bartels & Golub technique).
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- int gel_bg(PER *P, MAT *V, PER *Q, int k1, int k2,
--    void (*func)(int i, int k, double f), double tol, double eps,
--    int *Unz, double work[]);
--
-- *Description*
--
-- The gel_bg routine uses Bartels & Golub version of the gaussian
-- elimination technique to transform the matrix U = P*V*Q to the upper
-- triangular form, where V is the given sparse matrix, P and Q are the
-- given permutation matrices.
--
-- (The matrix U has implicit representation, therefore all operations
-- are actually performed on the matrix V, that differs from U only in
-- the order of rows and columns determining by the permutation matrices
-- P and Q.)
--
-- It is assumed that on entry the matrix U has the following special
-- structure:
--
--     1   k1       k2   n
-- 1   x x x x x x x x x x
--     . x x x x x x x x x
-- k1  . . * * * * * * * *
--     . . * * * * * * * *
--     . . * . * * * * * *
--     . . * . . * * * * *
--     . . * . . . * * * *
-- k2  . . * . . . . * * *
--     . . . . . . . . x x
-- n   . . . . . . . . . x
--
-- Hence only the submatrix with rows k1, k1+1, ..., k2 and columns
-- k1, k1+1, ..., n should be transformed (elements of this submatrix
-- are marked by '*', all other elements of the matrix U are marked by
-- 'x').
--
-- The result of the transformation is the matrices P', V', and Q' that
-- define the upper triangular matrix U' = P'*V'*Q' (these new matrices
-- are stored instead the source matrices P, V, and Q).
--
-- To eliminate subdiagonal elements of the matrix U the routine applies
-- to this matrix elementary transformations of the following types: row
-- permutation, column permutation, and subtraction one row multiplied
-- by a number from other row (gaussian transformation). Each time when
-- gaussian transformation is applied to the matrix U (in order to
-- eliminate one subdiagonal element), the gel_bg routine calls the func
-- routine in order that the calling routine would have possibility to
-- accumulate all information about these transformations (information
-- about row and column permutations are accumulated in the matrices P
-- and Q by the gel_bg routine).
--
-- The point where the func routine is called, and the meaning of the
-- parameters passed to this routine can be explained by means of the
-- following general scheme of elimination process.
--
-- Before elimination the routine shifts columns k1+1, k1+2, ..., k2 of
-- the matrix U by one position to the left and moves the column k1 to
-- the position k2. As a result the matrix U becomes an upper Hessenberg
-- matrix (elements which should be eliminated are marked by '#'):
--
--     1   k1       k2   n
-- 1   x x x x x x x x x x
--     . x x x x x x x x x
-- k1  . . * * * * * * * *
--     . . # * * * * * * *
--     . . . # * * * * * *
--     . . . . # * * * * *
--     . . . . . # * * * *
-- k2  . . . . . . # * * *
--     . . . . . . . . x x
-- n   . . . . . . . . . x
--
-- Then the routine uses gaussian elimination to nullify all subdiagonal
-- elements of the matrix U.
--
-- Bartels & Golub technique assumes partial pivoting in the active
-- (leftmost) column of the active submatrix. Before k-th elimination
-- step (k = k1, k1+1, ..., k2-1) k-th (active) column of the active
-- submatrix has only one subdiagonal element, hence the choice of the
-- pivot is limited only by two elements u[k,k] and u[k+1,k]. In the
-- original Bartels & Golub proposes to choose that element which has
-- largest magnitude for the sake of good numerical stability. However
-- the gel_bg routine tries to improve the sparsity of the matrix U,
-- possibly at the expense of accuracy. The following heuristic rule is
-- used by the routine, where the parameter tol (0 < tol < 1) is the
-- given tolerance:
--
-- if |u[k+1,k]| < tol * |u[k,k]|, the routine chooses u[k,k];
--
-- if |u[k,k]| < tol * |u[k+1,k]|, the routine chooses u[k+1,k];
--
-- in other cases the routine chooses u[k,k] or u[k+1,k] depending on
-- which one has less non-zeros.
--
-- (Therefore the original Bartels & Golub technique corresponds to the
-- case when tol = 1.)
--
-- So, the general scheme of the elimination process is the following:
--
-- (* permute columns of the matrix U *)
-- ...;
-- for (k = k1; k < k2; k++)
-- {  (* k-th elimination step starts here *)
--    (* choose the pivot element u[k,k] or u[k+1,k] *)
--    ...;
--    (* if u[k+1,k] has been chosen, permute rows k and k+1 of the
--       matrix U *)
--    if (...) ...;
--    (* now the pivot element is u[k,k] *)
--    assert(u[k,k] != 0.0);
--    (* eliminate non-zero subdiagonal element u[k+1,k] *)
--    i = k+1;
--    if (u[i,k] == 0.0) continue;
--    (* compute gaussian multiplier *)
--    f = u[i,k] / u[k,k];
--    (* (i-th row of U) := (i-th row of U) - f * (k-th row of U) *)
--    ...;
--    assert(u[i,k] == 0.0);
--    (* i-th row of the matrix U has been transformed *)
--    func(i, k, f);
-- }
-- assert(u[k2,k2] != 0.0);
--
-- Should note that row numbers passed to the func routine correspond to
-- the matrix U (not to V!). The correspondence between row and column
-- numbers of the matrices U = P*V*Q and V is determined by the row
-- permutation matrix P and the column permutation matrix Q. So, if an
-- element u[i,j] of the matrix U corresponds to an element v[i',j'] of
-- the matrix V, the following formulae take a place:
--
-- i' = P->row[i], j' = Q->col[j], i = P->col[i'], j = Q->row[j'].
--
-- The parameter eps is the given tolerance used for checking diagonal
-- elements of the transformed matrix U'. The performed transformation
-- is considered to be satisfactory (from the numerical point of view)
-- if the following condition is true for all diagonal elements of the
-- matrix U':
--
-- |u'[k,k]| >= eps * max|u'[k,*]| > 0,
--
-- i.e. if each diagonal element of the matrix U' is non-zero and it is
-- not too small among other elements in the same row. (This condition
-- gives some guarantee that the solution of the system with the matrix
-- U' will be relatively accurate.)
--
-- Should note that the mentioned condition is checked only for rows
-- k1, k1+1, ..., k2.
--
-- On entry the variable Unz should define the total number of non-zeros
-- in the source matrix U. On exit this variable will define the total
-- number of non-zeros in the transformed matrix U'. (If on entry Unz is
-- set to zero, on exit Unz will define the difference between total
-- numbers of non-zeros in the matrices U' and U.)
--
-- The auxiliary array work should have at least n elements, where n is
-- the order of the matrix U.
--
-- *Returns*
--
-- The gel_bg routine returns one of the following error codes:
--
--  0 - no errors;
-- -k - if k1 <= k < k2, on k-th elimination step both elements u[k,k]
--      and u[k+1,k] are found equal to zero; if k = k2, after the last
--      elimination step the element u[k2,k2] is found equal to zero;
-- +k - if k1 <= k < k2, on k-th elimination step numerical stability
--      condition (see above) has been violated; if k = k2, this
--      condition has been violated for the element u[k2,k2].
--
-- Should note that if the matrix U passed to the gel_bg routine is
-- already a result of preceding transformations of some initial matrix
-- U0, the appearence of numerical errors in this routine is usually
-- connected with all sequence of transformations. Hence in such case
-- instead this routine the general gaussian elimination should be
-- applied directly to the initial matrix U0.
--
-- *Reference*
--
-- Bartels R.H., Golub G.H. The simplex method of linear programming
-- using LU decomposition. Comm. ACM, 12 (1969), pp. 266-68. */

#define iU(i) (P->col[i])
/* converts row number of V to row number of U */

#define iV(i) (P->row[i])
/* converts row number of U to row number of V */

#define jU(j) (Q->row[j])
/* converts column number of V to column number of U */

#define jV(j) (Q->col[j])
/* converts column number of U to column number of V */

int gel_bg(PER *P, MAT *V, PER *Q, int k1, int k2,
      void (*func)(int i, int k, double f), double tol, double eps,
      int *Unz, double work[])
{     ELEM *vpq, *viq, *e, *ee, *en;
      int n = V->m, i, j, k, p, q, nz1, nz2, ret = 0;
      double rmax1, rmax2, f, drop = 1e-15;
      if (V->m != V->n)
         fault("gel_bg: transformed matrix is not square");
      if (!(P->n == n && Q->n == n))
         fault("gel_bg: permutation matrices have invalid order");
      if (!(1 <= k1 && k1 <= k2 && k2 <= n))
         fault("gel_bg: initial active submatrix has invalid position");
      /* shift columns k1+1, k1+2, ..., k2 of the matrix U to the left
         by one position and move the column k1 to the position k2 */
      for (k = k1; k < k2; k++)
      {  /* permute columns k and k+1 of the matrix U */
         int t1, t2;
         t1 = jV(k); t2 = jV(k+1);
         jV(k)   = t2; jU(t2) = k;
         jV(k+1) = t1; jU(t1) = k+1;
      }
      /* clear auxiliary array */
      for (j = 1; j <= n; j++) work[j] = 0.0;
      /* main loop of gaussian elimination */
      for (k = k1; k < k2; k++)
      {  /* the goal of k-th step is to nullify the subdiagonal element
            u[k+1,k] */
         p = iV(k);   /* k-th row of U    = p-th row of V */
         q = jV(k);   /* k-th column of U = q-th column of V */
         i = iV(k+1); /* k+1-th row of U  = i-th row of V */
         /* search for element u[k,k] = v[p,q], count non-zero elements,
            and determine largest of absolute values of elements in k-th
            row of the matrix U */
         vpq = NULL; nz1 = 0; rmax1 = 0.0; /* = max|v[p,*]| */
         for (e = V->row[p]; e != NULL; e = e->row)
         {  if (jU(e->j) < k || e->val == 0.0)
err:           fault("gel_bg: transformed matrix has invalid pattern");
            if (e->j == q) vpq = e;
            nz1++;
            if (rmax1 < fabs(e->val)) rmax1 = fabs(e->val);
         }
         /* search for element u[k+1,k] = v[i,q], count non-zeros, and
            determine largest of absolute values of elements in k+1-th
            row of the matrix U */
         viq = NULL; nz2 = 0; rmax2 = 0.0; /* = max|v[i,*]| */
         for (e = V->row[i]; e != NULL; e = e->row)
         {  if (jU(e->j) < k || e->val == 0.0) goto err;
            if (e->j == q) viq = e;
            nz2++;
            if (rmax2 < fabs(e->val)) rmax2 = fabs(e->val);
         }
         /* if u[k,k] = u[k+1,k] = 0, it's impossible to choose the
            pivot element */
         if (vpq == NULL && viq == NULL)
         {  ret = -k;
            goto done;
         }
         /* if u[k+1,k] = 0 then u[k,k] != 0; hence only checking the
            numerical stability condition is needed */
         if (viq == NULL) goto chk;
         /* if u[k,k] = 0 then u[k+1,k] != 0; hence permutation k-th
            and k+1-th rows of the matrix U and checking the numerical
            stability condition are needed */
         if (vpq == NULL) goto per;
         /* if |u[k+1,k]| < tol * |u[k,k]|, the element u[k,k] should
            be chosen as pivot */
         if (fabs(viq->val) < tol * fabs(vpq->val)) goto chk;
         /* if |u[k,k]| < tol * |u[k+1,k]|, the element u[k+1,k] should
            be chosen as pivot */
         if (fabs(vpq->val) < tol * fabs(viq->val)) goto per;
         /* use freedom to choose u[k,k] or u[k+1,k] depending which
            row (k-th or k+1-th) has less non-zeros */
         if (nz1 <= nz2) goto chk;
per:     /* permute k-th and k+1-th rows of the matrix U */
         p = iV(k+1); i = iV(k);
         iV(k)   = p; iU(p) = k;
         iV(k+1) = i; iU(i) = k+1;
         e = vpq; vpq = viq; viq = e;
         f = rmax1; rmax1 = rmax2; rmax2 = f;
chk:     /* check the numerical stability condition for the diagonal
            (i.e. pivot) element u[k,k] */
         if (fabs(vpq->val) < eps * rmax1)
         {  ret = +k;
            goto done;
         }
         /* if u[k+1,k] = 0, no elimination is needed */
         if (viq == NULL) continue;
         /* compute gaussian multiplier f = v[i,q]/v[p,q] */
         f = viq->val / vpq->val;
         /* work := (i-th row of V) */
         for (e = V->row[i]; e != NULL; e = e->row)
            work[e->j] = e->val;
         /* work := work - f * (p-th row of V) */
         for (e = V->row[p]; e != NULL; e = e->row)
            work[e->j] -= f * e->val;
         /* the eliminated element u[k+1,k] = v[i,q] should be exact
            zero */
         work[q] = 0.0;
         /* now new elements of i-th row of the matrix V are placed in
            the array work */
         /* replace existing elements of i-th row */
         for (e = V->row[i]; e != NULL; e = e->row)
         {  j = e->j;
            /* if |v[i,j]| < drop * max|v[i,*]|, where drop is relative
               drop tolerance, the element v[i,j] can be replaced by
               exact zero; this slightly decreases accuracy but allows
               to improve sparsity */
            if (fabs(work[j]) < drop * rmax2) work[j] = 0.0;
            /* replace element v[i,j] */
            e->val = work[j], work[j] = 0.0;
         }
         /* create new elements, which appeared in i-th row as a result
            of elimination */
         for (e = V->row[p]; e != NULL; e = e->row)
         {  j = e->j;
            /* (see notice above) */
            if (fabs(work[j]) < drop * rmax2) work[j] = 0.0;
            if (work[j] == 0.0) continue;
            /* create new (non-zero) element v[i,j] */
            new_elem(V, i, j, work[j]), (*Unz)++, work[j] = 0.0;
         }
         /* (now the array work again contains all zeros) */
         /* i-th row of the matrix V has been transformed */
         func(iU(i) /* = k+1 */, k, f);
         /* now all zeros appeared in i-th row of the matrix V as a
            result of elimination should be removed */
         for (e = V->row[i], V->row[i] = NULL; e != NULL; e = en)
         {  en = e->row;
            if (e->val != 0.0)
            {  /* v[i,j] != 0, keep this element */
               e->row = V->row[i], V->row[i] = e;
               continue;
            }
            /* v[i,j] = 0; remove this element from the list of j-th
               column of the matrix V */
            j = e->j;
            if (V->col[j] == e)
               V->col[j] = e->col;
            else
            {  for (ee = V->col[j]; ee != NULL; ee = ee->col)
                  if (ee->col == e) break;
               insist(ee != NULL);
               ee->col = e->col;
            }
            /* return the element v[i,j] to the memory pool */
            free_atom(V->pool, e), (*Unz)--;
         }
         /* k-th elimination step is finished */
      }
      /* check the numerical stability condition for the last diagonal
         element u[k2,k2] */
      vpq = NULL; rmax1 = 0.0;
      for (e = V->row[iV(k)]; e != NULL; e = e->row)
      {  if (e->j == jV(k)) vpq = e;
         if (rmax1 < fabs(e->val)) rmax1 = fabs(e->val);
      }
      if (vpq == NULL)
         ret = -k;
      else if (fabs(vpq->val) < eps * rmax1)
         ret = +k;
done: /* return to the calling program */
      return ret;
}

/* eof */
