/* glpchol/solve_chol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpchol.h"

/*----------------------------------------------------------------------
-- solve_chol - solve linear system using Cholesky factorization.
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- void solve_chol(CHOL *chol, double x[], double work[]);
--
-- *Description*
--
-- The solve_chol routine obtains solution of the linear system A*x = b,
-- where A is symmetric positive definite coefficient matrix, x is dense
-- vector of unknowns, b is dense vector of right-hand sides.
--
-- The parameter chol should specify Cholseky factorization of the
-- coefficient matrix A computed by means of the decomp_chol routine.
-- If the decomp_chol routine detected that A is close to semidefinite
-- (i.e. chol->sing > 0), the system A*x = b should be consistent.
--
-- On entry the array x should contain elements of the vector b in
-- locations x[1], x[2], ..., x[n], where n is order of the system. On
-- exit this array will contain the vector x in the same locations.
--
-- The auxiliary array work should contain at least 1+n elements. If
-- this parameter is NULL, the routine automatically allocates and frees
-- the working array. */

void solve_chol(CHOL *chol, double x[], double _work[])
{     int n = chol->n;
      double *work = _work;
      if (_work == NULL) work = ucalloc(1+n, sizeof(double));
      /* P*A*P' = U'*U; A = P'*U'*U*P; inv(A) = P'*inv(U)*inv(U')*P */
      per_vec(work, chol->P, x);
      ut_solve(chol->U, work);
      u_solve(chol->U, work);
      iper_vec(x, chol->P, work);
      if (_work == NULL) ufree(work);
      return;
}

/* eof */
