/* glpbbm/branch_drtom.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glpbbm.h"

/*----------------------------------------------------------------------
-- branch_drtom - choose a branch using Driebeek and Tomlin heuristic.
--
-- *Synopsis*
--
-- #include "glpbbm.h"
-- int branch_drtom(BBDATA *bb, int *what);
--
-- *Description*
--
-- The branch_drtom routine chooses a branching variable which is an
-- integer structural variable xS[j] (1 <= j <= n).
--
-- The routine also sets the parameter what which tells the solver what
-- of two subproblems should be solved after the current problem:
--
-- 'L' - that one where xS[j] >= new lower bound = ceil(beta)  > beta
-- 'U' - that one where xS[j] <= new upper bound = floor(beta) < beta
--
-- where beta is the (fractional) value of the variable xS[j] in basis
-- solution of the current problem.
--
-- *Returns*
--
-- The branch_drtom routine returns a number of the variable xS[j].
--
-- *Heuristic*
--
-- The branch_drtom routine is based on easy heuristic proposed in:
--
-- Driebeek N.J. An algorithm for the solution of mixed-integer
-- programming problems, Management Science, 12: 576-87 (1966)
--
-- and improved in:
--
-- Tomlin J.A. Branch and bound methods for integer and non-convex
-- programming, in J.Abadie (ed.), Integer and Nonlinear Programming,
-- North-Holland, Amsterdam, pp. 437-50 (1970).
--
-- Note that the implementation of this heuristic is time-expensive,
-- because computing one-step degradation (see the routines below) for
-- each basic integer structural variable requires one BTRAN. */

/*----------------------------------------------------------------------
-- degrad - compute degradation of the objective function.
--
-- If tagp = 'L', this routine computes degradation (worsening) of the
-- objective function when the basic variable xB[p] increasing goes to
-- its new lower bound ceil(beta), where beta is value of this variable
-- in the current basis solution.
--
-- If tagp = 'U', this routine computes degradation (worsening) of the
-- objective function when the basic variable xB[p] decreasing goes to
-- its new upper bound floor(beta).
--
-- The array ap is the p-th row of the current simplex table. The array
-- cbar is the vector of reduced costs of non-basic variables.
--
-- In order to compute degradation the routine implicitly performs one
-- iteration of the dual simplex method (note that the current basis is
-- noy changed).
--
-- Note that actual degradation of the objective function (which would
-- happen if the corresponding problem with the new bound of xB[p] were
-- solved completely) is not less than the one-step degradation computed
-- by this routine. In other word this routine computes an estimation
-- (lower bound) of the true degradation. */

static double degrad(BBDATA *bb, int p, int tagp, double ap[],
      double cbar[])
{     int m = bb->m, n = bb->n, q, k;
      double cur_val, new_val, delta, deg;
      /* determine the current value of the variable xB[p] (note that
         the current value is assumed to be non-integer) */
      cur_val = bb->bbar[p];
      /* determine the new value which the variable xB[p] would have if
         it would leave the basis */
      switch (tagp)
      {  case 'L':
            /* xB[p] increases and goes to its new lower bound */
            new_val = ceil(cur_val);
            insist(new_val > cur_val);
            break;
         case 'U':
            /* xB[p] deacreses and goes to its new upper bound */
            new_val = floor(cur_val);
            insist(new_val < cur_val);
            break;
         default:
            insist(tagp != tagp);
      }
      /* choose the non-basic variable xN[q] which would enter the
         basis instead xB[p] in order to keep dual feasibility of the
         current basis solution (note that the current basis is assumed
         to be optimal and therefore dual feasible) */
      q = dual_col(bb->rsm, tagp, ap, cbar, 1e-10);
      /* if the choice is impossible, the problem with the new bound of
         xB[p] has no (primal) feasible solution; formally in this case
         the degradation of the objective function is infinite */
      if (q == 0) return DBL_MAX;
      /* so far as the p-th row of the current simplex table is
         xB[p] = ... + ap[q] * xN[q] + ..., where ap[q] is an influence
         coefficient, increment of xN[q] in the adjacent basis solution
         is delta(xN[q]) = delta(xB[p]) / ap[q], where delta(xB[p]) is
         increment of xB[p] which is known */
      delta = (new_val - cur_val) / ap[q];
      /* Tomlin noticed that if the variable xN[q] is of integer kind,
         its increment should be not less than one (in absolute value)
         in order that the new subproblem could have an integer feasible
         solution */
      k = bb->rsm->indn[q]; /* x[k] = xN[q] */
      if (m+1 <= k && k <= m+n && bb->mip->kind[k-m])
      {  /* x[k] = xN[q] is integer structural variable */
         if (-1.0 < delta && delta <  0.0) delta = -1.0;
         if ( 0.0 < delta && delta < +1.0) delta = +1.0;
      }
      /* so far as the objective row of the current simplex table is
         Z = ... + cbar[q] * xN[q] + ..., where cbar[q] is the reduced
         cost of xN[q], knowing increment of xN[q] we can determine the
         corresponding degradation of the objective function Z */
      deg = cbar[q] * delta;
      /* since the current basis solution is optimal and the new bound
         of xB[p] makes the basis primal infeasible, the degradation is
         always positive (in case of minimization) that corresponds to
         worsening of the objective function (however due to round-off
         errors the degradation can be slightly negative) */
      return deg;
}

/*----------------------------------------------------------------------
-- branch_drtom - choose a branching variable.
--
-- This routine looks up the list of integer structural variable and
-- chooses that one which being limited by its new lower or upper bound
-- would involve greatest degradation of the objective function. */

int branch_drtom(BBDATA *bb, int *what)
{     int m = bb->m, n = bb->n, i, j, k, p, this = 0;
      double beta, deg_L, deg_U, deg_max = -DBL_MAX;
      double *c, *pi, *cbar, *zeta, *ap;
      /* allocate working arrays */
      c = ucalloc(1+m+n, sizeof(double));
      pi = ucalloc(1+m, sizeof(double));
      cbar = ucalloc(1+n, sizeof(double));
      zeta = ucalloc(1+m, sizeof(double));
      ap = ucalloc(1+n, sizeof(double));
      /* build the expanded vector of coefficients of the objective
         function (the case of maximization is reduced to the case of
         minimization to simplify program logic) */
      for (i = 1; i <= m; i++) c[i] = 0.0;
      for (j = 1; j <= n; j++)
         c[m+j] = (bb->mip->dir == '-' ? +1.0 : -1.0) * bb->mip->c[j];
      /* compute simplex multipliers */
      eval_pi(bb->rsm, c, pi);
      /* compute reduced costs of non-basic variables */
      eval_cbar(bb->rsm, c, pi, cbar);
      /* look up the list of structural variables */
      for (j = 1; j <= n; j++)
      {  /* skip continuous variable */
         if (!bb->mip->kind[j]) continue;
         /* x[k] = xS[j] */
         k = m + j;
         /* skip non-basic variable */
         if (bb->rsm->posx[k] < 0) continue;
         /* xB[p] = x[k] */
         p = +bb->rsm->posx[k];
         insist(1 <= p && p <= m);
         /* beta = value of xS[j] in the current basis solution */
         beta = bb->bbar[p];
         /* skip basic variable which has an integer value */
         if (beta == floor(beta)) continue;
         /* compute p-th row of inv(B) */
         eval_zeta(bb->rsm, p, zeta);
         /* compute p-th row of the simplex table */
         eval_row(bb->rsm, zeta, ap);
         /* compute degradation (worsening) of the objective function
            when xS[j] = xB[p] goes to its new lower bound */
         deg_L = degrad(bb, p, 'L', ap, cbar);
         /* compute degradation (worsening) of the objective function
            when xS[j] = xB[p] goes to its new upper bound */
         deg_U = degrad(bb, p, 'U', ap, cbar);
         /* choose that variable which involves maximal degradation of
            the objective function and tell the solver to get next the
            problem where the degradation is less than for the other */
         if (deg_max < deg_L || deg_max < deg_U)
         {  this = j;
            if (deg_L > deg_U)
               deg_max = deg_L, *what = 'U';
            else
               deg_max = deg_U, *what = 'L';
         }
      }
      insist(1 <= this && this <= n);
      /* free working arrays */
      ufree(c);
      ufree(pi);
      ufree(cbar);
      ufree(zeta);
      ufree(ap);
      /* return to the calling program */
      return this;
}

/* eof */
