/* glpapi/glp_read_lpm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpk.h"
#include "glplang.h"

/*----------------------------------------------------------------------
-- glp_read_lpm - read linear programming model written in GLPK/L.
--
-- *Synopsis*
--
-- #include "glpk.h:
-- int glp_read_lpm(char *fname);
--
-- *Description*
--
-- The glp_read_lpm routine reads an LP model written in the modeling
-- language GLPK/L from the text file whose name is the character string
-- fname into the workspace.
--
-- As a rule the workspace should be empty before a call to the
-- glp_read_lpm routine, i.e. the workspace should contain no rows and
-- no columns.
--
-- For detailed description of GLPK/L modeling language see the program
-- documentation.
--
-- *Control parameters*
--
-- The behavior of the glp_read_lpm routine depends on the following
-- control parameters:
--
-- fn_gener (name of the file to output generated LP/MIP problem).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_read_lpm(char *fname)
{     struct prob *prob = NULL;
      int m, n, i, j;
      char fn_gener[MAX_NAME+1];
      /* initialize the language processor environment */
      if (initialize(fname) != 0) goto fail;
      /* set error handling */
      pdb->flag = 1;
      if (setjmp(pdb->jump)) goto fail;
      /* parse model description */
      load_model();
      /* output generated LP/MIP problem to the specified file in plain
         text format */
      glp_get_cpar("fn_gener", fn_gener);
      if (fn_gener[0] != '\0')
      {  int ret;
         ret = gener_lp(fn_gener);
         if (ret != 0) goto fail;
      }
      /* create data structure for generating LP/MIP */
      prob = create_prob();
      m = prob->m;
      n = prob->n;
      /* set problem name */
      glp_set_cpar("problem", pdb->model_name);
      /* create columns that correspond model variables */
      for (j = 1; j <= n; j++)
      {  VAR *var = prob->memb[m+j]->link;
         char *name = gener_name(prob, m+j);
         int type;
         if (glp_create_item(GLP_COL, name))
         {  error("glp_read_lpm: error on creating column `%s'", name);
            goto fail;
         }
         if (var->kind) glp_set_kind(GLP_COL, GLP_YES);
         switch (var->type)
         {  case 'F': type = GLP_FR; break;
            case 'L': type = GLP_LO; break;
            case 'U': type = GLP_UP; break;
            case 'D': type = GLP_DB; break;
            case 'S': type = GLP_FX; break;
            default: insist(var->type != var->type);
         }
         glp_set_bounds(GLP_COL, type, var->lb, var->ub);
      }
      /* create rows that correspond model constraints; build the
         constraint matrix */
      for (i = 1; i <= m; i++)
      {  CONS *cons = prob->memb[i]->link;
         char *name = gener_name(prob, i);
         struct elem *form, *e;
         int type;
         if (glp_create_item(GLP_ROW, name))
         {  error("glp_read_lpm: error on creating row `%s'", name);
            goto fail;
         }
         form = build_form(prob, i);
         if (form == NULL) goto fail;
         for (e = form; e != NULL; e = e->next)
         {  if (e->j == 0)
            {  if (cons->type == 'F')
               {  error("glp_read_lpm: free row `%s' has constant term",
                     name);
                  goto fail;
               }
               cons->lb -= e->val, cons->ub -= e->val;
            }
            else
            {  glp_find_item(GLP_COL, gener_name(prob, m+e->j));
               glp_new_coef(e->val);
            }
         }
         erase_form(prob, form);
         switch (cons->type)
         {  case 'F': type = GLP_FR; break;
            case 'L': type = GLP_LO; break;
            case 'U': type = GLP_UP; break;
            case 'D': type = GLP_DB; break;
            case 'S': type = GLP_FX; break;
            default: insist(cons->type != cons->type);
         }
         glp_set_bounds(GLP_ROW, type, cons->lb, cons->ub);
      }
      /* set the objective function */
      switch (prob->obj_dir)
      {  case '-':
            insist(glp_set_ipar("obj_dir", GLP_MIN) == 0);
            break;
         case '+':
            insist(glp_set_ipar("obj_dir", GLP_MAX) == 0);
            break;
         default:
            insist(prob->obj_dir != prob->obj_dir);
      }
      glp_set_rpar("c0", 0.0);
      if (prob->obj_row == 0)
      {  glp_set_ipar("option", GLP_ANY);
         glp_set_cpar("obj_row", "");
      }
      else
      {  glp_set_ipar("option", GLP_FIN);
         glp_set_cpar("obj_row", gener_name(prob, prob->obj_row));
      }
      /* free auxiliary data structure */
      delete_prob(prob);
      /* terminate the language processor environment */
      terminate();
      /* model has been read successfully */
      return 0;
fail: /* the operation failed */
      error("glp_read_lpm: processing terminated due to errors");
      if (prob != NULL) delete_prob(prob);
      if (pdb != NULL) terminate();
      return 1;
}

/* eof */
