/* glpapi/glp_integer.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpk.h"
#include "glpbbm.h"
#include "glprsm.h"

/*----------------------------------------------------------------------
-- glp_integer - solve MIP problem using branch-and-bound procedure.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_integer(void);
--
-- *Description*
--
-- The glp_integer is a MIP problem solver based on the branch-and-bound
-- procedure and the revised dual simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Since many MIP problems may take a long time, the glp_integer routine
-- reports some visual information about current status of the search.
-- The information is sent to stdout approximately once per second and
-- has the following format:
--
--    +nnn: mip = xxx; lp = yyy (aaa; sss)
--
-- where nnn is total number of simplex iteration, xxx is a value of
-- the objective function that corresponds to the best MIP solution
-- (if this solution has not been found yet, xxx is the text "not found
-- yet"), yyy is a value of the objective function that corresponds to
-- the initial relaxed optimal solution (it is not changed during all
-- the optimization process), aaa is number of subroblems in the active
-- list, sss is number of subproblems which have been solved yet.
--
-- Note that currently this solver uses easy heuristics for branching
-- and backtracking and therefore it is not perfect. Most probably this
-- solver fits for MIP problems that have not so many integer variables
-- (several tens, not more). Of course, hard or very large MIP problems
-- can't be solved by this routine.
--
-- *Control parameters*
--
-- The behavior of the glp_integer routine depends on a set of control
-- parameters which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note, for example, if the problem
--     has no integer feasible solution, the solver returns zero code);
-- 1 - it's not possible to start solving the problem due to incorrect
--     data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

int glp_integer(void)
{     LP *lp;
      LPSOL *sol = NULL;
      struct rsm1_cp rsm_cp;
      int m, n, option, form, branch, btrack, ret;
      struct bbm1_cp bbm_cp;
      /* extract LP problem data from the workspace */
      lp = extract_lp();
      if (lp == NULL)
      {  ret = 1;
         goto done;
      }
      if (lp->kind == NULL)
      {  error("glp_integer: problem has no integer variables");
         ret = 1;
         goto done;
      }
      m = lp->m, n = lp->n;
      /* create LP solution block */
      sol = create_lpsol(m, n);
      /* STAGE 1: FIND OPTIMAL SOLUTION OF RELAXED LP PROBLEM */
      /* set control parameters */
      rsm_cp.what = 2; /* optimal solution is required */
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     rsm_cp.form = 0; break;
         case GLP_RFI_BG:  rsm_cp.form = 1; break;
         case GLP_RFI_FT:  rsm_cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("scale", &rsm_cp.scale);
      glp_get_ipar("spx_use_dual", &rsm_cp.dual);
      glp_get_ipar("spx_steep", &rsm_cp.steep);
      glp_get_ipar("spx_relax", &rsm_cp.relax);
      glp_get_rpar("tol_bnd", &rsm_cp.tol_bnd);
      glp_get_rpar("tol_dj", &rsm_cp.tol_dj), rsm_cp.tol_dj *= 0.15;
      glp_get_rpar("tol_piv", &rsm_cp.tol_piv);
      rsm_cp.iter_max = 0;
      glp_get_ipar("round", &rsm_cp.round);
      /* solve LP problem by means of the revised simplex method */
      ret = rsm1_driver(lp, sol, &rsm_cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* check if the relaxed problem has been solved successfully */
      if (!(ret == 0 && sol->status == 'O'))
      {  error("glp_integer: integer optimization not possible");
         if (ret != 0) ret = 2;
         goto done;
      }
      /* STAGE 2: SOLVE MIP PROBLEM USING BRANCH-AND-BOUND METHOD */
      /* set control parameters */
      glp_get_ipar("option", &option);
      switch (option)
      {  case GLP_INI:     bbm_cp.what = 0; break;
         case GLP_ANY:     bbm_cp.what = 1; break;
         case GLP_FIN:     bbm_cp.what = 2; break;
         default:          insist(option != option);
      }
      glp_get_ipar("mip_branch", &branch);
      switch (branch)
      {  case GLP_FIRST:   bbm_cp.branch = BB_FIRST; break;
         case GLP_LAST:    bbm_cp.branch = BB_LAST;  break;
         case GLP_DRTOM:   bbm_cp.branch = BB_DRTOM; break;
         default:          insist(branch != branch);
      }
      glp_get_ipar("mip_btrack", &btrack);
      switch (btrack)
      {  case GLP_FIFO:    bbm_cp.btrack = BB_FIFO;  break;
         case GLP_LIFO:    bbm_cp.btrack = BB_LIFO;  break;
         case GLP_BESTP:   bbm_cp.btrack = BB_BESTP; break;
         default:          insist(btrack != btrack);
      }
      glp_get_rpar("tol_int", &bbm_cp.tol_int);
      glp_get_rpar("tol_obj", &bbm_cp.tol_obj);
      glp_get_ipar("spx_form", &form);
      switch (form)
      {  case GLP_EFI:     bbm_cp.form = 0; break;
         case GLP_RFI_BG:  bbm_cp.form = 1; break;
         case GLP_RFI_FT:  bbm_cp.form = 2; break;
         default:          insist(form != form);
      }
      glp_get_ipar("spx_steep", &bbm_cp.steep);
      glp_get_ipar("spx_relax", &bbm_cp.relax);
      glp_get_rpar("tol_bnd", &bbm_cp.tol_bnd);
      glp_get_rpar("tol_dj", &bbm_cp.tol_dj);
      glp_get_rpar("tol_piv", &bbm_cp.tol_piv);
      bbm_cp.iter_max = 0;
      glp_get_ipar("round", &bbm_cp.round);
      /* solve MIP problem by means of branch-and-bound method */
      ret = bbm1_driver(lp, sol, &bbm_cp);
      /* store solution back to the workspace */
      store_lpsol(sol);
      /* check if the integer problem has been solved successfully */
      if (ret != 0) ret = 2;
done: /* free working data structures */
      if (lp != NULL) delete_lp(lp);
      if (sol != NULL) delete_lpsol(sol);
      /* return to the application program */
      return ret;
}

/* eof */
