/* glpapi/glp_get_bounds.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_bounds - get bounds of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_bounds(int what, int *type, double *lb, double *ub);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_bounds routine stores the type, the
-- lower bound and the upper bound of the current row (i.e. of the
-- corresponding auxiliary variable) to locations, which parameters
-- type, lb, and ub point to, respectively.
--
-- If what = GLP_COL, the glp_get_bounds routine stores the same
-- information, but for the current column (i.e. for the corresponding
-- structural variable).
--
-- If some of pointers type, lb, or ub is NULL, the corresponding value
-- is not stored.
--
-- Types and the corresponding bounds of rows and columns are shown in
-- the following table:
--
--     Type          Bounds              Note
--    ------    -----------------    -------------
--    GLP_FR    -inf <  x <  +inf    free variable
--    GLP_LO      lb <= x <  +inf    lower bound
--    GLP_UP    -inf <  x <=  ub     upper bound
--    GLP_DB      lb <= x <=  ub     double bound
--    GLP_FX            x  =  lb     fixed variable
--
-- where x is the auxiliary (in case of row) or structural (in case of
-- column) variable.
--
-- If the current row/column has no lower or/and upper bound, the
-- corresponding values of lb or/and ub will be set to zero. If the
-- current row is og GLP_FX type, both values lb and ub will be set to
-- the fixed value of the row/column.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_bounds(int what, int *type, double *lb, double *ub)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      if (type != NULL) *type = item->type;
      if (lb != NULL) *lb = item->lb;
      if (ub != NULL) *ub = item->ub;
      return 0;
}

/* eof */
