/* glpsol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- This program is a stand-alone LP/MIP solver based on the revised
-- simplex method and the branch-and-bound procedure.
----------------------------------------------------------------------*/

static char *version = "GLPSOL -- GLPK LP/MIP Solver, Version 2.3";

static char *dat_fname = "";
/* name of the input text file which contains LP problem data in MPS or
   LPM format */

static int dat_fmt = 0;
/* format of the input text file (0 means MPS, 1 means LPM) */

static char *out_fname = "";
/* name of the output text file to which the final basis solution will
   be sent using printable format (optional) */

static int nomip = 0;
/* if this flag is set, the solver considers all integer variables as
   continuous (this allows solving MIP problem as pure LP) */

/*----------------------------------------------------------------------
-- display_help - display help.
--
-- This routine displays help information about the program as it is
-- required by the GNU Coding Standards. */

static void display_help(char *my_name)
{     print("Usage: %s [options...] data-file", my_name);
      print("");
      print("General options:");
      print("   --mps             read LP data using MPS format (defaul"
         "t)");
      print("   --lpm             read LP data using LPM format");
      print("   --min             objective function should be minimize"
         "d (default)");
      print("   --max             objective function should be maximize"
         "d");
      print("   --ini             only initial solution is required");
      print("   --any             any feasible solution is required");
      print("   --fin             final solution is required (default)")
         ;
      print("   -o filename, --output filename");
      print("                     send solution to filename using print"
         "able format");
      print("   --textbook        use options --noscale, --efi, --prima"
         "l, --nosteep,");
      print("                     --norelax, --first, and --lifo by def"
         "ault");
      print("   --scale           scale the problem (default)");
      print("   --noscale         do not scale the problem");
      print("   --efi             use EFI");
      print("   --rfi-bg          use RFI + Bartels & Golub updating te"
         "chnique");
      print("                     (default)");
      print("   --rfi-ft          use RFI + Forrest & Tomlin updating t"
         "echnique");
      print("   --primal          find feasible solution using primal s"
         "implex");
      print("                     (default)");
      print("   --dual            find feasible solution using dual sim"
         "plex");
      print("   --steep           use steepest edge technique (default)"
         );
      print("   --nosteep         use standard \"textbook\" pricing");
      print("   --relax           use Harris' two-pass ratio test (defa"
         "ult)");
      print("   --norelax         use standard \"textbook\" ratio test")
         ;
      print("   --sum             enable multiplets in constraint matri"
         "x (will be");
      print("                     replaced by their sum)");
      print("   --nosum           disable multiplets in constraint matr"
         "ix (default)");
      print("   -h, --help        display this help information and exi"
         "t");
      print("   -v, --version     display program version and exit");
      print("");
      print("Options specific to MIP:");
      print("   --nomip           consider all integer variables as con"
         "tinuous (this");
      print("                     allows solving MIP problem as pure LP"
         ")");
      print("   --first           branch on first integer variable");
      print("   --last            branch on last integer variable");
      print("   --drtom           branch using heuristic by Driebeck an"
         "d Tomlin");
      print("                     (default)");
      print("   --fifo            backtrack using FIFO strategy");
      print("   --lifo            backtrack using LIFO strategy");
      print("   --bestp           backtrack using the best projection h"
         "euristic");
      print("                     (default)");
      print("");
      print("Please, report bugs to <bug-glpk@gnu.org>");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- display_version - display version.
--
-- This routine displays version information for the program as it is
-- required by the GNU Coding Standards. */

static void display_version(void)
{     print("%s", version);
      print("Copyright (C) 2000, 2001 Andrew Makhorin <mao@gnu.org>");
      print("This program is free software; you may redistribute it und"
         "er the terms of");
      print("the GNU General Public License. This program has absolutel"
         "y no warranty.");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- process_cmdline - process command line parameters.
--
-- This routine processes parameters specified in command line. */

static void process_cmdline(int argc, char *argv[])
{     int k;
      for (k = 1; k < argc; k++)
      {  if (strcmp(argv[k], "--mps") == 0)
            dat_fmt = 0;
         else if (strcmp(argv[k], "--lpm") == 0)
            dat_fmt = 1;
         else if (strcmp(argv[k], "--min") == 0)
            glp_set_ipar("obj_dir", GLP_MIN);
         else if (strcmp(argv[k], "--max") == 0)
            glp_set_ipar("obj_dir", GLP_MAX);
         else if (strcmp(argv[k], "--ini") == 0)
            glp_set_ipar("option", GLP_INI);
         else if (strcmp(argv[k], "--any") == 0)
            glp_set_ipar("option", GLP_ANY);
         else if (strcmp(argv[k], "--fin") == 0)
            glp_set_ipar("option", GLP_FIN);
         else if (strcmp(argv[k], "-o") == 0 ||
                  strcmp(argv[k], "--output") == 0)
         {  k++;
            if (k == argc)
            {  error("No output file name specified");
               exit(EXIT_FAILURE);
            }
            if (out_fname[0] != '\0')
            {  error("Only one output file allowed");
               exit(EXIT_FAILURE);
            }
            out_fname = argv[k];
         }
         else if (strcmp(argv[k], "--textbook") == 0)
         {  glp_set_ipar("scale", GLP_NO);
            glp_set_ipar("spx_form", GLP_EFI);
            glp_set_ipar("spx_use_dual", GLP_NO);
            glp_set_ipar("spx_steep", GLP_NO);
            glp_set_ipar("spx_relax", GLP_NO);
            glp_set_ipar("branch", GLP_FIRST);
            glp_set_ipar("btrack", GLP_LIFO);
         }
         else if (strcmp(argv[k], "--scale") == 0)
            glp_set_ipar("scale", GLP_YES);
         else if (strcmp(argv[k], "--noscale") == 0)
            glp_set_ipar("scale", GLP_NO);
         else if (strcmp(argv[k], "--efi") == 0)
            glp_set_ipar("spx_form", GLP_EFI);
         else if (strcmp(argv[k], "--rfi-bg") == 0)
            glp_set_ipar("spx_form", GLP_RFI_BG);
         else if (strcmp(argv[k], "--rfi-ft") == 0)
            glp_set_ipar("spx_form", GLP_RFI_FT);
         else if (strcmp(argv[k], "--primal") == 0)
            glp_set_ipar("spx_use_dual", GLP_NO);
         else if (strcmp(argv[k], "--dual") == 0)
            glp_set_ipar("spx_use_dual", GLP_YES);
         else if (strcmp(argv[k], "--steep") == 0)
            glp_set_ipar("spx_steep", GLP_YES);
         else if (strcmp(argv[k], "--nosteep") == 0)
            glp_set_ipar("spx_steep", GLP_NO);
         else if (strcmp(argv[k], "--relax") == 0)
            glp_set_ipar("spx_relax", GLP_YES);
         else if (strcmp(argv[k], "--norelax") == 0)
            glp_set_ipar("spx_relax", GLP_NO);
         else if (strcmp(argv[k], "--sum") == 0)
            glp_set_ipar("sum_aij", GLP_YES);
         else if (strcmp(argv[k], "--nosum") == 0)
            glp_set_ipar("sum_aij", GLP_NO);
         else if (strcmp(argv[k], "-h") == 0 ||
                  strcmp(argv[k], "--help") == 0)
            display_help(argv[0]);
         else if (strcmp(argv[k], "-v") == 0 ||
                  strcmp(argv[k], "--version") == 0)
            display_version();
         else if (strcmp(argv[k], "--nomip") == 0)
            nomip = 1;
         else if (strcmp(argv[k], "--first") == 0)
            glp_set_ipar("mip_branch", GLP_FIRST);
         else if (strcmp(argv[k], "--last") == 0)
            glp_set_ipar("mip_branch", GLP_LAST);
         else if (strcmp(argv[k], "--drtom") == 0)
            glp_set_ipar("mip_branch", GLP_DRTOM);
         else if (strcmp(argv[k], "--fifo") == 0)
            glp_set_ipar("mip_btrack", GLP_FIFO);
         else if (strcmp(argv[k], "--lifo") == 0)
            glp_set_ipar("mip_btrack", GLP_LIFO);
         else if (strcmp(argv[k], "--bestp") == 0)
            glp_set_ipar("mip_btrack", GLP_BESTP);
         else if (argv[k][0] == '-' ||
                 (argv[k][0] == '-' && argv[k][1] == '-'))
         {  error("Invalid option `%s'; try %s --help", argv[k],
               argv[0]);
            exit(EXIT_FAILURE);
         }
         else
         {  if (dat_fname[0] != '\0')
            {  error("Only one input file allowed");
               exit(EXIT_FAILURE);
            }
            dat_fname = argv[k];
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- main - main routine.
--
-- This main routine is called by the control program and manages the
-- solving process. */

int main(int argc, char *argv[])
{     int option, nc_int, ret;
      clock_t t_start;
      /* initialize GLPK API */
      glp_initialize();
      /* process command line parameters */
      process_cmdline(argc, argv);
      /* read LP problem data from the input file */
      if (dat_fname[0] == '\0')
      {  error("No input file name specified");
         exit(EXIT_FAILURE);
      }
      glp_get_ipar("option", &option);
      if (dat_fmt == 0)
      {  if (glp_read_mps(dat_fname) != 0)
         {  error("Can't read MPS file");
            exit(EXIT_FAILURE);
         }
      }
      else
      {  if (glp_read_lpm(dat_fname) != 0)
         {  error("Can't read LPM file");
            exit(EXIT_FAILURE);
         }
      }
      glp_set_ipar("option", option);
      /* solve LP problem */
      t_start = clock();
      glp_get_ipar("nc_int", &nc_int);
      ret = ((nomip || nc_int == 0) ? glp_simplex() : glp_integer());
      switch (ret)
      {  case 0:
            break;
         case 1:
            error("Can't solve the problem because of errors");
            exit(EXIT_FAILURE);
         case 2:
            error("Optimization terminated because of solver failure");
            break;
         default:
            insist(ret != ret);
      }
      /* display statistics */
      print("Time used: %.1f secs",
         (double)(clock() - t_start) / (double)CLOCKS_PER_SEC);
      print("Memory used: %.1fM (%d bytes)",
         (double)mem_tpeak / (1024.0 * 1024.0), mem_tpeak);
      /* write final solution found by the solver */
      if (out_fname[0] != '\0' && glp_print_sol(out_fname) != 0)
      {  error("Can't write solution");
         exit(EXIT_FAILURE);
      }
      /* terminate GLPK API */
      glp_terminate();
      /* exit to the control program */
      return 0;
}

/* eof */
