/* glpprob/save_lpsol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <float.h>
#include <stdio.h>
#include <string.h>
#include "glpprob.h"

/*----------------------------------------------------------------------
-- save_lpsol - write LP solution to text file using plain format.
--
-- *Synopsis*
--
-- #include "glpprob.h"
-- int save_lpsol(LPSOL *sol, char *fname);
--
-- *Description*
--
-- The save_lpsol routine writes the LP problem basis solution block,
-- which sol points to, to the text file, whose name is the character
-- string fname, using the following plain format:
--
-- 'LPSOL'           \n
-- m n status objval \n
-- tagx valx dx      \n    |
-- . . .                   |-> m+n lines
-- tagx valx dx      \n    |
-- 'EOD'             \n
--
-- where:
-- m      - number of rows (auxiliary variables);
-- n      - number of columns (structural variables);
-- status - status of solution:
--          '?' - solution is undefined;
--          'O' - solution is optimal;
--          'F' - solution is feasible;
--          'I' - solution is infeasible;
--          'N' - problem has no feasible solution;
--          'U' - problem has unbounded solution.
-- objval - value of the objective function;
-- tagx   - status of auxiliary or structural variable:
--          '?' - status is undefined;
--          'B' - basis variable;
--          'L' - non-basis variable on its lower bound;
--          'U' - non-basis variable on its upper bound;
--          'F' - non-basis free variable;
--          'S' - non-basis fixed variable.
-- valx   - value of variable;
-- dx     - reduced cost (marginal value) of variable.
--
-- *Returns*
--
-- If the operation was successful, the save_lpsol routine returns zero.
-- Otherwise the routine returns non-zero. */

int save_lpsol(LPSOL *sol, char *fname)
{     FILE *fp;
      int k;
      print("save_lpsol: writing LP solution block to `%s'...", fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("save_lp: unable to create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* header */
      fprintf(fp, "'LPSOL'\n");
      fprintf(fp, "%d %d '%c' %.*g\n", sol->m, sol->n, sol->status,
         DBL_DIG, sol->objval);
      /* solution */
      for (k = 1; k <= sol->m+sol->n; k++)
      {  fprintf(fp, "'%c' %.*g %.*g\n", sol->tagx[k], DBL_DIG,
            sol->valx[k], DBL_DIG, sol->dx[k]);
      }
      fprintf(fp, "'EOD'\n");
      fflush(fp);
      if (ferror(fp))
      {  error("save_lp: writing error on `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
