/* glpgel/solve_lu.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpgel.h"

/*----------------------------------------------------------------------
-- solve_lu - solve linear system using LU-factorization.
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- void solve_lu(LU *lu, int flag, double x[], double work[]);
--
-- *Description*
--
-- The solve_lu routine obtains solution of the linear system A*x = b
-- or A'*x = b, where A is coefficient matrix, A' is matrix transposed
-- to A, x is dense vector of unknowns, b is dense vector of right-hand
-- sides.
--
-- The parameter lu should define LU-factorization of the coefficient
-- matrix A computed by means of the build_lu routine. If the build_lu
-- reported failure, the solve_lu routine should not be used.
--
-- The parameter flag determines what system will be solved. If flag
-- is 0, the routine obtains solution of A*x = b. Otherwise the routine
-- obtains solution of A'*x = b, where A' is transposed to A. (In the
-- latter case LU-factorization should correspond to A, not to A'!)
--
-- Before a call the array x should contain elements of the vector b
-- in locations x[1], x[2], ..., x[n], where n is order of the system.
-- After a call this array will contain the vector x in the same
-- locations.
--
-- The auxiliary array work should have at least 1+n elements. */

void solve_lu(LU *lu, int flag, double x[], double work[])
{     if (flag == 0)
      {  /* A = P*L*U*Q => inv(A) = Q'*inv(U)*inv(L)*P' */
         iper_vec(work, lu->P, x);
         l_solve(lu->L, work);
         u_solve(lu->U, work);
         iper_vec(x, lu->Q, work);
      }
      else
      {  /* A' = Q'*U'*L'*P' => inv(A') = P*inv(L')*inv(U')*Q */
         per_vec(work, lu->Q, x);
         ut_solve(lu->U, work);
         lt_solve(lu->L, work);
         per_vec(x, lu->P, work);
      }
      return;
}

/* eof */
