/* glpk.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPK_H
#define _GLPK_H

#define GLP_MAX_NAME 255
/* maximal length of item name (not including '\0') */

#define GLP_NO       0     /* false */
#define GLP_YES      1     /* true */

#define GLP_ROW      101   /* item is row */
#define GLP_COL      102   /* item is column */

#define GLP_FR       201   /* free variable:  -inf <  x[k] <  +inf */
#define GLP_LO       202   /* lower bound:    l[k] <= x[k] <  +inf */
#define GLP_UP       203   /* upper bound:    -inf <  x[k] <= u[k] */
#define GLP_DB       204   /* double bound:   l[k] <= x[k] <= u[k] */
#define GLP_FX       205   /* fixed variable: l[k]  = x[k]  = u[k] */

#define GLP_INI      301   /* only initial solution should be found */
#define GLP_ANY      302   /* only feasible solution should be found */
#define GLP_MIN      303   /* objective function should be minimized */
#define GLP_MAX      304   /* objective function should be maximized */

#define GLP_UNDEF    401   /* solution is undefined */
#define GLP_OPT      402   /* solution is optimal */
#define GLP_FEAS     403   /* solution is feasible */
#define GLP_INFEAS   404   /* solution is infeasible */
#define GLP_NOFEAS   405   /* problem has no feasible solution */
#define GLP_UNBND    406   /* problem has unbounded solution */

#define GLP_BS       501   /* basis variable */
#define GLP_NL       502   /* non-basis variable on its lower bound */
#define GLP_NU       503   /* non-basis variable on its upper bound */
#define GLP_NF       504   /* non-basis free variable */
#define GLP_NS       505   /* non-basis fixed variable */

#define GLP_EFI      601   /* EFI */
#define GLP_RFI_BG   602   /* RFI + Bartels & Golub updating tech */
#define GLP_RFI_FT   603   /* RFI + Forrest & Tomlin updating tech */

#define GLP_D_PARS   0x01  /* dump control parameters */
#define GLP_D_ROWS   0x10  /* dump rows */
#define GLP_D_RMAT   0x20  /* dump coefficients for each row */
#define GLP_D_COLS   0x40  /* dump columns */
#define GLP_D_CMAT   0x80  /* dump coefficients for each column */
#define GLP_D_ALL    0xFF  /* dump all information */

extern int glp_check_name(char *name);
/* check symbolic name for correctness */

extern int glp_create_item(int what, char *name);
/* create new row or column */

extern int glp_dump_ws(char *fname, int what);
/* dump GLPK API workspace */

extern int glp_find_item(int what, char *name);
/* find row or column by name */

extern int glp_first_coef(int what);
/* find the first constraint coefficient */

extern int glp_first_item(int what);
/* find the first row or column */

extern int glp_get_activity(int what, int *tagx, double *valx,
      double *dx);
/* get activity of the current row or column */

extern int glp_get_bounds(int what, int *type, double *lb, double *ub);
/* get bounds of the current row or column */

extern int glp_get_coef(double *val);
/* get the current constraint coefficient */

extern int glp_get_cpar(char *name, char *val);
/* get value of text control parameter */

extern int glp_get_kind(int what, int *kind);
/* get kind of the current row or column */

extern int glp_get_ipar(char *name, int *val);
/* get value of integer control parameter */

extern int glp_get_name(int what, char *name);
/* get name of the current row or column */

extern int glp_get_rpar(char *name, double *val);
/* get value of real control parameter */

extern int glp_get_seqn(int what, int *seqn);
/* get seqn attribute of the current row or column */

extern int glp_initialize(void);
/* initialize GLPK application program interface */

extern int glp_interior(void);
/* solve problem by means of the interior point method */

extern int glp_new_coef(double val);
/* create new constraint coefficient */

extern int glp_next_coef(int what);
/* find the next constraint coefficient */

extern int glp_next_item(int what);
/* find the next row or column */

extern int glp_print_sol(char *fname);
/* write problem solution using printable format */

extern int glp_read_mps(char *fname);
/* read problem data using MPS format */

extern int glp_set_activity(int what, int tagx, double valx, double dx);
/* set activity of the current row or column */

extern int glp_set_bounds(int what, int type, double lb, double ub);
/* set bounds of the current row or column */

extern int glp_set_cpar(char *name, char *val);
/* set value of text control parameter */

extern int glp_set_ipar(char *name, int val);
/* set value of integer control parameter */

extern int glp_set_kind(int what, int kind);
/* set kind of the current row or column */

extern int glp_set_name(int what, char *name);
/* change name of the current row or column */

extern int glp_set_rpar(char *name, double val);
/* set value of real control parameter */

extern int glp_set_seqn(int what, int seqn);
/* set seqn attribute of the current row or column */

extern int glp_simplex(void);
/* solve problem by means of the revised simplex method */

extern int glp_terminate(void);
/* terminate GLPK application program interface */

extern int glp_write_mps(char *fname);
/* write problem data using MPS format */

#endif

/* eof */
