/* glpapi/glp_simplex.c */

/*** WARNING: THIS ROUTINE IS UNDER RECONSTRUCTION ***/

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <time.h>
#include "glpk.h"
#include "glprsm.h"

static int rsm_dir;
/* optimization direction flag:
   '-' - objective function should be minimized
   '+' - objective function should be maximized */

static int rsm_obj;
/* row (auxiliary variable) number that defines the objective function
   (1 <= obj <= m); obj may be equal to zero to indicate that the
   objective function is identically equal to zero */

static double *rsm_R; /* double R[1+m]; */
/* row scaling diagonal matrix; R[0] is not used; R[1], ..., R[m] are
   diagonal elements of the matrix R */

static double *rsm_S; /* double S[1+n]; */
/* column scaling diagonal matrix; S[0] is not used; S[1], ..., S[n] are
   diagonal elements of the matrix S */

/* the matrix of constraint coefficients for the scaled problem has the
   form R*A*S, where A is the original (unscaled) matrix; if the scaling
   is not used, R and S are unity matrices */

static double *rsm_bbar; /* double bbar[1+m]; */
/* bbar[0] is not used; bbar[i] is the current value of the basis
   variable xB[i] (1 <= i <= m) */

static double *rsm_cost; /* double cost[1+m+n]; */
/* cost[0] is not used; cost[k] is a coefficient of the auxiliary
   objective function at the variable x[k] (1 <= k <= m+n); this
   objective function is to be always minimized */

static double *rsm_pi; /* double pi[1+m]; */
/* pi[0] is not used; pi[i] is a simplex multiplier corresponding to the
   i-th equality constraint (1 <= i <= m) */

static double *rsm_cbar; /* double cbar[1+n]; */
/* cbar[0] is not used; cbar[j] is the reduced cost (marginal) of the
   non-basis variable xN[j] (1 <= j <= n); they correspond to the
   auxiliary objective function */

static double *rsm_gvec; /* double gvec[1+n]; */
/* gvec[0] is not used; gvec[j] is the weight coefficient for the j-th
   column (1 <= j <= n) of the current simplex table; these weights are
   used for the primal steepest edge pricing (in order to use the
   standard "textbook" pricing these weights should be set to 1) */

static double *rsm_dvec; /* double dvec[1+m]; */
/* dvec[0] is not used; dvec[i] is the weight coefficient for the i-th
   row (1 <= i <= m) of the current simplex table; these weights are
   used for the dual steepest edge pricing (in order to use the standard
   "textbook" pricing these weights should be set to 1) */

static int rsm_p;
/* number of the basis variable xB[p] which has been chosen to leave the
   basis (if p = 0 then choice is impossible; if p < 0 then the chosen
   non-basis variable xN[q] should be moved from the current bound to
   the opposite one) */

static int rsm_tagp;
/* it is a tag which should be assigned to the basis variable xB[p] when
   this variable will have left the basis */

static double *rsm_bp; /* double bp[1+m]; */
/* p-th row of inv(B); bp[0] is not used; elements are placed in
   locations bp[1], ..., bp[m] */

static double *rsm_ap; /* double ap[1+n]; */
/* ap[0] is not used; ap[j] is the coefficient at the non-basis variable
   xN[j] in p-th row of the current simplex table that corresponds to
   the basis variable xB[p] (1 <= p <= m) */

static int rsm_q;
/* number of the non-basis variable xN[q] which has been chosen to enter
   the basis (if q = 0 then choice is impossible) */

static double *rsm_aq; /* double aq[1+m]; */
/* aq[0] is not used; aq[i] is the coefficient at the non-basis variable
   xN[q] in i-th row of the current simplex table that corresponds to
   the basis variable xB[i] (1 <= i <= m) */

static int rsm_prim_steep;
/* entering variable option (primal simplex):
   0 - use "textbook" technique
   1 - use steepest edge technique (this means updating gvec on each
       primal simplex iteration) */

static int rsm_prim_relax;
/* leaving variable option (primal simplex):
   0 - use "textbook" ratio test
   1 - use a technique proposed by P.Harris */

static int rsm_dual_steep;
/* leaving variable option (dual simplex):
   0 - use "textbook" technique
   1 - use steepest edge technique (this means updating dvec on each
       dual simplex iteration) */

static int rsm_dual_relax;
/* entering variable option (dual simplex):
   0 - use "textbook" ratio test
   1 - use a technique proposed by P.Harris */

static double rsm_tol_bnd;
/* tolerance used for checking primal feasibility */

static double rsm_tol_dj;
/* tolerance used for checking dual feasibility */

static double rsm_tol_piv;
/* tolerance used for pivoting */

static int rsm_check_basis;
/* if this flag is set, the routines check basis segment data structures
   for correctness on each simplex iteration (only for debugging) */

static int rsm_check_gvec;
/* if this flag is set, the routines check the accuracy of the updated
   weights gvec comparing them with exact values on each simplex
   iteration (only for debugging) */

static int rsm_check_dvec;
/* if this flag is set, the routines check the accuracy of the updated
   weights dvec comparing them with exact values on each simplex
   iteration (only for debugging) */

static int rsm_fin_out;
/* if this flag is set, the searching routines display final solution at
   the end of the stage */

static clock_t rsm_t_last;
/* most recent time at which visual information was displayed */

static int rsm_defect(RSM *rsm);
/* determine defect of current basis solution */

static int rsm_dual_opt(RSM *rsm);
/* find optimal solution by dual simplex method */

static int rsm_find_feas(RSM *rsm);
/* find primal feasible solution */

static double rsm_infsum(RSM *rsm);
/* compute sum of infeasibilities */

static double rsm_objval(RSM *rsm);
/* obtain current value of objective function */

static int rsm_primal_opt(RSM *rsm);
/* find optimal solution by primal simplex method */

static void rsm_scaling(RSM *rsm);
/* scale linear programming problem data */

/*----------------------------------------------------------------------
-- glp_simplex - solve problem by means of the revised simplex method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_simplex(void);
--
-- *Description*
--
-- The glp_simplex routine is a LP problem solver based on the two-phase
-- revised simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_simplex routine performs the following actions:
--
-- preparing internal data structures;
-- searching for feasible basis solution (pahse I);
-- searching for optimal basis solution (phase II);
-- storing the computed solution to the workspace.
--
-- Since large scale problems may take a long time, the glp_simplex
-- routine reports some information about the current status of the
-- searching. This information is sent to stdout approximately once per
-- second and has the following format:
--
--    *nnn:   objval = xxx   infsum = yyy (ddd)
--
-- where nnn is an iteration count, xxx is the current value of the
-- objective function (which is unscaled and has correct sign), yyy is
-- the current value of sum of infeasibilities (which is scaled, so may
-- be used only for visual estimating), ddd is the current number of
-- basis fixed variables. If the asterisk (*) precedes to nnn, the
-- solver is searching for optimal so-lution (i.e. the feasible solution
-- has been found yet), otherwise the solver is searching for some
-- feasible solution.
--
-- Note that the glp_simplex solver is not perfect. Although it has been
-- successfully tested on a wide set of LP problems, there are hard
-- problems, which can't be solved by this solver.
--
-- *Control parameters*
--
-- The behavior of the glp_simplex routine depends on a number of
-- control parameters, which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- are fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note that, for example, if the
--     problem has no feasible solution, the code returned by the solver
--     will be equal to zero);
-- 1 - it's not possible to start solving the problem because of
--     incorrect data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

static RSM *rsm;
/* pointer to the revised simplex method common block */

static void store_sol(int status);
/* store basis solution to the workspace */

int glp_simplex(void)
{     /* control parameters */
      int obj_dir, scale, form, relax, use_dual, steep, sum_aij;
      double tol_aij, tol_bnd, tol_dj, tol_piv;
      char obj_row[GLP_MAX_NAME+1];
      /* local variables */
      int m, n, i, j, k, kase, ret;
      print("Solving LP problem...");
      rsm = NULL;
      /* extract relevant control parameters */
      insist(glp_get_ipar("obj_dir", &obj_dir) == 0);
      insist(glp_get_cpar("obj_row", obj_row) == 0);
      insist(glp_get_ipar("scale", &scale) == 0);
      insist(glp_get_ipar("spx_form", &form) == 0);
      insist(glp_get_ipar("spx_relax", &relax) == 0);
      insist(glp_get_ipar("spx_steep", &steep) == 0);
      insist(glp_get_ipar("spx_use_dual", &use_dual) == 0);
      insist(glp_get_ipar("sum_aij", &sum_aij) == 0);
      insist(glp_get_rpar("tol_aij", &tol_aij) == 0);
      insist(glp_get_rpar("tol_bnd", &tol_bnd) == 0);
      insist(glp_get_rpar("tol_dj", &tol_dj) == 0);
      insist(glp_get_rpar("tol_piv", &tol_piv) == 0);
      /* renumber and count rows */
      m = 0;
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW)) glp_set_seqn(GLP_ROW, ++m);
      if (m == 0)
      {  error("glp_simplex: problem has no rows");
         ret = 1;
         goto done;
      }
      /* renumber and count columns */
      n = 0;
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL)) glp_set_seqn(GLP_COL, ++n);
      if (n == 0)
      {  error("glp_simplex: problem has no columns");
         ret = 1;
         goto done;
      }
      /* create revised simplex method common block */
      rsm = umalloc(sizeof(RSM));
      /* --- problem data segment --- */
      rsm->m = m;
      rsm->n = n;
      rsm->type = ucalloc(1+m+n, sizeof(int));
      rsm->lb = ucalloc(1+m+n, sizeof(double));
      rsm->ub = ucalloc(1+m+n, sizeof(double));
      /* determine types and bounds of variables */
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int seqn, type;
            double lb, ub;
            glp_get_seqn(what, &seqn);
            glp_get_bounds(what, &type, &lb, &ub);
            k = (kase == 0 ? 0 : m) + seqn;
            insist(1 <= k && k <= m+n);
            switch (type)
            {  case GLP_FR:
                  rsm->type[k] = 'F';
                  rsm->lb[k] = rsm->ub[k] = 0.0;
                  break;
               case GLP_LO:
                  rsm->type[k] = 'L';
                  rsm->lb[k] = lb, rsm->ub[k] = 0.0;
                  break;
               case GLP_UP:
                  rsm->type[k] = 'U';
                  rsm->lb[k] = 0.0, rsm->ub[k] = ub;
                  break;
               case GLP_DB:
                  rsm->type[k] = 'D';
                  rsm->lb[k] = lb, rsm->ub[k] = ub;
                  insist(lb < ub);
                  break;
               case GLP_FX:
                  rsm->type[k] = 'S';
                  rsm->lb[k] = rsm->ub[k] = lb;
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* create expanded constraint matrix A~ = (I | -A) */
      rsm->A = create_mat(m, m+n);
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
      {  int ret;
         glp_get_seqn(GLP_ROW, &i);
         /* coefficient at auxiliary variable */
         new_elem(rsm->A, i, i, 1.0);
         /* coefficients at structural variables */
         for (ret = glp_first_coef(GLP_ROW); ret == 0;
              ret = glp_next_coef(GLP_ROW))
         {  double val;
            glp_get_seqn(GLP_COL, &j);
            glp_get_coef(&val);
            if (val == 0.0 || fabs(val) < tol_aij) continue;
            new_elem(rsm->A, i, m+j, -val);
         }
      }
      /* objective function */
      rsm_dir = (obj_dir != GLP_MAX ? '-' : '+');
      if (obj_row[0] == '\0')
         rsm_obj = 0;
      else
      {  insist(glp_find_item(GLP_ROW, obj_row) == 0);
         insist(glp_get_seqn(GLP_ROW, &rsm_obj) == 0);
         insist(1 <= rsm_obj && rsm_obj <= m);
      }
      /* --- problem scaling segment --- */
      rsm_R = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) rsm_R[i] = 1.0;
      rsm_S = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) rsm_S[j] = 1.0;
      /* --- basis solution segment --- */
      rsm->posx = ucalloc(1+m+n, sizeof(int));
      rsm->indb = ucalloc(1+m, sizeof(int));
      rsm->indn = ucalloc(1+n, sizeof(int));
      rsm->tagn = ucalloc(1+n, sizeof(int));
      /* all auxiliary variables are basis */
      for (i = 1; i <= m; i++)
      {  k = i; /* x[k] = xB[i] */
         rsm->posx[k] = +i;
         rsm->indb[i] =  k;
      }
      /* all structural variables are non-basis */
      for (j = 1; j <= n; j++)
      {  k = m+j; /* x[k] = xN[j] */
         rsm->posx[k] = -j;
         rsm->indn[j] =  k;
         switch (rsm->type[k])
         {  case 'F':
               rsm->tagn[j] = 'F';
               break;
            case 'L':
               rsm->tagn[j] = 'L';
               break;
            case 'U':
               rsm->tagn[j] = 'U';
               break;
            case 'D':
               rsm->tagn[j] = 'L';
               break;
            case 'S':
               rsm->tagn[j] = 'S';
               break;
            default:
               insist(rsm->type[k] != rsm->type[k]);
         }
      }
      /* --- simplex method segment --- */
      switch (form)
      {  case GLP_EFI:
            /* EFI */
            rsm->efi = create_efi(m);
            rsm->rfi = NULL;
            break;
         case GLP_RFI_BG:
            /* RFI + Bartels & Golub updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_BG;
            break;
         case GLP_RFI_FT:
            /* RFI + Forrest & Tomlin updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_FT;
            break;
         default:
            insist(form != form);
      }
      rsm_bbar = ucalloc(1+m, sizeof(double));
      rsm_cost = ucalloc(1+m+n, sizeof(double));
      rsm_pi = ucalloc(1+m, sizeof(double));
      rsm_cbar = ucalloc(1+n, sizeof(double));
      rsm_dvec = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) rsm_dvec[i] = 1.0;
      rsm_gvec = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) rsm_gvec[j] = 1.0;
      rsm_p = 0;
      rsm_tagp = 0;
      rsm_bp = ucalloc(1+m, sizeof(double));
      rsm_ap = ucalloc(1+n, sizeof(double));
      rsm_q = 0;
      rsm_aq = ucalloc(1+m, sizeof(double));
      /* --- control parameters segment --- */
      rsm_prim_steep = steep;
      rsm_prim_relax = relax;
      rsm_dual_steep = steep;
      rsm_dual_relax = relax;
      rsm_tol_bnd = tol_bnd;
      rsm_tol_dj = tol_dj;
      rsm_tol_piv = tol_piv;
      rsm_check_basis = 0;
      rsm_check_gvec = 0;
      rsm_check_dvec = 0;
      rsm_fin_out = 1;
      /* --- statistics segment --- */
      rsm->iter = 0;
      rsm_t_last = 0;
      /* the following fragment sums multiplets, stores each sum to the
         first element from the corresponding multiplet, and nullifies
         other elements from the same multiplet (of course, all singlets
         remain unchanged) */
      sort_mat(rsm->A);
      for (i = 1; i <= m; i++)
      {  ELEM *ee = NULL, *e;
         /* ee points to the first element of multiplet; e points to the
            current element */
         for (e = rsm->A->row[i]; e != NULL; e = e->row)
         {  if (ee == NULL || ee->j < e->j)
               ee = e;
            else
            {  insist(ee != e && ee->j == e->j);
               if (!sum_aij)
               {  print("glp_simplex: constraint matrix has multiplets")
                     ;
                  ret = 1;
                  goto done;
               }
               ee->val += e->val, e->val = 0.0;
            }
         }
      }
      /* now all zero elements and tiny elements that might appear as
         the result of summation should be removed from the constraint
         matrix */
      scrape_mat(rsm->A, tol_aij);
      /* scale the problem (if required) */
      if (scale) rsm_scaling(rsm);
      /* initialize weights (if steepest edge technique is used) */
      if (rsm_prim_steep) init_gvec(rsm, rsm_gvec);
      if (rsm_dual_steep) init_dvec(rsm, rsm_dvec);
      /* compute the initial basis solution (if required) */
      if (obj_dir == GLP_INI)
      {  /* the initial basis matrix is a unity matrix */
         insist(invert_b(rsm) == 0);
         /* compute initial values of basis variables */
         eval_bbar(rsm, rsm_bbar);
         /* check initial solution for primal feasibility */
         if (check_bbar(rsm, rsm_bbar, rsm_tol_bnd) == 0)
         {  print("Initial solution is FEASIBLE");
            store_sol(GLP_FEAS);
         }
         else
         {  print("Initial solution is INFEASIBLE");
            store_sol(GLP_INFEAS);
         }
         /* no more optimization */
         ret = 0;
         goto done;
      }
bfs:  /* searching for feasible solution */
      print("Searching for feasible solution...");
      /* reinvert the current basis matrix (this is usually needed to
         improve accuracy of the representation) */
      if (invert_b(rsm))
sing: {  error("Numerical problems with basis matrix");
         error("Sorry, basis recovery procedure not implemented");
         ret = 2;
         goto done;
      }
      if (!use_dual)
      {  /* use primal simplex method and implicit artificial variables
            technique */
         ret = rsm_find_feas(rsm);
      }
      else
      {  /* use dual simplex method and special objective function to
            make the initial basis be dual feasible */
         for (k = 1; k <= m+n; k++) rsm_cost[k] = 0.0;
         for (j = 1; j <= n; j++)
         {  k = rsm->indn[j]; /* x[k] = xN[j] */
            switch (rsm->tagn[j])
            {  case 'F': rsm_cost[k] =  0.0; break;
               case 'L': rsm_cost[k] = +1.0; break;
               case 'U': rsm_cost[k] = -1.0; break;
               case 'S': rsm_cost[k] =  0.0; break;
               default: insist(rsm->tagn[j] != rsm->tagn[j]);
            }
         }
         ret = rsm_dual_opt(rsm);
      }
      /* analyze return code */
      switch (ret)
      {  case 0:
            /* feasible solution found */
            store_sol(GLP_FEAS);
            if (obj_dir != GLP_ANY) break;
            print("FEASIBLE SOLUTION FOUND");
            ret = 0;
            goto done;
         case 1:
            /* problem has no feasible solution */
            print("PROBLEM HAS NO FEASIBLE SOLUTION");
            store_sol(GLP_NOFEAS);
            ret = 0;
            goto done;
         case 2:
            /* numerical stability lost */
            print("Numerical stability lost");
            goto bfs;
         case 3:
            /* numerical problems with basis matrix */
            goto sing;
         default:
            insist(ret != ret);
      }
      /* searching for optimal solution */
      print("Searching for optimal solution...");
      {  /* set the original objective function (actually the objective
            function is minimized) */
         int k;
         for (k = 1; k <= m+n; k++) rsm_cost[k] = 0.0;
         if (rsm_obj != 0)
            rsm_cost[rsm_obj] = (rsm_dir == '-' ? +1.0 : -1.0);
      }
      ret = rsm_primal_opt(rsm);
      /* analyze returns code */
      switch (ret)
      {  case 0:
            /* optimal solution found */
            print("OPTIMAL SOLUTION FOUND");
            store_sol(GLP_OPT);
            ret = 0;
            break;
         case 1:
            /* problem has unbounded solution */
            print("PROBLEM HAS UNBOUNDED SOLUTION");
            store_sol(GLP_UNBND);
            ret = 0;
            break;
         case 2:
            /* numerical stability lost */
            print("Numerical stability lost");
            goto bfs;
         case 3:
            /* numerical problems with basis matrix */
            goto sing;
         default:
            insist(ret != ret);
      }
done: /* delete revised simplex method common block */
      if (rsm != NULL)
      {  ufree(rsm->type);
         ufree(rsm->lb);
         ufree(rsm->ub);
         delete_mat(rsm->A);
         ufree(rsm_R);
         ufree(rsm_S);
         ufree(rsm->posx);
         ufree(rsm->indb);
         ufree(rsm->indn);
         ufree(rsm->tagn);
         if (rsm->efi != NULL) delete_efi(rsm->efi);
         if (rsm->rfi != NULL) delete_rfi(rsm->rfi);
         ufree(rsm_bbar);
         ufree(rsm_cost);
         ufree(rsm_pi);
         ufree(rsm_cbar);
         ufree(rsm_dvec);
         ufree(rsm_gvec);
         ufree(rsm_bp);
         ufree(rsm_ap);
         ufree(rsm_aq);
         ufree(rsm);
      }
      /* return to the application program */
      return ret;
}

/*----------------------------------------------------------------------
-- store_sol - store basis solution to the workspace.
--
-- This routine stores all information related to the basis solution
-- found by the solver to the workspace. The parameter status specifies
-- the status of the solution. */

static void store_sol(int status)
{     int m = rsm->m, n = rsm->n, round, i, j, k, kase, ret;
      double tol_bnd, tol_dj;
      insist(glp_get_ipar("round", &round) == 0);
      insist(glp_get_rpar("tol_bnd", &tol_bnd) == 0);
      insist(glp_get_rpar("tol_dj", &tol_dj) == 0);
      /* set the solution status */
      insist(glp_set_ipar("status", status) == 0);
      /* set the original objective function */
      for (k = 1; k <= m+n; k++) rsm_cost[k] = 0.0;
      if (rsm_obj != 0) rsm_cost[rsm_obj] = 1.0;
      /* compute values of all basis variables and reduced costs of all
         non-basis variables */
      eval_bbar(rsm, rsm_bbar);
      eval_pi(rsm, rsm_cost, rsm_pi);
      eval_cbar(rsm, rsm_cost, rsm_pi, rsm_cbar);
      /* store solution back to the workspace */
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int seqn, type, tagx;
            double lb, ub, valx, dx;
            glp_get_seqn(what, &seqn);
            glp_get_bounds(what, &type, &lb, &ub);
            k = (kase == 0 ? 0 : m) + seqn;
            insist(1 <= k && k <= m+n);
            if (rsm->posx[k] > 0)
            {  /* basis variable */
               i = +rsm->posx[k]; /* x[k] = xB[i] */
               tagx = GLP_BS;
               if (fabs(rsm_bbar[i]) < tol_bnd)
                  valx = 0.0;
               else
                  valx = rsm_bbar[i];
               dx = 0.0;
            }
            else
            {  /* non-basis variable */
               j = -rsm->posx[k]; /* x[k] = xN[j] */
               switch (rsm->tagn[j])
               {  case 'L': tagx = GLP_NL; break;
                  case 'U': tagx = GLP_NU; break;
                  case 'F': tagx = GLP_NF; break;
                  case 'S': tagx = GLP_NS; break;
                  default: insist(rsm->tagn[j] != rsm->tagn[j]);
               }
               valx = eval_xn(rsm, j);
               if (fabs(rsm_cbar[j]) < tol_dj)
                  dx = 0.0;
               else
                  dx = rsm_cbar[j];
            }
            /* unscale values and reduced costs */
            if (k <= m)
            {  /* auxiliary variable */
               valx /= rsm_R[k];
               dx   *= rsm_R[k];
            }
            else
            {  /* structural variable */
               valx *= rsm_S[k-m];
               dx   /= rsm_S[k-m];
            }
            /* all reduced costs should be divided by the scale factor
               R[obj], because cost[obj] = 1 corresponds to the *scaled*
               problem */
            if (rsm_obj != 0) dx /= rsm_R[rsm_obj];
            /* assign tagx, valx, and dx to x[k] */
            insist(glp_set_activity(what, tagx, valx, dx) == 0);
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- rsm_scaling - scale linear programming problem data.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_scaling(RSM *rsm);
--
-- *Description*
--
-- The rsm_scaling routine performs scaling of the linear programming
-- problem data. At first the routine computes diagonal scaling matrices
-- R and S. After that the routine uses the matrices R and S to scale
-- lower and upper bounds of variables and constraint matrix. New scaled
-- data are stored in the structure RSM which rsm points to instead old
-- unscaled data.
--
-- In order to scale constraint matrix the rsm_scaling routine uses
-- geometric mean scaling that follows equilibration scaling.
--
-- Let the formulation of the original (unscaled) problem has the form:
--
--    xR = A * xS       (equality constraints)
--    lR <= xR <= uR    (bounds of auxiliary variables)
--    lS <= xS <= uS    (bounds of structural variables)
--
-- As a result of scaling the original constraint matrix A is replaced
-- by the scaled matrix A' = R*A*S, where R and S are diagonal scaling
-- matrices. Thus, the formulation of the scaled problem will have the
-- form:
--
--    xR' = A' * xS'
--    lR' <= xR' <= uR'
--    lS' <= xS' <= uS'
--
-- where:
--
--    A'  = R * A * S
--    xR' = R * xR
--    lR' = R * lR
--    uR' = R * uR
--    xS' = inv(S) * xS
--    lS' = inv(S) * lS
--    uS' = inv(S) * uS
--
-- The scaling matrices R and S are saved in the common block and used
-- later in order to descale the computed solution. */

static void rsm_scaling(RSM *rsm)
{     int m = rsm->m, n = rsm->n, i, j, k;
      /* compute scaling matrices */
      {  MAT *A;
         A = create_mat(m, n);
         submatrix(A, rsm->A, 1, m, m+1, m+n);
         gm_scaling(A, rsm_R, rsm_S, 0, 0.01, 20);
         eq_scaling(A, rsm_R, rsm_S, 0);
         delete_mat(A);
      }
      /* scale bounds of auxiliary variables */
      for (i = 1; i <= m; i++)
      {  k = i;
         rsm->lb[k] *= rsm_R[i];
         rsm->ub[k] *= rsm_R[i];
      }
      /* scale bound of structural variables */
      for (j = 1; j <= n; j++)
      {  k = m+j;
         rsm->lb[k] /= rsm_S[j];
         rsm->ub[k] /= rsm_S[j];
      }
      /* scale matrix of constraint coefficients */
      for (j = 1; j <= n; j++)
      {  ELEM *e;
         k = m+j;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            e->val *= (rsm_R[e->i] * rsm_S[j]);
      }
      return;
}

/*----------------------------------------------------------------------
-- rsm_objval - obtain current value of objective function.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_objval(RSM *rsm);
--
-- *Returns*
--
-- The rsm_objval routine returns the current value of the original
-- objective function which is not scaled and has correct sign. */

static double rsm_objval(RSM *rsm)
{     int obj = rsm_obj;
      double fun;
      if (obj == 0)
         fun = 0.0;
      else
      {  if (rsm->posx[obj] > 0)
            fun = rsm_bbar[+rsm->posx[obj]];
         else
            fun = eval_xn(rsm, -rsm->posx[obj]);
         fun /= rsm_R[obj];
      }
      return fun;
}

/*----------------------------------------------------------------------
-- rsm_infsum - compute sum of infeasibilities.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_infsum(RSM *rsm);
--
-- *Returns*
--
-- The routine returns the sum of infeasibilities for the current basis
-- solution. Should note that this sum is scaled, so it may be used only
-- for visual estimating. */

static double rsm_infsum(RSM *rsm)
{     int m = rsm->m, i, k;
      double sum = 0.0;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'L' ||
             rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
            if (rsm_bbar[i] < rsm->lb[k])
               sum += (rsm->lb[k] - rsm_bbar[i]);
         if (rsm->type[k] == 'U' ||
             rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
            if (rsm_bbar[i] > rsm->ub[k])
               sum += (rsm_bbar[i] - rsm->ub[k]);
      }
      return sum;
}

/*----------------------------------------------------------------------
-- rsm_defect - determine defect of current basis solution.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_defect(RSM *rsm);
--
-- *Description*
--
-- The rsm_defect routine returns the defect of the current basis
-- solution, which is number of fixed basis variables (should note that
-- it is not a defect of the basis matrix!). */

static int rsm_defect(RSM *rsm)
{     int m = rsm->m, i, def = 0;
      for (i = 1; i <= m; i++)
         if (rsm->type[rsm->indb[i]] == 'S') def++;
      return def;
}

/*----------------------------------------------------------------------
-- rsm_find_feas - find primal feasible solution.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_find_feas(RSM *rsm);
--
-- *Description*
--
-- The rsm_find_feas routine searches for primal feasible solution using
-- primal simplex method and implicit artificial variables technique.
--
-- At first the routine computes current values of the basis variables
-- and replaces basis variables, which violate their bounds, by implicit
-- artificial variables in order to construct feasible basis solution:
--
-- if bbar[i] < lB[i] - eps, i.e. if the basis variable xB[i] violates
-- its lower bound, the routine replaces the constraint xB[i] >= lB[i]
-- by constraint xB[i] <= lB[i] introducing implicit artificial variable
-- which satisfies its upper (!) bound;
--
-- if bbar[i] > ub[i] + eps, i.e. if the basis variable xB[i] violates
-- its upper bound, the routine replaces the constraint xB[i] <= uB[i]
-- by constraint xB[i] >= uB[i] introducing implicit artificial variable
-- which satisfies its lower (!) bound.
--
-- In both cases eps = tol*max(|lb[i]|,1) or eps = tol*max(|ub[i]|,1)
-- (depending on what bound is checked), where tol = 0.30*tol_bnd.
--
-- Should note that actually no new variables are introduced to the
-- problem. The routine just replaces types and values of bounds by new
-- ones.
--
-- Implicit artificial variables correspond to resduals. Therefore the
-- goal is to turn out all implicit variables from the basis that allows
-- to eliminate corresponding residuals. To do this the routine uses the
-- special objective function. If there is the implicit artificial
-- variable xB[i] >= lB[i], the routine sets its coefficient to +1 in
-- order to minimize the corresponding residual from the point of view
-- of the original problem. Analogously, if there is the implicit
-- artificial variable xB[i] <= uB[i], the routine sets its coefficient
-- to -1 again in order to minimize the corresponding residual. Other
-- coefficient are set to zero.
--
-- Should note that when some artificial variable becomes non-basis,
-- the corresponding residual vanishes. Hence the artificial objective
-- function is changed on each iteration of the simplex method.
--
-- Let the basis variable xB[p] leaves the basis. If it is the implicit
-- artificial variable (that recognized by changed type and bounds of
-- this variable), the routine restores its type and bounds to original
-- ones, because in the adjacent basis the corresponding residual will
-- be zero. 
--
-- In the case of degeneracy all implicit artificial variables may
-- become zero being basis variables. In this case the routine doesn't
-- wait these variables to leave the basis and stops the searching,
-- since such basis solution may be considered as feasible.
--
-- In any case (whether feasible solution was found or not) the routine
-- restores original types and bounds of all variables after the search
-- was finished.
--
-- *Returns*
--
-- The rsm_find_feas routine returns one of the following codes:
--
-- 0 - primal feasible solution found;
-- 1 - problem has no feasible solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static int *orig_type;
/* original types of variables */

static double *orig_lb, *orig_ub;
/* original bounds of variables */

static void display1(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      if (need || rsm_t_last == 0 ||
          clock() - rsm_t_last > CLOCKS_PER_SEC)
      {  int *type, def;
         double *lb, *ub, val, sum;
         type = rsm->type, lb = rsm->lb, ub = rsm->ub;
         rsm->type = orig_type;
         rsm->lb = orig_lb, rsm->ub = orig_ub;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         rsm->type = type, rsm->lb = lb, rsm->ub = ub;
         print(" %6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm_t_last = clock();
      }
      return;
}

static int rsm_find_feas(RSM *rsm)
{     int m = rsm->m, n = rsm->n, i, k, ret;
      double *y, *w, tol;
      /* allocate working arrays */
      orig_type = ucalloc(1+m+n, sizeof(int));
      orig_lb = ucalloc(1+m+n, sizeof(double));
      orig_ub = ucalloc(1+m+n, sizeof(double));
      if (rsm_prim_steep)
      {  y = ucalloc(1+m, sizeof(double));
         w = ucalloc(1+m, sizeof(double));
      }
      /* save original types and bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  orig_type[k] = rsm->type[k];
         orig_lb[k] = rsm->lb[k];
         orig_ub[k] = rsm->ub[k];
      }
      /* compute current values of basis variables */
      eval_bbar(rsm, rsm_bbar);
      /* make artificial feasible basis solution */
      tol = 0.30 * rsm_tol_bnd;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'L' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  if (check_rr(rsm_bbar[i], rsm->lb[k], tol) == -2)
            {  /* xB[i] violates its lower bound */
               rsm->type[k] = 'U';
               rsm->lb[k] = 0.0;
               rsm->ub[k] = orig_lb[k];
               continue;
            }
         }
         if (rsm->type[k] == 'U' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  if (check_rr(rsm_bbar[i], rsm->ub[k], tol) == +2)
            {  /* xB[i] violates its upper bound */
               rsm->type[k] = 'L';
               rsm->lb[k] = orig_ub[k];
               rsm->ub[k] = 0.0;
               continue;
            }
         }
      }
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         eval_bbar(rsm, rsm_bbar);
         /* display information about current basis solution */
         display1(0);
         /* check current solution for numerical stability */
         if (check_bbar(rsm, rsm_bbar, rsm_tol_bnd))
         {  /* numerical stability lost */
            if (rsm_fin_out) display1(1);
            ret = 2;
            break;
         }
         /* check current solution for primal feasibility */
         {  int *type; double *lb, *ub;
            type = rsm->type, lb = rsm->lb, ub = rsm->ub;
            rsm->type = orig_type;
            rsm->lb = orig_lb, rsm->ub = orig_ub;
            ret = check_bbar(rsm, rsm_bbar, 0.30 * rsm_tol_bnd);
            rsm->type = type, rsm->lb = lb, rsm->ub = ub;
            if (ret == 0)
            {  /* the current basis is primal feasible */
               if (rsm_fin_out) display1(1);
               break;
            }
         }
         /* construct auxiliary objective function */
         for (k = 1; k <= m+n; k++)
         {  rsm_cost[k] = 0.0;
            if (rsm->type[k] == orig_type[k]) continue;
            /* if type[k] differs from orig_type[k], x[k] is implicit
               artificial variable (which should be basis variable!) */
            insist(rsm->posx[k] > 0);
            if (rsm->type[k] == 'L')
            {  /* x[k] should be decreased, since actually x[k] violates
                  its upper bound */
               rsm_cost[k] = +1.0;
            }
            else if (rsm->type[k] == 'U')
            {  /* x[k] should be increased, since actually x[k] violates
                  its lower bound */
               rsm_cost[k] = -1.0;
            }
            else
               insist(rsm->type[k] != rsm->type[k]);
         }
         /* compute simplex multipliers */
         eval_pi(rsm, rsm_cost, rsm_pi);
         /* compute reduced costs of non-basis variables */
         eval_cbar(rsm, rsm_cost, rsm_pi, rsm_cbar);
         /* choose non-basis variable xN[q] */
         rsm_q = pivot_col(rsm, rsm_cost, rsm_cbar, rsm_gvec,
            rsm_tol_dj);
         if (rsm_q == 0)
         {  /* problem has no feasible solution */
            if (rsm_fin_out) display1(1);
            ret = 1;
            break;
         }
         /* compute pivot column of simplex table */
         eval_col(rsm, rsm_q, rsm_aq, 1);
         /* choose basis variable xB[p] */
         if (!rsm_prim_relax)
         {  /* use standard "textbook" ratio test */
            rsm_p = pivot_row(rsm, rsm_q, rsm_cbar[rsm_q] > 0.0,
               rsm_aq, rsm_bbar, &rsm_tagp, rsm_tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_p = harris_row(rsm, rsm_q, rsm_cbar[rsm_q] > 0.0,
               rsm_aq, rsm_bbar, &rsm_tagp, rsm_tol_piv,
               0.10 * rsm_tol_bnd);
         }
         if (rsm_p == 0)
         {  /* it should never be */
            fault("rsm_find_feas: internal logic error");
         }
         /* restore original bounds of xB[p] */
         if (rsm_p > 0)
         {  int p = rsm_p;
            k = rsm->indb[p]; /* x[k] = xB[p] */
            if (rsm->type[k] != orig_type[k])
            {  /* x[k] is implicit artificial variable */
               insist(rsm_tagp == 'L' || rsm_tagp == 'U');
               rsm_tagp = (rsm_tagp == 'L' ? 'U' : 'L');
               rsm->type[k] = orig_type[k];
               rsm->lb[k]   = orig_lb[k];
               rsm->ub[k]   = orig_ub[k];
               if (rsm->type[k] == 'S') rsm_tagp = 'S';
            }
         }
         /* update weights (if steepest edge pricing is used) */
         if (rsm_prim_steep && rsm_p > 0)
         {  double *ap = ucalloc(1+n, sizeof(double));
            eval_zeta(rsm, rsm_p, w);
            eval_row(rsm, w, ap);
            update_gvec(rsm, rsm_gvec, rsm_p, rsm_q, ap, rsm_aq, w);
            ufree(ap);
         }
         /* jump to the adjacent basis */
         if (change_b(rsm, rsm_p, rsm_tagp, rsm_q) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm_prim_steep && rsm_check_gvec)
            print("check_gvec: %g", check_gvec(rsm, rsm_gvec));
         /* end of main loop */
      }
      /* restore original bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  rsm->type[k] = orig_type[k];
         rsm->lb[k] = orig_lb[k];
         rsm->ub[k] = orig_ub[k];
      }
      /* free working arrays */
      ufree(orig_type), ufree(orig_lb), ufree(orig_ub);
      if (rsm_prim_steep) ufree(y), ufree(w);
      /* return to the calling program */
      return ret;
}

/*----------------------------------------------------------------------
-- rsm_primal_opt - find optimal solution by primal simplex method.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_primal_opt(RSM *rsm);
--
-- *Description*
--
-- The rsm_primal_opt routine searches for optimal solution using primal
-- simplex method. It assumes that the current basis solution is primal
-- feasible.
--
-- *Returns*
--
-- The rsm_primal_opt routine returns one of the following codes:
--
-- 0 - optimal solution found;
-- 1 - problem has unbounded solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static void display2(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      if (need || rsm_t_last == 0 ||
          clock() - rsm_t_last > CLOCKS_PER_SEC)
      {  int def;
         double val, sum;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         print("*%6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm_t_last = clock();
      }
      return;
}

static int rsm_primal_opt(RSM *rsm)
{     int m = rsm->m, ret;
      double *y, *w;
      /* allocate working arrays */
      if (rsm_prim_steep)
      {  y = ucalloc(1+m, sizeof(double));
         w = ucalloc(1+m, sizeof(double));
      }
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         eval_bbar(rsm, rsm_bbar);
         /* display information about current basis solution */
         display2(0);
         /* check current solution for numerical stability */
         if (check_bbar(rsm, rsm_bbar, rsm_tol_bnd))
         {  if (rsm_fin_out) display2(1);
            ret = 2;
            break;
         }
         /* compute simplex multipliers */
         eval_pi(rsm, rsm_cost, rsm_pi);
         /* compute reduced costs of non-basis variables */
         eval_cbar(rsm, rsm_cost, rsm_pi, rsm_cbar);
         /* choose non-basis variable xN[q] */
         rsm_q = pivot_col(rsm, rsm_cost, rsm_cbar, rsm_gvec,
            rsm_tol_dj);
         if (rsm_q == 0)
         {  /* optimal solution found */
            if (rsm_fin_out) display2(1);
            ret = 0;
            break;
         }
         /* compute pivot column of simplex table */
         eval_col(rsm, rsm_q, rsm_aq, 1);
         /* choose basis variable xB[p] */
         if (!rsm_prim_relax)
         {  /* use standard "textbook" ratio test */
            rsm_p = pivot_row(rsm, rsm_q, rsm_cbar[rsm_q] > 0.0,
               rsm_aq, rsm_bbar, &rsm_tagp, rsm_tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_p = harris_row(rsm, rsm_q, rsm_cbar[rsm_q] > 0.0,
               rsm_aq, rsm_bbar, &rsm_tagp, rsm_tol_piv,
               0.10 * rsm_tol_bnd);
         }
         if (rsm_p == 0)
         {  /* problem has unbounded solution */
            if (rsm_fin_out) display2(1);
            ret = 1;
            break;
         }
         /* update weights (if steepest edge pricing is used) */
         if (rsm_prim_steep && rsm_p > 0)
         {  double *ap = ucalloc(1+rsm->n, sizeof(double));
            eval_zeta(rsm, rsm_p, w);
            eval_row(rsm, w, ap);
            update_gvec(rsm, rsm_gvec, rsm_p, rsm_q, ap, rsm_aq, w);
            ufree(ap);
         }
         /* jump to the adjacent basis */
         if (change_b(rsm, rsm_p, rsm_tagp, rsm_q) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm_prim_steep && rsm_check_gvec)
            print("check_gvec: %g", check_gvec(rsm, rsm_gvec));
      }
      /* free working arrays */
      if (rsm_prim_steep) ufree(y), ufree(w);
      /* returns to the calling program */
      return ret;
}

/*----------------------------------------------------------------------
-- rsm_dual_opt - find optimal solution by dual simplex method.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_dual_opt(RSM *rsm);
--
-- *Description*
--
-- The rsm_dual_opt routine searches for optimal solution using dual
-- simplex method. It assumes that the current basis solution is dual
-- feasible.
--
-- This routine may be used to find primal feasible solution by means
-- of usual practice.
--
-- *Returns*
--
-- The rsm_dual_opt routine returns one of the following codes:
--
-- 0 - optimal solution found;
-- 1 - problem has no (primal) feasible solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static void display3(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      if (need || rsm_t_last == 0 ||
          clock() - rsm_t_last > CLOCKS_PER_SEC)
      {  int def;
         double val, sum;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         print(" %6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm_t_last = clock();
      }
      return;
}

static int rsm_dual_opt(RSM *rsm)
{     int m = rsm->m, ret;
      double *w;
      /* allocate working array */
      if (rsm_dual_steep) w = ucalloc(1+m, sizeof(double));
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         eval_bbar(rsm, rsm_bbar);
         /* display information about current basis solution */
         display3(0);
         /* compute simplex multipliers */
         eval_pi(rsm, rsm_cost, rsm_pi);
         /* compute reduced costs of non-basis variables */
         eval_cbar(rsm, rsm_cost, rsm_pi, rsm_cbar);
         /* check current solution for numerical stability */
         if (check_cbar(rsm, rsm_cost, rsm_cbar, rsm_tol_dj))
         {  /* numerical stability lost */
            if (rsm_fin_out) display3(1);
            ret = 2;
            break;
         }
         /* check current solution for primal feasibility */
         if (check_bbar(rsm, rsm_bbar, 0.30 * rsm_tol_bnd) == 0)
         {  /* the current basis is primal feasible */
            if (rsm_fin_out) display3(1);
            ret = 0;
            break;
         }
         /* choose basis variable xB[p] */
         rsm_p = dual_row(rsm, rsm_bbar, rsm_dvec,
            &rsm_tagp, 0.30 * rsm_tol_bnd);
         if (rsm_p == 0)
         {  /* it should never be because the current basis solution is
               primal infeasible */
            fault("rsm_dual_opt: internal logic error");
         }
         /* compute p-th row of the inverse inv(B) */
         eval_zeta(rsm, rsm_p, rsm_bp);
         /* compute p-th row of the current simplex table */
         eval_row(rsm, rsm_bp, rsm_ap);
         /* choose non-basis variable xN[q] */
         if (!rsm_dual_relax)
         {  /* use standard "textbook" ratio test */
            rsm_q = dual_col(rsm, rsm_tagp, rsm_ap, rsm_cbar,
               rsm_tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_q = harris_col(rsm, rsm_tagp, rsm_ap, rsm_cost,rsm_cbar,
               rsm_tol_piv, 0.10 * rsm_tol_dj);
         }
         if (rsm_q == 0)
         {  /* problem has no (primal) feasible solution */
            if (rsm_fin_out) display3(1);
            ret = 1;
            break;
         }
         /* correct tagp if xB[p] is fixed variable (see comments to
            routines rsm_dual_row, rsm_dual_col, and rsm_harris col) */
         if (rsm->type[rsm->indb[rsm_p]] == 'S')
            rsm_tagp = 'S';
         /* update weights (if steepest edge pricing is used) */
         if (rsm_dual_steep)
         {  eval_col(rsm, rsm_q, rsm_aq, 1);
            update_dvec(rsm, rsm_dvec, rsm_p, rsm_q, rsm_ap,
               rsm_aq, w);
         }
         /* compute q-th column of the current simplex table (this is
            not needed if steepest edge pricing is used, because this
            column is computed by the rsm_update_dvec routine) */
         if (!rsm_dual_steep) eval_col(rsm, rsm_q, rsm_aq, 1);
         /* jump to the adjacent basis */
         if (change_b(rsm, rsm_p, rsm_tagp, rsm_q) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm_dual_steep && rsm_check_dvec)
            print("check_dvec: %g", check_dvec(rsm, rsm_dvec));
         /* end of main loop */
      }
      /* free working array */
      if (rsm_dual_steep) ufree(w);
      /* return to the calling program */
      return ret;
}

/* eof */
