/* glpipm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- This program is a demo solver based on a *tentative* implementation
-- of the primal-dual interior point method. It is intended for solving
-- large scale linear programming problems.
----------------------------------------------------------------------*/

static char *version =
   "GLPIPM -- GLPK Interior-Point Demo Solver, Version 2.0";

static char *mps_fname = "";
/* the name of the input text file that contains LP problem data in MPS
   format */

static char *out_fname = "";
/* the name of the output text file to which the final solution should
   be sent using printable format (optional) */

/*----------------------------------------------------------------------
-- display_help - display help.
--
-- This routine displays help information about the program as it is
-- required by the GNU Coding Standards. */

static void display_help(char *my_name)
{     print("Usage: %s [options...] mps-file", my_name);
      print("");
      print("Options:");
      print("   --any             only feasible solution should be foun"
         "d");
      print("   --min             objective function should be minimize"
         "d (default)");
      print("   --max             objective function should be maximize"
         "d");
      print("   -o filename, --output filename");
      print("                     send final solution to filename using"
         " printable");
      print("                     format");
      print("   --sum             enable multiplets in constraint matri"
         "x (will be");
      print("                     replaced by their sum)");
      print("   --nosum           disable multiplets in constraint matr"
         "ix (default)");
      print("   -h, --help        display this help information and exi"
         "t");
      print("   -v, --version     display program version and exit");
      print("");
      print("Please, report bugs to <bug-glpk@gnu.org>");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- display_version - display version.
--
-- This routine displays version information for the program as it is
-- required by the GNU Coding Standards. */

static void display_version(void)
{     print("%s", version);
      print("Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.r"
         "u, mao@gnu.org>");
      print("This program is free software; you may redistribute it und"
         "er the terms of");
      print("the GNU General Public License. This program has absolutel"
         "y no warranty.");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- process_cmdline - process command line parameters.
--
-- This routine processes parameters specified in command line. */

static void process_cmdline(int argc, char *argv[])
{     int k;
      for (k = 1; k < argc; k++)
      {  if (strcmp(argv[k], "--any") == 0)
            insist(glp_set_ipar("obj_dir", GLP_ANY) == 0);
         else if (strcmp(argv[k], "--min") == 0)
            insist(glp_set_ipar("obj_dir", GLP_MIN) == 0);
         else if (strcmp(argv[k], "--max") == 0)
            insist(glp_set_ipar("obj_dir", GLP_MAX) == 0);
         else if (strcmp(argv[k], "-o") == 0 ||
                  strcmp(argv[k], "--output") == 0)
         {  k++;
            if (k == argc)
            {  error("No output file name specified");
               exit(EXIT_FAILURE);
            }
            if (out_fname[0] != '\0')
            {  error("Only one output file allowed");
               exit(EXIT_FAILURE);
            }
            out_fname = argv[k];
         }
         else if (strcmp(argv[k], "--sum") == 0)
            insist(glp_set_ipar("sum_aij", GLP_YES) == 0);
         else if (strcmp(argv[k], "--nosum") == 0)
            insist(glp_set_ipar("sum_aij", GLP_NO) == 0);
         else if (strcmp(argv[k], "-h") == 0 ||
                  strcmp(argv[k], "--help") == 0)
            display_help(argv[0]);
         else if (strcmp(argv[k], "-v") == 0 ||
                  strcmp(argv[k], "--version") == 0)
            display_version();
         else if (argv[k][0] == '-' ||
                 (argv[k][0] == '-' && argv[k][1] == '-'))
         {  error("Invalid option `%s'; try %s --help", argv[k],
               argv[0]);
            exit(EXIT_FAILURE);
         }
         else
         {  if (mps_fname[0] != '\0')
            {  error("Only one input file allowed");
               exit(EXIT_FAILURE);
            }
            mps_fname = argv[k];
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- main - main routine.
--
-- This main routine is called by the control program and manages the
-- solving process. */

int main(int argc, char *argv[])
{     int obj_dir;
      clock_t t_start;
      /* initialize GLPK API */
      glp_initialize();
      /* process command line parameters */
      process_cmdline(argc, argv);
      /* read LP problem data from MPS file */
      if (mps_fname[0] == '\0')
      {  error("No input MPS file specified");
         exit(EXIT_FAILURE);
      }
      glp_get_ipar("obj_dir", &obj_dir);
      if (glp_read_mps(mps_fname) != 0)
      {  error("Can't read MPS file");
         exit(EXIT_FAILURE);
      }
      glp_set_ipar("obj_dir", obj_dir);
      /* solve LP problem */
      t_start = clock();
      if (glp_interior() == 1)
      {  error("Can't solve the problem because of errors");
         exit(EXIT_FAILURE);
      }
      /* display statistics */
      print("Time used: %.1f secs",
         (double)(clock() - t_start) / (double)CLOCKS_PER_SEC);
      print("Memory used: %.1fM (%d bytes)",
         (double)mem_tpeak / (1024.0 * 1024.0), mem_tpeak);
      /* write final solution found by the solver */
      if (out_fname[0] != '\0' && glp_print_sol(out_fname) != 0)
      {  error("Can't write solution");
         exit(EXIT_FAILURE);
      }
      /* terminate GLPK API */
      glp_terminate();
      /* exit to the control program */
      return 0;
}

/* eof */
