/* glpset.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPSET_H
#define _GLPSET_H

#define print                 glp_print
#define err_flag              glp_err_flag
#define error                 glp_error
#define fault                 glp_fault
#define _insist               glp_insist
#define mem_flag              glp_mem_flag
#define mem_limit             glp_mem_limit
#define mem_total             glp_mem_total
#define mem_tpeak             glp_mem_tpeak
#define mem_count             glp_mem_count
#define mem_cpeak             glp_mem_cpeak
#define umalloc               glp_umalloc
#define ucalloc               glp_ucalloc
#define ufree                 glp_ufree
#define create_pool           glp_create_pool
#define get_atom              glp_get_atom
#define free_atom             glp_free_atom
#define get_atomv             glp_get_atomv
#define clear_pool            glp_clear_pool
#define delete_pool           glp_delete_pool
#define open_data             glp_open_data
#define scan_data             glp_scan_data
#define close_data            glp_close_data

extern void print(char *fmt, ...);
/* print informative message */

extern int err_flag;
/* if this flag is set, the error routine abnormally terminates program
   execution (i.e. it acts like the fault routine) */

extern void error(char *fmt, ...);
/* print error message and continue program execution */

extern void fault(char *fmt, ...);
/* print error message and terminate program execution */

#define insist(expr) \
   ((void)((expr) || (_insist(#expr, __FILE__, __LINE__), 1)))

extern void _insist(char *expr, char *file, int line);
/* check for logical condition */

/* some processors need data to be properly aligned; the align_boundary
   macro defines the boundary which should fit for all data types; the
   align_datasize macro allows enlarging size of data item in order the
   immediately following data of any type should be properly aligned */

#define align_boundary sizeof(double)

#define align_datasize(size) \
   ((((size) + (align_boundary - 1)) / align_boundary) * align_boundary)

struct mem_desc
{     /* memory block descriptor */
      int size;
      /* size of block (in bytes, including descriptor) */
      int flag;
      /* descriptor flag */
      /* actual data starts here (there may be a "hole" between the flag
         field and actual data because of data alignment) */
};

extern int mem_flag;
/* value used as memory block descriptor flag */

extern int mem_limit;
/* maximal amount of memory (in bytes) that is available for dynamic
   allocation */

extern int mem_total;
/* total amount of memory (in bytes) that is currently allocated (the
   sum of size fields for all memory block descriptors) */

extern int mem_tpeak;
/* peak value of mem_total reached during program execution */

extern int mem_count;
/* total number of currently allocated memory blocks */

extern int mem_cpeak;
/* peak value of mem_count reached during program execution */

extern void *umalloc(int size);
/* allocate memory block */

extern void *ucalloc(int nmemb, int size);
/* allocate and clear memory block */

extern void ufree(void *ptr);
/* free memory block allocated by umalloc or ucalloc */

typedef struct POOL POOL;

struct POOL
{     /* memory pool (a set of atoms) */
      int size;
      /* size of each atom in bytes (1 <= size <= 256); if size = 0,
         different atoms may have different sizes */
      void *avail;
      /* pointer to linked list of free atoms */
      void *link;
      /* pointer to linked list of allocated blocks (it points to the
         last recently allocated block) */
      int used;
      /* number of bytes used in the last allocated block */
      void *stock;
      /* pointer to linked list of free blocks */
      int count;
      /* total number of allocated atoms */
};

#define POOL_SIZE 8000
/* the size of each memory block for all pools (may be increased if
   necessary) */

extern POOL *create_pool(int size);
/* create memory pool */

extern void *get_atom(POOL *pool);
/* allocate atom of fixed size */

extern void free_atom(POOL *pool, void *ptr);
/* free an atom */

extern void *get_atomv(POOL *pool, int size);
/* allocate atom of variable size */

extern void clear_pool(POOL *pool);
/* free all atoms */

extern void delete_pool(POOL *pool);
/* delete memory pool */

typedef struct DATA DATA;

struct DATA
{     /* data stream */
      char *fn;
      /* name of an input text file */
      void *fp; /* FILE *fp; */
      /* stream assigned to the input text file */
      int cn;
      /* number of read text lines */
};

extern DATA *open_data(char *fname);
/* open data stream */

extern int scan_data(DATA *ds, ...);
/* read data from data stream */

extern void close_data(DATA *ds);
/* close data stream */

#endif

/* eof */
