/* glpeta.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPETA_H
#define _GLPETA_H

#include "glpset.h"

#define create_eta            glp_create_eta
#define prp_term              glp_prp_term
#define app_term              glp_app_term
#define h_solve               glp_h_solve
#define ht_solve              glp_ht_solve
#define reset_eta             glp_reset_eta
#define delete_eta            glp_delete_eta

/* The structure HFILE defines so called eta-file which is a square
-- matrix H represented in the following product form:
--
--    H = H(1) * H(2) * ... * H(t),
--
-- where H(1), H(2), ..., H(t) are matrix multipliers or terms, t is
-- total number of terms (size of eta-file). Note that when t = 0, H is
-- formally considered as unity matrix.
--
-- Each term H(k), k = 1,2,...,t, is an elementary square matrix that
-- differs from the unity matrix only in one element h[i,j]:
--
--       1     j     n           1   j       n
--    1  1 . . . . . .        1  1 . . . . . .
--       . 1 . . . . .           . 1 . . . . .
--       . . 1 . . . .           . . 1 . . . .
--    i  . . . h . . .           . . . 1 . . .
--       . . . . 1 . .        i  . . h . 1 . .
--       . . . . . 1 .           . . . . . 1 .
--    n  . . . . . . 1        n  . . . . . . 1
--
--       (a) if i = j            (b) if i != j
--
-- Terms of eta-file are defined by the structure HTERM. */

typedef struct HFILE HFILE;
typedef struct HTERM HTERM;

struct HFILE
{     /* eta-file */
      int n;
      /* matrix order */
      POOL *pool;
      /* memory pool holding eta-terms */
      HTERM *head;
      /* pointer to the first eta-term H(1) */
      HTERM *tail;
      /* pointer to the last eta-term H(t) */
};

struct HTERM
{     /* eta-term H(k) */
      int i;
      /* row number (1 <= i <= n) */
      int j;
      /* column number (1 <= j <= n) */
      double val;
      /* element value */
      HTERM *prev;
      /* pointer to the previous eta-term H(k-1) */
      HTERM *next;
      /* pointer to the next eta-term H(k+1) */
};

extern HFILE *create_eta(int n);
/* create eta-file */

extern HTERM *prp_term(HFILE *eta, int i, int j, double val);
/* prepend new eta-term to eta-file */

extern HTERM *app_term(HFILE *eta, int i, int j, double val);
/* append new eta-term to eta-file */

extern double *h_solve(HFILE *eta, double x[]);
/* solve linear system H*x = b using eta-file */

extern double *ht_solve(HFILE *eta, double x[]);
/* solve transposed linear system H'*x = b using eta-file */

extern void reset_eta(HFILE *eta);
/* make eta-file empty */

extern void delete_eta(HFILE *eta);
/* delete eta-file */

#endif

/* eof */
