/* glprsm/rsm_update_gamma.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_update_gamma - update weights (steepest edge technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_update_gamma(RSM *rsm, double y[], double w[]);
--
-- *Description*
--
-- The rsm_update_gamma routine recomputes weights, which are used in
-- steepest edge technique. It is assumed that weights for the current
-- basis are stored in the array gamma, p is the number of the basis
-- variable xB[p] chosen to leave the basis, and q is the number of the
-- non-basis variable xN[q] chosen to enter the basis. On exit the array
-- gamma will contain weights for the adjacent basis.
--
-- The arrays y and w are auxiliary. They should have at least m+1
-- elements, where m is number of rows.
--
-- This routine should be called before the basis will be changed. It
-- uses the following formulae to recompute weights:
--
--                   m      2
--     gamma  = 1 + SUM alfa   ,
--          q       j=1     i,q
--
--               -T
--          y = B   e ,
--                   p
--
--               -T ~
--          w = B   A ,
--                   q
--
--    ----        T
--    alfa    = (N  y) / alfa
--        p,j     j          p,q
--
--               T
--          t = N  w,
--               j
--
--     -----                 ---- 2             ----         ---- 2
--     gamma  = max(gamma  + alfa    gamma  - 2 alfa   , 1 + alfa   ),
--          j            j       p,j      q         p,j          p,j
--
--    -----                 2
--    gamma  = gamma  / alfa    ,
--         q        q       p,q
--
-- where:
--
-- q is the number of the non-basis variable (xN)q chosen to enter the
-- basis;
--
-- p is the number of the basis variable (xB)p chosen to leave the
-- basis;
--
-- gamma[q] is the weight coefficient at the variable (xN)q for the
-- current basis (the corresponding formula is used to compute this
-- coefficient exactly, because q-th column of the current simplex table
-- is available);
--
-- alfa[i,q] are elements of q-th column of the current simplex table;
--
-- B is the basis matrix corresponding to the current (not adjacent!)
-- basis;
--
-- e[p] is the unity vector of the form (0,...,0,1,0,...,0), where 1 is
-- placed in p-th position;
--
-- y is an auxiliary vector (p-th row of the matrix inv(B)). This vector
-- is computed once;
--
-- A~[q] is q-th column of the current simplex table;
--
-- w is an auxiliary vector computed once;
--
-- N[j] is the column of the expanded constraint matrix that corresponds
-- to the non-basis variable (xN)j, j != q;
--
-- alfa[p,q] is the pivot element of the current simplex table;
--
-- ----
-- alfa[p,j] are elements of p-th row of the adjacent simplex table;
--
-- t is an auxiliary quantity;
--
-- gamma[j] is the weight coefficient at the variable (xN)j for the
-- current basis;
--
-- -----
-- gamma[j] is the weight coefficient at the variable (xN)j for the
-- adjacent basis.
--
-- -----
-- gamma[q] is the weight coefficient at the variable (xN)j for the
-- adjacent basis.
--
-- For details see:
-- D.Goldfarb, J.K.Reid. A practicable steepest-edge algorithm. Math.
-- Prog., 12, 1977, 361-71. */

void rsm_update_gamma(RSM *rsm, double y[], double w[])
{     ELEM *e;
      int m = rsm->m, n = rsm->n, p = rsm->p, q = rsm->q, i, j, k;
      double *gamma = rsm->gamma, apj, t, t1, t2;
      insist(1 <= p && p <= m && 1 <= q && q <= n);
      gamma[q] = 1.0;
      for (i = 1; i <= m; i++) gamma[q] += rsm->aq[i] * rsm->aq[i];
      for (i = 1; i <= m; i++) y[i] = 0.0;
      y[p] = 1.0;
      rsm_btran(rsm, y);
      for (i = 1; i <= m; i++) w[i] = rsm->aq[i];
      rsm_btran(rsm, w);
      for (j = 1; j <= n; j++)
      {  if (j == q) continue;
         apj = 0.0;
         k = rsm->indn[j];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            apj += e->val * y[e->i];
         apj /= rsm->aq[p];
         t = 0.0;
         k = rsm->indn[j];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            t += e->val * w[e->i];
         t1 = gamma[j] + apj * apj * gamma[q] - 2.0 * apj * t;
         t2 = 1.0 + apj * apj;
         gamma[j] = t1 > t2 ? t1 : t2;
      }
      gamma[q] /= (rsm->aq[p] * rsm->aq[p]);
      return;
}

/* eof */
