/* glprsm/rsm_primal_opt.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stdlib.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_primal_opt - find optimal solution by primal simplex method.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_primal_opt(RSM *rsm);
--
-- *Description*
--
-- The rsm_primal_opt routine searches for optimal solution using primal
-- simplex method. It assumes that the current basis solution is primal
-- feasible.
--
-- *Returns*
--
-- The rsm_primal_opt routine returns one of the following codes:
--
-- 0 - optimal solution found;
-- 1 - problem has unbounded solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static RSM *_rsm;
/* revised simplex method common block */

static void display(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      RSM *rsm = _rsm;
      if (need || rsm->t_last == 0 ||
          clock() - rsm->t_last > CLOCKS_PER_SEC)
      {  int def;
         double val, sum;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         print("*%6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm->t_last = clock();
      }
      return;
}

int rsm_primal_opt(RSM *rsm)
{     int m = rsm->m, ret;
      double *y, *w;
      _rsm = rsm;
      /* allocate working arrays */
      if (rsm->prim_steep)
      {  y = ucalloc(1+m, sizeof(double));
         w = ucalloc(1+m, sizeof(double));
      }
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         rsm_set_bbar(rsm);
         /* display information about current basis solution */
         display(0);
         /* check current solution for numerical stability */
         if (rsm_check_bbar(rsm, rsm->tol_bnd))
         {  if (rsm->fin_out) display(1);
            ret = 2;
            break;
         }
         /* compute simplex multipliers */
         rsm_set_pi(rsm);
         /* compute reduced costs of non-basis variables */
         rsm_set_cbar(rsm);
         /* choose non-basis variable xN[q] */
         rsm_pivot_col(rsm, rsm->tol_dj);
         if (rsm->q == 0)
         {  /* optimal solution found */
            if (rsm->fin_out) display(1);
            ret = 0;
            break;
         }
         /* compute pivot column of simplex table */
         rsm_set_aq(rsm);
         /* choose basis variable xB[p] */
         if (!rsm->prim_relax)
         {  /* use standard "textbook" ratio test */
            rsm_pivot_row(rsm, rsm->tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_harris_row(rsm, rsm->tol_piv, 0.10 * rsm->tol_bnd);
         }
         if (rsm->p == 0)
         {  /* problem has unbounded solution */
            if (rsm->fin_out) display(1);
            ret = 1;
            break;
         }
         /* update weights (if steepest edge pricing is used) */
         if (rsm->prim_steep && rsm->p > 0) rsm_update_gamma(rsm, y, w);
         /* jump to the adjacent basis */
         if (rsm_change_basis(rsm) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm->prim_steep && rsm->check_gamma)
            print("check_gamma: %g", rsm_check_gamma(rsm));
      }
      /* free working arrays */
      if (rsm->prim_steep) ufree(y), ufree(w);
      /* returns to the calling program */
      return ret;
}

/* eof */
