/* glprsm/rsm_pivot_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_pivot_col.c - choose non-basis variable (primal simplex).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_pivot_col(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_pivot_col routine chooses the non-basic variable xN[q]
-- (i.e. pivot column of the current simplex table), which will enter
-- the basis. If the current solution is optimal and hence the choice
-- of xN[q] is impossible, the routine sets q to zero. It is assumed
-- that the objective function should be minimized and the array cbar
-- contains reduced costs of all non-basis variables.
--
-- The parameter tol is an absolute tolerance. If reduced cost cbar[j]
-- is too small, i.e. if |cbar[j]| < tol, then non-basis variable xN[j]
-- is not considered by the routine.
--
-- This routine chooses that non-basis variable, which satisfies to the
-- following condition:
--
--    xN[j] is free           (LP_NF) and cbar[j] != 0, or
--    xN[j] is on lower bound (LP_NL) and cbar[j] <  0, or
--    xN[j] is on upper bound (LP_NU) and cbar[j] >  0
--
-- (i.e. changing of this variable in the feasible direction implies
-- decreasing of the objective function), and which has largest absolute
-- value of its weighted reduced cost:
--
--    cbar[j] / sqrt(gamma[j]),
--
-- where cbar[j] is the reduced cost of xN[j], gamma[j] is the weight
-- of j-th column of the current simplex table.
--
-- If all gamma[j] are equal to 1, the rsm_pivot_col routine implements
-- standard Dantzig's pricing technique. If gamma[j] reflects euclidian
-- norm of j-th column of the current simplex table, the rsm_pivot_col
-- routine implements steepest edge pricing technique that was proposed
-- by Goldfarb and Reid. */

void rsm_pivot_col(RSM *rsm, double tol)
{     int n = rsm->n, j;
      double big, temp;
      rsm->q = 0, big = 0.0;
      for (j = 1; j <= n; j++)
      {  if (rsm->cbar[j] == 0.0 || fabs(rsm->cbar[j]) < tol)
            continue;
         if (rsm->tagn[j] == LP_NF ||
             rsm->tagn[j] == LP_NL && rsm->cbar[j] < 0.0 ||
             rsm->tagn[j] == LP_NU && rsm->cbar[j] > 0.0)
         {  temp = fabs(rsm->cbar[j]) / sqrt(rsm->gamma[j]);
            if (big < temp) rsm->q = j, big = temp;
         }
      }
      return;
}

/* eof */
