/* glprsm/rsm_harris_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_harris_col - choose non-basis variable (dual, Harris technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_harris_col(RSM *rsm, double tol, double tol1);
--
-- *Description*
--
-- The rsm_harris_col performs the same operation as the rsm_dual_col
-- routine, but using the dual version of a technique, proposed by
-- P.Harris. This routine acts like rsm_harris_row routine (which may
-- used in primal simplex), however the former operates with reduced
-- costs (dual variables) instead basis (primal) variables.
--
-- The parameter tol has the same meaning as for the rsm_dual_col
-- routine.
--
-- The parameter tol1 specifies an absolute tolerance, which relaxes
-- (zero) bounds of reduced costs on the first pass.
--
-- *Returns*
--
-- The rsm_harris_col routine returns absolute value of the reduced
-- cost of the chosen basis variable xB[p], which will take place when
-- this variable will leave the basis. */

double rsm_harris_col(RSM *rsm, double tol, double tol1)
{     int n = rsm->n, j;
      double big, eps, temp, teta, tmax;
#     define gap tol1
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (j = 1; j <= n; j++)
         if (big < fabs(rsm->ap[j])) big = fabs(rsm->ap[j]);
      eps = tol * big;
      /* turn to the case of increasing xB[p] in order to simplify
         program logic (remember that the objective function should be
         minimized) */
      if (rsm->tagp == LP_NU)
         for (j = 1; j <= n; j++) rsm->ap[j] = - rsm->ap[j];
      /* initial settings for the first pass */
      tmax = DBL_MAX;
      /* the first look through the list of non-basis variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (rsm->ap[j] == 0.0 || fabs(rsm->ap[j]) < eps) continue;
         /* analyze main cases */
         if (rsm->tagn[j] == LP_NF)
         {  /* xN[j] is free variable */
            if (rsm->ap[j] > 0.0) goto lo; else goto up;
         }
         else if (rsm->tagn[j] == LP_NL)
         {  /* xN[j] is on its lower bound */
            if (rsm->ap[j] < 0.0) continue;
lo:         temp = (rsm->cbar[j] + gap) / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NU)
         {  /* xN[j] is on its upper bound */
            if (rsm->ap[j] > 0.0) continue;
up:         temp = (rsm->cbar[j] - gap) / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NS)
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            insist(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced costs of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* compute maximal allowable change of reduced cost of the
            basis variable xB[p] */
         if (tmax > temp) tmax = temp;

      }
      /* initial settings for the second pass */
      rsm->q = 0, teta = DBL_MAX, big = 0.0;
      /* the second look through the list of non-basis variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (rsm->ap[j] == 0.0 || fabs(rsm->ap[j]) < eps) continue;
         /* analyze main cases */
         if (rsm->tagn[j] == LP_NF)
         {  /* xN[j] is free variable */
            temp = 0.0;
         }
         else if (rsm->tagn[j] == LP_NL)
         {  /* xN[j] is on its lower bound */
            if (rsm->ap[j] < 0.0) continue;
            temp = rsm->cbar[j] / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NU)
         {  /* xN[j] is on its upper bound */
            if (rsm->ap[j] > 0.0) continue;
            temp = rsm->cbar[j] / rsm->ap[j];
         }
         else if (rsm->tagn[j] == LP_NS)
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            insist(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced costs of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the dual version of Harris' rule */
         if (temp <= tmax &&  big < fabs(rsm->ap[j]))
         {  rsm->q = j;
            teta = temp;
            big = fabs(rsm->ap[j]);
         }
      }
      /* restore original signs of the coefficients ap[j] */
      if (rsm->tagp == LP_NU)
         for (j = 1; j <= n; j++) rsm->ap[j] = - rsm->ap[j];
      return teta;
#     undef gap
}

/* eof */
