/* glprsm/rsm_dual_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_dual_row - choose basis variable (dual simplex).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_dual_row(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_dual_row routine chooses the basis variable xB[p] (i.e.
-- pivot row of the current simplex table), which will leave the basis.
-- If the current solution is optimal and hence the choice of xB[p] is
-- impossible, the routine sets p to zero.
--
-- The parameter tol is a relative tolerance. If the residual of some
-- basis variable xB[i] is not above than eps = tol*max(|lB[i]|, 1) or
-- eps = tol*max(|uB[i]|,1) (depending on what bound is violated), then
-- xB[i] is considered as feasible and not chosen.
--
-- This routine chooses that basis variable, which has largest weighted
-- residual:
--
--    (lB[i] - xB[i]) / sqrt(dzeta[i]),   if xB[i] < lB[i], or
--    (xB[i] - uB[i]) / sqrt(dzeta[i]),   if xB[i] > uB[i],
--
-- where dzeta[i] is the weight of i-th row of the current simplex
-- table.
--
-- If all dzeta[i] are equal to 1, the rsm_dual_row routine implements
-- standard "textbook" pivoting technique. Otherwise the routine allows
-- to use some dual steepest edge pivoting technique.
--
-- SHOULD NOTE that if the choosen basis variable xB[p] is fixed, tagp
-- will be set to LP_NL or LP_NU, but not to LP_NS. This is used to
-- recognise how this variable should be changed in order to eliminate
-- its residual: in case of LP_NL it should be decreased and in case of
-- LP_NU it should be increased (as if it were double-bounded variable
-- with lB[i] = uB[i]). Surely tagp should be replaced by LP_NS later.
--
-- *Returns*
--
-- The rsm_dual_row routine returns the total number of residuals. */

int rsm_dual_row(RSM *rsm, double tol)
{     int m = rsm->m, i, k, count = 0;
      double big, temp;
      rsm->p = 0, rsm->tagp = -1, big = 0.0;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_LO || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  /* xB[i] has lower bound */
            if (rsm_check_dev(rsm->bbar[i], rsm->lb[k], tol) == -2)
            {  count++;
               temp = (rsm->lb[k] - rsm->bbar[i]) / sqrt(rsm->dzeta[i]);
               if (big < temp)
                  rsm->p = i, rsm->tagp = LP_NL, big = temp;
            }
         }
         if (rsm->type[k] == LP_UP || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  /* xB[i] has upper bound */
            if (rsm_check_dev(rsm->bbar[i], rsm->ub[k], tol) == +2)
            {  count++;
               temp = (rsm->bbar[i] - rsm->ub[k]) / sqrt(rsm->dzeta[i]);
               if (big < temp)
                  rsm->p = i, rsm->tagp = LP_NU, big = temp;
            }
         }
      }
      return count;
}

/* eof */
