/* glprsm/rsm_check_cbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_check_cbar - check current solution for numerical stability.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_check_cbar(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_check_cbar routine checks numerical stability of the current
-- basis solution. The solution is considered as stable if:
--
--    -tol <= cbar[j] <= +tol   when xN[j] is free variable,
--    -tol <= cbar[j] <  +inf   when xN[j] is on its lower bound,
--    -inf <  cbar[j] <= +tol   when xN[j] is on its upper bound,
--    -inf <  cbar[j] <  +inf   when xN[j] is fixed variable,
--
-- where tol is the given absolute tolerance, cbar[j] is the reduced
-- cost of non-basis variable xN[j].
--
-- *Returns*
--
-- If the current basis solution is stable, the routine returns zero.
-- Otherwise the routine returns non-zero. */

int rsm_check_cbar(RSM *rsm, double tol)
{     int n = rsm->n, j;
      for (j = 1; j <= n; j++)
      {  if (rsm->tagn[j] == LP_NF || rsm->tagn[j] == LP_NL)
            if (rsm->cbar[j] < -tol) return 1;
         if (rsm->tagn[j] == LP_NF || rsm->tagn[j] == LP_NU)
            if (rsm->cbar[j] > +tol) return 1;
      }
      return 0;
}

/* eof */
