/* glpprob/load_lpsol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpprob.h"

/*----------------------------------------------------------------------
-- load_lpsol - read LP solution from text file using plain format.
--
-- *Synopsis*
--
-- #include "glpprob.h"
-- LPSOL *load_lpsol(char *fname);
--
-- *Description*
--
-- The load_lpsol routine reads an LP problem basis solution block from
-- the text file, whose name is the character string fname, using plain
-- format (it is described in comments to the routine save_lpsol).
--
-- *Returns*
--
-- If the operation was successful, the load_lpsol routine returns a
-- pointer to the loaded solution block. Otherwise the routine returns
-- NULL. */

LPSOL *load_lpsol(char *fname)
{     DATA *ds;
      LPSOL *sol = NULL;
      int m, n, k;
      double objval, valx, dx;
      char str[255+1];
      print("load_lpsol: reading LP solution block from `%s'...",
         fname);
      ds = open_data(fname);
      if (ds == NULL) goto fail;
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "LPSOL"))
      {  error("%s:%d: marker `LPSOL' not found", ds->fn, ds->cn);
         goto fail;
      }
      if (scan_data(ds, 'i', &m, 'i', &n, 's', str, 'e', &objval, 0))
         goto fail;
      if (m <= 0)
      {  error("%s:%d: invalid number of rows", ds->fn, ds->cn);
         goto fail;
      }
      if (n <= 0)
      {  error("%s:%d: invalid number of columns", ds->fn, ds->cn);
         goto fail;
      }
      if (strcmp(str, "?") && strcmp(str, "O") && strcmp(str, "F") &&
          strcmp(str, "I") && strcmp(str, "N") && strcmp(str, "U"))
      {  error("%s:%d: invalid status of solution", ds->fn, ds->cn);
         goto fail;
      }
      sol = create_lpsol(m, n);
      sol->status = str[0];
      sol->objval = objval;
      for (k = 1; k <= m+n; k++)
      {  if (scan_data(ds, 's', str, 'e', &valx, 'e', &dx, 0))
            goto fail;
         if (strcmp(str, "?") && strcmp(str, "B") && strcmp(str, "L") &&
             strcmp(str, "U") && strcmp(str, "F") && strcmp(str, "S"))
         {  error("%s:%d: invalid status of variable", ds->fn, ds->cn);
            goto fail;
         }
         sol->tagx[k] = str[0];
         sol->valx[k] = valx;
         sol->dx[k] = dx;
      }
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "EOD"))
      {  error("%s:%d: marker `EOD' not found", ds->fn, ds->cn);
         goto fail;
      }
      close_data(ds);
      return sol;
fail: if (ds != NULL) close_data(ds);
      if (sol != NULL) delete_lpsol(sol);
      return NULL;
}

/* eof */
