/* glpapi/glp_simplex.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glpk.h"
#include "glprsm.h"

/*----------------------------------------------------------------------
-- glp_simplex - solve problem by means of the revised simplex method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_simplex(void);
--
-- *Description*
--
-- The glp_simplex routine is a LP problem solver based on the two-phase
-- revised simplex method.
--
-- This routine obtains problem data from the workspace, solves problem,
-- and stores the computed solution and other relevant information back
-- to the workspace in order that the application program could use this
-- information for further processing.
--
-- Generally, the glp_simplex routine performs the following actions:
--
-- preparing internal data structures;
-- searching for feasible basis solution (pahse I);
-- searching for optimal basis solution (phase II);
-- storing the computed solution to the workspace.
--
-- Since large scale problems may take a long time, the glp_simplex
-- routine reports some information about the current status of the
-- searching. This information is sent to stdout approximately once per
-- second and has the following format:
--
--    *nnn:   objval = xxx   infsum = yyy (ddd)
--
-- where nnn is an iteration count, xxx is the current value of the
-- objective function (which is unscaled and has correct sign), yyy is
-- the current value of sum of infeasibilities (which is scaled, so may
-- be used only for visual estimating), ddd is the current number of
-- basis fixed variables. If the asterisk (*) precedes to nnn, the
-- solver is searching for optimal so-lution (i.e. the feasible solution
-- has been found yet), otherwise the solver is searching for some
-- feasible solution.
--
-- Note that the glp_simplex solver is not perfect. Although it has been
-- successfully tested on a wide set of LP problems, there are hard
-- problems, which can't be solved by this solver.
--
-- *Control parameters*
--
-- The behavior of the glp_simplex routine depends on a number of
-- control parameters, which are described in the program documentation.
--
-- If the problem is not very hard, default values of control parameters
-- are fit for most cases, so the user needn't take care of them.
--
-- *Returns*
--
-- 0 - no errors. This code means that the solver has successfully
--     finished solving the problem (note that, for example, if the
--     problem has no feasible solution, the code returned by the solver
--     will be equal to zero);
-- 1 - it's not possible to start solving the problem because of
--     incorrect data. All diagnostics was sent to stderr;
-- 2 - the solver is not able to solve the problem. All diagnostics was
--     sent to stderr. */

static RSM *rsm;
/* pointer to the revised simplex method common block */

static void store_sol(int status);
/* store basis solution to the workspace */

int glp_simplex(void)
{     /* control parameters */
      int obj_dir, scale, form, relax, use_dual, steep, sum_aij;
      double tol_aij, tol_bnd, tol_dj, tol_piv;
      char obj_row[GLP_MAX_NAME+1];
      /* local variables */
      int m, n, i, j, k, kase, ret;
      print("Solving LP problem...");
      rsm = NULL;
      /* extract relevant control parameters */
      insist(glp_get_ipar("obj_dir", &obj_dir) == 0);
      insist(glp_get_cpar("obj_row", obj_row) == 0);
      insist(glp_get_ipar("scale", &scale) == 0);
      insist(glp_get_ipar("spx_form", &form) == 0);
      insist(glp_get_ipar("spx_relax", &relax) == 0);
      insist(glp_get_ipar("spx_steep", &steep) == 0);
      insist(glp_get_ipar("spx_use_dual", &use_dual) == 0);
      insist(glp_get_ipar("sum_aij", &sum_aij) == 0);
      insist(glp_get_rpar("tol_aij", &tol_aij) == 0);
      insist(glp_get_rpar("tol_bnd", &tol_bnd) == 0);
      insist(glp_get_rpar("tol_dj", &tol_dj) == 0);
      insist(glp_get_rpar("tol_piv", &tol_piv) == 0);
      /* renumber and count rows */
      m = 0;
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW)) glp_set_seqn(GLP_ROW, ++m);
      if (m == 0)
      {  error("glp_simplex: problem has no rows");
         ret = 1;
         goto done;
      }
      /* renumber and count columns */
      n = 0;
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL)) glp_set_seqn(GLP_COL, ++n);
      if (n == 0)
      {  error("glp_simplex: problem has no columns");
         ret = 1;
         goto done;
      }
      /* create revised simplex method common block */
      rsm = umalloc(sizeof(RSM));
      /* --- problem data segment --- */
      rsm->m = m;
      rsm->n = n;
      rsm->type = ucalloc(1+m+n, sizeof(int));
      rsm->lb = ucalloc(1+m+n, sizeof(double));
      rsm->ub = ucalloc(1+m+n, sizeof(double));
      /* determine types and bounds of variables */
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int seqn, type;
            double lb, ub;
            glp_get_seqn(what, &seqn);
            glp_get_bounds(what, &type, &lb, &ub);
            k = (kase == 0 ? 0 : m) + seqn;
            insist(1 <= k && k <= m+n);
            switch (type)
            {  case GLP_FR:
                  rsm->type[k] = LP_FR;
                  rsm->lb[k] = rsm->ub[k] = 0.0;
                  break;
               case GLP_LO:
                  rsm->type[k] = LP_LO;
                  rsm->lb[k] = lb, rsm->ub[k] = 0.0;
                  break;
               case GLP_UP:
                  rsm->type[k] = LP_UP;
                  rsm->lb[k] = 0.0, rsm->ub[k] = ub;
                  break;
               case GLP_DB:
                  rsm->type[k] = LP_DB;
                  rsm->lb[k] = lb, rsm->ub[k] = ub;
                  insist(lb < ub);
                  break;
               case GLP_FX:
                  rsm->type[k] = LP_FX;
                  rsm->lb[k] = rsm->ub[k] = lb;
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* create expanded constraint matrix A~ = (I | -A) */
      rsm->A = create_mat(m, m+n);
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
      {  int ret;
         glp_get_seqn(GLP_ROW, &i);
         /* coefficient at auxiliary variable */
         new_elem(rsm->A, i, i, 1.0);
         /* coefficients at structural variables */
         for (ret = glp_first_coef(GLP_ROW); ret == 0;
              ret = glp_next_coef(GLP_ROW))
         {  double val;
            glp_get_seqn(GLP_COL, &j);
            glp_get_coef(&val);
            if (val == 0.0 || fabs(val) < tol_aij) continue;
            new_elem(rsm->A, i, m+j, -val);
         }
      }
      /* objective function */
      rsm->dir = (obj_dir != GLP_MAX ? LP_MIN : LP_MAX);
      if (obj_row[0] == '\0')
         rsm->obj = 0;
      else
      {  insist(glp_find_item(GLP_ROW, obj_row) == 0);
         insist(glp_get_seqn(GLP_ROW, &rsm->obj) == 0);
         insist(1 <= rsm->obj && rsm->obj <= m);
      }
      /* --- quadratic programming segment --- */
      rsm->qp_flag = 0;
      rsm->mm = 0;
      rsm->nn = 0;
      /* --- problem scaling segment --- */
      rsm->R = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) rsm->R[i] = 1.0;
      rsm->S = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) rsm->S[j] = 1.0;
      /* --- basis solution segment --- */
      rsm->posx = ucalloc(1+m+n, sizeof(int));
      rsm->indb = ucalloc(1+m, sizeof(int));
      rsm->indn = ucalloc(1+n, sizeof(int));
      rsm->tagn = ucalloc(1+n, sizeof(int));
      /* all auxiliary variables are basis */
      for (i = 1; i <= m; i++)
      {  k = i; /* x[k] = xB[i] */
         rsm->posx[k] = +i;
         rsm->indb[i] =  k;
      }
      /* all structural variables are non-basis */
      for (j = 1; j <= n; j++)
      {  k = m+j; /* x[k] = xN[j] */
         rsm->posx[k] = -j;
         rsm->indn[j] =  k;
         switch (rsm->type[k])
         {  case LP_FR:
               rsm->tagn[j] = LP_NF;
               break;
            case LP_LO:
               rsm->tagn[j] = LP_NL;
               break;
            case LP_UP:
               rsm->tagn[j] = LP_NU;
               break;
            case LP_DB:
               rsm->tagn[j] = LP_NL;
               break;
            case LP_FX:
               rsm->tagn[j] = LP_NS;
               break;
            default:
               insist(rsm->type[k] != rsm->type[k]);
         }
      }
      /* --- simplex method segment --- */
      switch (form)
      {  case GLP_EFI:
            /* EFI */
            rsm->efi = create_efi(m);
            rsm->rfi = NULL;
            break;
         case GLP_RFI_BG:
            /* RFI + Bartels & Golub updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_BG;
            break;
         case GLP_RFI_FT:
            /* RFI + Forrest & Tomlin updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_FT;
            break;
         default:
            insist(form != form);
      }
      rsm->bbar = ucalloc(1+m, sizeof(double));
      rsm->cost = ucalloc(1+m+n, sizeof(double));
      rsm->pi = ucalloc(1+m, sizeof(double));
      rsm->cbar = ucalloc(1+n, sizeof(double));
      rsm->dzeta = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) rsm->dzeta[i] = 1.0;
      rsm->gamma = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) rsm->gamma[j] = 1.0;
      rsm->p = 0;
      rsm->tagp = 0;
      rsm->bp = ucalloc(1+m, sizeof(double));
      rsm->ap = ucalloc(1+n, sizeof(double));
      rsm->q = 0;
      rsm->aq = ucalloc(1+m, sizeof(double));
      /* --- control parameters segment --- */
      rsm->prim_steep = steep;
      rsm->prim_relax = relax;
      rsm->dual_steep = steep;
      rsm->dual_relax = relax;
      rsm->tol_bnd = tol_bnd;
      rsm->tol_dj = tol_dj;
      rsm->tol_piv = tol_piv;
      rsm->check_basis = 0;
      rsm->check_dzeta = 0;
      rsm->check_gamma = 0;
      rsm->fin_out = 1;
      /* --- statistics segment --- */
      rsm->iter = 0;
      rsm->t_last = 0;
      /* the following fragment sums multiplets, stores each sum to the
         first element from the corresponding multiplet, and nullifies
         other elements from the same multiplet (of course, all singlets
         remain unchanged) */
      sort_mat(rsm->A);
      for (i = 1; i <= m; i++)
      {  ELEM *ee = NULL, *e;
         /* ee points to the first element of multiplet; e points to the
            current element */
         for (e = rsm->A->row[i]; e != NULL; e = e->row)
         {  if (ee == NULL || ee->j < e->j)
               ee = e;
            else
            {  insist(ee != e && ee->j == e->j);
               if (!sum_aij)
               {  print("glp_simplex: constraint matrix has multiplets")
                     ;
                  ret = 1;
                  goto done;
               }
               ee->val += e->val, e->val = 0.0;
            }
         }
      }
      /* now all zero elements and tiny elements that might appear as
         the result of summation should be removed from the constraint
         matrix */
      scrape_mat(rsm->A, tol_aij);
      /* scale the problem (if required) */
      if (scale) rsm_scaling(rsm);
      /* initialize weights (if steepest edge technique is used) */
      if (rsm->prim_steep) rsm_init_gamma(rsm);
      if (rsm->dual_steep) rsm_init_dzeta(rsm);
      /* compute the initial basis solution (if required) */
      if (obj_dir == GLP_INI)
      {  /* the initial basis matrix is a unity matrix */
         insist(rsm_invert(rsm) == 0);
         /* compute initial values of basis variables */
         rsm_set_bbar(rsm);
         /* check initial solution for primal feasibility */
         if (rsm_check_bbar(rsm, rsm->tol_bnd) == 0)
         {  print("Initial solution is FEASIBLE");
            store_sol(GLP_FEAS);
         }
         else
         {  print("Initial solution is INFEASIBLE");
            store_sol(GLP_INFEAS);
         }
         /* no more optimization */
         ret = 0;
         goto done;
      }
bfs:  /* searching for feasible solution */
      print("Searching for feasible solution...");
      /* reinvert the current basis matrix (this is usually needed to
         improve accuracy of the representation) */
      if (rsm_invert(rsm))
sing: {  error("Numerical problems with basis matrix");
         error("Sorry, basis recovery procedure not implemented");
         ret = 2;
         goto done;
      }
      if (!use_dual)
      {  /* use primal simplex method and implicit artificial variables
            technique */
         ret = rsm_find_feas(rsm);
      }
      else
      {  /* use dual simplex method and special objective function to
            make the initial basis be dual feasible */
         for (k = 1; k <= m+n; k++) rsm->cost[k] = 0.0;
         for (j = 1; j <= n; j++)
         {  k = rsm->indn[j]; /* x[k] = xN[j] */
            switch (rsm->tagn[j])
            {  case LP_NF: rsm->cost[k] =  0.0; break;
               case LP_NL: rsm->cost[k] = +1.0; break;
               case LP_NU: rsm->cost[k] = -1.0; break;
               case LP_NS: rsm->cost[k] =  0.0; break;
               default: insist(rsm->tagn[j] != rsm->tagn[j]);
            }
         }
         ret = rsm_dual_opt(rsm);
      }
      /* analyze return code */
      switch (ret)
      {  case 0:
            /* feasible solution found */
            store_sol(GLP_FEAS);
            if (obj_dir != GLP_ANY) break;
            print("FEASIBLE SOLUTION FOUND");
            ret = 0;
            goto done;
         case 1:
            /* problem has no feasible solution */
            print("PROBLEM HAS NO FEASIBLE SOLUTION");
            store_sol(GLP_NOFEAS);
            ret = 0;
            goto done;
         case 2:
            /* numerical stability lost */
            print("Numerical stability lost");
            goto bfs;
         case 3:
            /* numerical problems with basis matrix */
            goto sing;
         default:
            insist(ret != ret);
      }
      /* searching for optimal solution */
      print("Searching for optimal solution...");
      {  /* set the original objective function (actually the objective
            function is minimized) */
         int k;
         for (k = 1; k <= m+n; k++) rsm->cost[k] = 0.0;
         if (rsm->obj != 0)
            rsm->cost[rsm->obj] = (rsm->dir == LP_MIN ? +1.0 : -1.0);
      }
      ret = rsm_primal_opt(rsm);
      /* analyze returns code */
      switch (ret)
      {  case 0:
            /* optimal solution found */
            print("OPTIMAL SOLUTION FOUND");
            store_sol(GLP_OPT);
            ret = 0;
            break;
         case 1:
            /* problem has unbounded solution */
            print("PROBLEM HAS UNBOUNDED SOLUTION");
            store_sol(GLP_UNBND);
            ret = 0;
            break;
         case 2:
            /* numerical stability lost */
            print("Numerical stability lost");
            goto bfs;
         case 3:
            /* numerical problems with basis matrix */
            goto sing;
         default:
            insist(ret != ret);
      }
done: /* delete revised simplex method common block */
      if (rsm != NULL)
      {  ufree(rsm->type);
         ufree(rsm->lb);
         ufree(rsm->ub);
         delete_mat(rsm->A);
         ufree(rsm->R);
         ufree(rsm->S);
         ufree(rsm->posx);
         ufree(rsm->indb);
         ufree(rsm->indn);
         ufree(rsm->tagn);
         if (rsm->efi != NULL) delete_efi(rsm->efi);
         if (rsm->rfi != NULL) delete_rfi(rsm->rfi);
         ufree(rsm->bbar);
         ufree(rsm->cost);
         ufree(rsm->pi);
         ufree(rsm->cbar);
         ufree(rsm->dzeta);
         ufree(rsm->gamma);
         ufree(rsm->bp);
         ufree(rsm->ap);
         ufree(rsm->aq);
         ufree(rsm);
      }
      /* return to the application program */
      return ret;
}

/*----------------------------------------------------------------------
-- store_sol - store basis solution to the workspace.
--
-- This routine stores all information related to the basis solution
-- found by the solver to the workspace. The parameter status specifies
-- the status of the solution. */

static void store_sol(int status)
{     int m = rsm->m, n = rsm->n, round, i, j, k, kase, ret;
      double tol_bnd, tol_dj;
      insist(glp_get_ipar("round", &round) == 0);
      insist(glp_get_rpar("tol_bnd", &tol_bnd) == 0);
      insist(glp_get_rpar("tol_dj", &tol_dj) == 0);
      /* set the solution status */
      insist(glp_set_ipar("status", status) == 0);
      /* set the original objective function */
      for (k = 1; k <= m+n; k++) rsm->cost[k] = 0.0;
      if (rsm->obj != 0) rsm->cost[rsm->obj] = 1.0;
      /* compute values of all basis variables and reduced costs of all
         non-basis variables */
      rsm_set_bbar(rsm);
      rsm_set_pi(rsm);
      rsm_set_cbar(rsm);
      /* store solution back to the workspace */
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int seqn, type, tagx;
            double lb, ub, valx, dx;
            glp_get_seqn(what, &seqn);
            glp_get_bounds(what, &type, &lb, &ub);
            k = (kase == 0 ? 0 : m) + seqn;
            insist(1 <= k && k <= m+n);
            if (rsm->posx[k] > 0)
            {  /* basis variable */
               i = +rsm->posx[k]; /* x[k] = xB[i] */
               tagx = GLP_BS;
               if (fabs(rsm->bbar[i]) < tol_bnd)
                  valx = 0.0;
               else
                  valx = rsm->bbar[i];
               dx = 0.0;
            }
            else
            {  /* non-basis variable */
               j = -rsm->posx[k]; /* x[k] = xN[j] */
               switch (rsm->tagn[j])
               {  case LP_NL: tagx = GLP_NL; break;
                  case LP_NU: tagx = GLP_NU; break;
                  case LP_NF: tagx = GLP_NF; break;
                  case LP_NS: tagx = GLP_NS; break;
                  default: insist(rsm->tagn[j] != rsm->tagn[j]);
               }
               valx = rsm_get_xn(rsm, j);
               if (fabs(rsm->cbar[j]) < tol_dj)
                  dx = 0.0;
               else
                  dx = rsm->cbar[j];
            }
            /* unscale values and reduced costs */
            if (k <= m)
            {  /* auxiliary variable */
               valx /= rsm->R[k];
               dx   *= rsm->R[k];
            }
            else
            {  /* structural variable */
               valx *= rsm->S[k-m];
               dx   /= rsm->S[k-m];
            }
            /* all reduced costs should be divided by the scale factor
               R[obj], because cost[obj] = 1 corresponds to the *scaled*
               problem */
            if (rsm->obj != 0) dx /= rsm->R[rsm->obj];
            /* assign tagx, valx, and dx to x[k] */
            insist(glp_set_activity(what, tagx, valx, dx) == 0);
         }
      }
      return;
}

/* eof */
