/* glpefi.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPEFI_H
#define _GLPEFI_H

#include "glpeta.h"
#include "glpgel.h"

#define create_efi            glp_create_efi
#define build_efi             glp_build_efi
#define efi_ftran             glp_efi_ftran
#define efi_btran             glp_efi_btran
#define update_efi            glp_update_efi
#define delete_efi            glp_delete_efi

/* The structure EFI defines Elimination Form of the Inverse, which is
-- the following representation of the current basis matrix B:
--
--    B = B0 * H,
--
-- where B0 is the initial basis matrix, H is the eta-matrix.
--
-- The initial basis matrix B0 has the form of LU-factorization:
--
--    B0 = P * L * U * Q,
--
-- where P and Q are permutation matrices, L is a lower triangular
-- matrix with unit diagonal, U is an upper triangular matrix.
--
-- The eta-matrix H has the form of eta-file:
--
--    H = H(1) * H(2) * ... * H(t),
--
-- where H(1), H(2), ..., H(t) are elementary matrix multipliers or
-- eta-terms, t is the total number of terms (size of eta-file). */

typedef struct EFI EFI;

struct EFI
{     /* elimination form of the inverse (EFI) */
      int m;
      /* order of basis matrix */
      LU *lu;
      /* initial basis matrix B0 in the form of LU-factorization */
      HFILE *eta;
      /* eta-matrix H in the form of eta-file */
      double *col;
      /* transformed column corresponding to that non-basis variable,
         which has been chosen to enter the basis */
      int flag;
      /* transformed column flag:
         0 - column is not prepared yet
         1 - column is ready for updating */
      double *work; /* double work[1+m]; */
      /* auxiliary array */
};

extern EFI *create_efi(int m);
/* create EFI */

extern int build_efi(EFI *efi, MAT *A, int indb[]);
/* build EFI for given basis matrix */

extern double *efi_ftran(EFI *efi, double z[], int save);
/* perform forward transformation (FTRAN) using EFI */

extern double *efi_btran(EFI *efi, double z[]);
/* perform backward transformation (BTRAN) using EFI */

extern int update_efi(EFI *efi, int p);
/* update EFI for adjacent basis matrix */

extern void delete_efi(EFI *efi);
/* delete EFI */

#endif

/* eof */
