/* glpimg/create_img.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <limits.h>
#include "glpimg.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- create_img - create raster image.
--
-- *Synopsis*
--
-- #include "glpimg.h"
-- IMG *create_img(int type, int xsize, int ysize);
--
-- *Description*
--
-- The create_img routine creates a raster image.
--
-- The parameter type specifies the image type:
--
-- IMG_2    2 colors (1 bit per pixel);
-- IMG_4    4 colors (2 bits per pixel);
-- IMG_16   16 colors (4 bits per pixel);
-- IMG_256  256 colors (8 bits per pixel).
--
-- The parameters xsize and ysize specify respectively width and height
-- of the image (in pixels).
--
-- *Returns*
--
-- The create_img returns a pointer to the created image. */

int (*save_img)(IMG *img, char *fname) = save_as_bmp;

IMG *create_img(int type, int xsize, int ysize)
{     IMG *img;
      int depth;
      assert(CHAR_BIT == 8);
      if (!(type == IMG_2 || type == IMG_4 || type == IMG_16 ||
            type == IMG_256))
         fault("create_img: invalid image type");
      if (!(1 <= xsize && xsize <= 16000))
         fault("create_img: invalid image width");
      if (!(1 <= ysize && ysize <= 16000))
         fault("create_img: invalid image height");
      img = umalloc(sizeof(IMG));
      img->type = type;
      img->xres = img->yres = 2835; /* 72 pixels per inch (25.4 mm) */
      img->xsize = xsize;
      img->ysize = ysize;
      switch (img->type)
      {  case IMG_2: /* 2 colors (1 bit per pixel) */
            depth = 2;
            img->len = (img->xsize + 7) / 8;
            img->color = 1;
            break;
         case IMG_4: /* 4 colors (2 bits per pixel) */
            depth = 4;
            img->len = (2 * img->xsize + 7) / 8;
            img->color = 3;
            break;
         case IMG_16: /* 16 colors (4 bits per pixel) */
            depth = 16;
            img->len = (4 * img->xsize + 7) / 8;
            img->color = 15;
            break;
         case IMG_256: /* 256 colors (8 bits per pixel) */
            depth = 256;
            img->len = img->xsize;
            img->color = 15;
            break;
         default:
            assert(img->type != img->type);
      }
      img->rgb = ucalloc(depth, sizeof(RGB));
      reset_rgb(img);
      img->scan = ucalloc(img->ysize, img->len);
      img->x0 = img->y0 = 0;
      img->x1 = img->y1 = 0;
      img->x2 = xsize - 1;
      img->y2 = ysize - 1;
      img->x = img->y = 0;
      return img;
}

/* eof */
