/* glpgel/build_lu.c */
 
/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpgel.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- build_lu - compute LU-factorization of given matrix.
--
-- *Synopsis*
--
-- #include "glpgel.h"
-- int build_lu(LU *lu, MAT *A, double tol, double lim,
--    void (*monit)(LU *lu, int kase, int i, int k, double f),
--    double *Amax, double *Ubig);
--
-- *Description*
--
-- The build_lu routine computes LU-factorization of the sparse matrix
-- A in the form A = P*L*U*Q, where L is lower triangular matrix with
-- unit diagonal, U is upper triangular matrix, P and Q are permutation
-- matrices. To perform factorization the build_lu routine uses the gel
-- routine which implements sparse gaussian elimination.
--
-- The LU-factorization which lu points to should be created before by
-- the create_lu routine and may be used many times for different
-- matrices of the same order to factorize them by means of the build_lu
-- routine. It is allowed to use lu->U as the input matrix A; otherwise
-- the matrix A remains unchanged.
--
-- The parameter tol is the tolerance used for threshold pivoting. It
-- should be in the range 0 < tol < 1. An element u[i,j] of the active
-- submatrix is considered as acceptable (from the point of view of the
-- numerical stability) to be a pivot, if |u[i,j]| >= tol * max|u[i,*]|
-- on some elimination step. Decreasing tol usually involves improving
-- sparsity at the expense of accuracy and vice versa. If the input
-- matrix is expected to be well conditioned, tol = 0.01-0.1 is a good
-- choice. For more hard matrices tol sholud be increased. For details
-- see the description of the gel routine.
--
-- The parameter lim specifies maximal allowable growth of elements of
-- the matrix U during factorization process. The process is considered
-- as numerically stable if the condition Ubig <= lim * Amax is true on
-- each step (parameters Amax and Ubig are described below). If double
-- corresponds to 16 decimal places, lim = 1e10-1e12 may be used.
--
-- The monit routine is intended for monitoring purposes. The build_lu
-- routine calls the monit routine in the following cases (recognised
-- by the parameter kase):
--
-- kase = 0: once before the beginning of factorization (parameters i,
--           k, and f should be ignored);
--
-- kase = 1: each time when the next elementary gaussian transformation
--           on the matrix U has been performed:
--
--           (i-th row of U) := (i-th row of U) - f * (k-th row of U),
--
--           where i is number of transformed row, k is number of pivot
--           row. As a result of this transformation the next non-zero
--           element is appended to the matrix L. Should note that
--           during factorization lu->L and lu->U differ from matrices
--           L and U in ordering of rows and columns that is determined
--           by permutation matrices P and Q. This is used to avoid
--           explicit permutations of rows and columns of L and U (for
--           details see the build_lu routine).
--
-- kase = 2: once after the end of factorization (parameters i, k, and
--           f should be ignored).
--
-- There is standard monitoring routine monit_lu, that can be passed to
-- the build_lu routine. Each second the monit_lu routine outputs the
-- following: n (order of the matrix A), k (number of elimination step),
-- nz(L) (number of non-zeros in the matrix L), and nz(U) (number of
-- non-zeros in the matrix U).
--
-- If the parameter monit is NULL, no monitoring is performed.
--
-- The variables Amax and Ubig (i.e. locations which these parameters
-- point to) are ignored before a call. After a call the variable Amax
-- will contain the maximum of absolute values of elements of the input
-- matrix A, and the variable Ubig will contain the maximum of absolute
-- values of those elements which appeared in the matrix U during
-- factorization.
--
-- Factorization may fail when the matrix A is numerically singular or
-- ill conditioned. In this case the matrices L and U will look as the
-- following (k is number of factorization step that failed):
--
--    1       k         n        1       k         n
-- 1  1 . . . . . . . . .     1  x x x x x x x x x x
--    x 1 . . . . . . . .        . x x x x x x x x x
--    x x 1 . . . . . . .        . . x x x x x x x x
--    x x x 1 . . . . . .        . . . x x x x x x x
-- k  x x x x 1 . . . . .     k  . . . . * * * * * *
--    x x x x x 1 . . . .        . . . . * * * * * *
--    x x x x x . 1 . . .        . . . . * * * * * *
--    x x x x x . . 1 . .        . . . . * * * * * *
--    x x x x x . . . 1 .        . . . . * * * * * *
-- n  x x x x x . . . . 1     n  . . . . * * * * * *
--
-- where '*' marks the submatrix of the matrix U that caused a problem.
-- Since A = P*L*U*Q even in case of error, this information may be used
-- to perform further analysis of the matrix A.
--
-- *Returns*
--
-- The build_lu routine returns one of the following error codes:
--
--  0 - no errors, factorization has been successfully computed;
-- -k - on k-th factorization step pivot can't be chosen because all
--      elements of the active submatrix are zeros;
-- +k - on k-th factorization step numerical stability condition (see
--      above) has been violated. */

static LU *lu;
/* LU-factorization of the matrix A */

static void (*monit)(LU *lu, int kase, int i, int k, double f);
/* monitoring routine */

static void func(int i, int k, double f);
/* auxiliary routine */

int build_lu(LU *_lu, MAT *A, double tol, double lim,
      void (*_monit)(LU *lu, int kase, int i, int k, double f),
      double *Amax, double *Ubig)
{     MAT *M, *V;
      DUFF *rs, *cs;
      int k, ret, dum = 0;
      double *rmax, *work;
      /* initialization */
      lu = _lu;
      if (!(A->m == lu->n && A->n == lu->n))
         fault("build_lu: order of LU-factorization conflicts with orde"
            "r of input matrix");
      if (!(0.0 < tol && tol < 1.0))
         fault("build_lu: invalid value of tol");
      if (lim < 1.0)
         fault("build_lu: invalid value of lim");
      monit = _monit;
      *Amax = *Ubig = 0.0;
      M = lu->L; /* M is alias for L */
      V = lu->U; /* V is alias for U */
      /* compute auxiliary factorization A = M*V, where L = P*M*P' and
         U = P*V*Q (order of rows and columns of the matrices M and V
         remain unchanged during transformation of the matrix U to upper
         triangular form) */
      /* P := I, M := I, V := A, Q := I */
      reset_per(lu->P);
      clear_mat(M);
      for (k = 1; k <= lu->n; k++) new_elem(M, k, k, 1.0);
      copy_mat(V, A);
      reset_per(lu->Q);
      /* allocate working storage */
      rs = create_duff(lu->n, lu->n);
      cs = create_duff(lu->n, lu->n);
      rmax = ucalloc(1+lu->n, sizeof(double));
      work = ucalloc(1+lu->n, sizeof(double));
      /* transform the matrix U = P*V*Q to upper triangular form using
         gaussian elimination; elementary gaussian transformations will
         be accumulated in the matrix M */
      if (monit != NULL) monit(lu, 0, 0, 0, 0.0);
      ret = gel(V, lu->P, lu->Q, func, tol, lim, &dum, Amax, Ubig, rs,
         cs, rmax, work);
      if (monit != NULL) monit(lu, 2, 0, 0, 0.0);
      /* compute final factorization A = P*L*U*Q */
      assert(sizeof(void *) <= sizeof(double));
      /* U := P*V*Q, L := P*M*PT, P := P', Q := Q' */
      copy_mat(lu->U, V);
      per_mat(lu->P, lu->U, (void **)work);
      mat_per(lu->U, lu->Q, (void **)work);
      copy_mat(lu->L, M);
      per_mat(lu->P, lu->L, (void **)work);
      inv_per(lu->P);
      mat_per(lu->L, lu->P, (void **)work);
      inv_per(lu->Q);
      /* free working storage */
      delete_duff(rs);
      delete_duff(cs);
      ufree(rmax);
      ufree(work);
      /* return to the calling program */
      return ret;
}

/*----------------------------------------------------------------------
-- func - accumulate elementary transformations.
--
-- The gel routine calls this routine each time when the following
-- elementary gaussian transformation is applied to the matrix U:
--
-- (i-th row of U) := (i-th row of U) - f * (k-th row of U).
--
-- This routine is used i, k, and f to append the next non-zero to the
-- matrix M that determines auxiliary factorization A = M*V. The matrix
-- M differs from the lower triangular matrix L only in symmetric order
-- of rows and columns. */

static void func(int i, int k, double f)
{     MAT *M = lu->L; /* M is alias for L */
      /* row numbers i and k correspond to the matrix U, but the matrix
         M compensates the corresponding elementary transformation of
         the matrix V, not U; therefore instead i and k we should use
         row numbers i' and k' that correspond to the matrix V and that
         is determined by the row permutation matrix P */
      int ii = lu->P->row[i], kk = lu->P->row[k];
      new_elem(M, ii, kk, f);
      if (monit != NULL) monit(lu, 1, i, k, f);
      return;
}

/* eof */
