namespace jnt.scimark2 {

using System;

/**
	SciMark2: A Java numerical benchmark measuring performance
	of computational kernels for FFTs, Monte Carlo simulation,
	sparse matrix computations, Jacobi SOR, and dense LU matrix
	factorizations.  

	Ported to C# by Rhys Weatherley <rweather@southern-storm.com.au>


*/


public class commandline
{

  /* Benchmark 5 kernels with individual Mflops.
	 "results[0]" has the average Mflop rate.

  */


	public static void Main(String[] args)
	{
		// Determine if the runtime engine has floating-point
		// support and the "Math" library.
		double a, b;
		try
		{
			a = Math.Sin(0.0);
			b = 0.0;
		}
		catch(NotImplementedException)
		{
			Console.WriteLine("Floating-point is not supported by the runtime engine");
			Console.WriteLine();
			return;
		}

		// default to the (small) cache-contained version

		double min_time = Constants.RESOLUTION_DEFAULT;

		int FFT_size = Constants.FFT_SIZE;
		int SOR_size =  Constants.SOR_SIZE;
		int Sparse_size_M = Constants.SPARSE_SIZE_M;
		int Sparse_size_nz = Constants.SPARSE_SIZE_nz;
		int LU_size = Constants.LU_SIZE;

		// look for runtime options

        if (args.Length > 0)
        {

			if (String.Compare(args[0], "-h", true) == 0 || 
				String.Compare(args[0], "-help", true) == 0)
			{
				Console.Out.WriteLine("Usage: [-large] [minimum_time]");
				return;
			}

			int current_arg = 0;
			if (String.Compare(args[current_arg], "-large", true) == 0)
			{
				FFT_size = Constants.LG_FFT_SIZE;
				SOR_size =  Constants.LG_SOR_SIZE;
				Sparse_size_M = Constants.LG_SPARSE_SIZE_M;
				Sparse_size_nz = Constants.LG_SPARSE_SIZE_nz;
				LU_size = Constants.LG_LU_SIZE;

				current_arg++;
			}

			if (args.Length > current_arg)
        		min_time = Double.Parse(args[current_arg]);
        }
        

		// run the benchmark
		Console.Out.WriteLine();
		Console.Out.WriteLine("Computing SciMark 2.0a scores (should take about 30 seconds) ...");

		double[] res = new double[6];
		Random R = new Random(Constants.RANDOM_SEED);

		res[1] = kernel.measureFFT( FFT_size, min_time, R);
		res[2] = kernel.measureSOR( SOR_size, min_time, R);
		res[3] = kernel.measureMonteCarlo(min_time, R);
		res[4] = kernel.measureSparseMatmult( Sparse_size_M, 
					Sparse_size_nz, min_time, R);
		res[5] = kernel.measureLU( LU_size, min_time, R);


		res[0] = (res[1] + res[2] + res[3] + res[4] + res[5]) / 5;


	    // print out results

		Console.Out.WriteLine();
		Console.Out.WriteLine("SciMark 2.0a");
		Console.Out.WriteLine();
		Console.Out.WriteLine("Composite Score: " + res[0]);
		Console.Out.Write("FFT ("+FFT_size+"): ");
		if (res[1]==0.0)
			Console.Out.WriteLine(" ERROR, INVALID NUMERICAL RESULT!");
		else
			Console.Out.WriteLine(res[1]);

		Console.Out.WriteLine("SOR ("+SOR_size+"x"+ SOR_size+"): "
				+ "  " + res[2]);
		Console.Out.WriteLine("Monte Carlo : " + res[3]);
		Console.Out.WriteLine("Sparse matmult (N="+ Sparse_size_M+ 
				", nz=" + Sparse_size_nz + "): " + res[4]);
		Console.Out.Write("LU (" + LU_size + "x" + LU_size + "): ");
		if (res[5]==0.0)
			Console.Out.WriteLine(" ERROR, INVALID NUMERICAL RESULT!");
		else
			Console.Out.WriteLine(res[5]);

#if false
		// print out System info (not relevant for C#)
		Console.Out.WriteLine();
		Console.Out.WriteLine("java.vendor: " + 
				System.getProperty("java.vendor"));
		Console.Out.WriteLine("java.version: " + 
				System.getProperty("java.version"));
		Console.Out.WriteLine("os.arch: " +
				System.getProperty("os.arch"));
		Console.Out.WriteLine("os.name: " +
				System.getProperty("os.name"));
		Console.Out.WriteLine("os.version: " +
				System.getProperty("os.version"));
#endif

	}
  
}

}
