/*
 * KeyCode.cs - Key code values.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

/// <summary>
/// <para>Special key codes for use with <c>Curses</c> classes.</para>
/// </summary>
///
/// <remarks>
/// <para>The <c>Input.GetNextChar</c> method gets the next character
/// from the input stream.  It returns either a Unicode character,
/// coerced to <see langword="int"/>, or one of the key codes
/// from this class.</para>
///
/// <para>The values of these constants have been shifted up by
/// 16 bits from their usual ncurses values.  This is to avoid
/// conflicts between key codes and Unicode characters.</para>
/// </remarks>
public sealed class KeyCode
{

	/// <summary>
	/// <para>Minimum curses key code.</para>
	/// </summary>
	public const int KEY_MIN		 = 0x01010000;

	/// <summary>
	/// <para>Break key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This key code may be unreliable on some platforms.
	/// The key may be reported as <c>'\u007F'</c> instead.</para>
	/// </remarks>
	public const int KEY_BREAK       = 0x01010000;

	/// <summary>
	/// <para>Down arrow key code.</para>
	/// </summary>
	public const int KEY_DOWN        = 0x01020000;

	/// <summary>
	/// <para>Up arrow key code.</para>
	/// </summary>
	public const int KEY_UP          = 0x01030000;

	/// <summary>
	/// <para>Left arrow key code.</para>
	/// </summary>
	public const int KEY_LEFT        = 0x01040000;

	/// <summary>
	/// <para>Right arrow key code.</para>
	/// </summary>
	public const int KEY_RIGHT       = 0x01050000;

	/// <summary>
	/// <para>Home key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The home key may be reported as either <c>KEY_HOME</c>,
	/// <c>KEY_BEG</c>, or <c>KEY_A1</c> depending upon the system
	/// and its current configuration.</para>
	/// </remarks>
	public const int KEY_HOME        = 0x01060000;

	/// <summary>
	/// <para>Backspace key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This key code may be unreliable on some platforms.
	/// The key may be reported as <c>'\u0008'</c> or <c>'\u007F'</c>
	/// instead.</para>
	///
	/// <para>A good rule of thumb is to recognise <c>KEY_BACKSPACE</c>,
	/// <c>KEY_BREAK</c>, <c>'\u0008'</c>, and <c>'\u007F'</c> as all
	/// meaning "back space".  This will catch most platforms, including
	/// those where the backspace key has been mapped in a weird manner.</para>
	/// </remarks>
	public const int KEY_BACKSPACE   = 0x01070000;

	/// <summary>
	/// <para>F0 key code.</para>
	/// </summary>
	public const int KEY_F0          = 0x01080000;

	/// <summary>
	/// <para>F1 key code.</para>
	/// </summary>
	public const int KEY_F1          = 0x01090000;

	/// <summary>
	/// <para>F2 key code.</para>
	/// </summary>
	public const int KEY_F2          = 0x010A0000;

	/// <summary>
	/// <para>F3 key code.</para>
	/// </summary>
	public const int KEY_F3          = 0x010B0000;

	/// <summary>
	/// <para>F4 key code.</para>
	/// </summary>
	public const int KEY_F4          = 0x010C0000;

	/// <summary>
	/// <para>F5 key code.</para>
	/// </summary>
	public const int KEY_F5          = 0x010D0000;

	/// <summary>
	/// <para>F6 key code.</para>
	/// </summary>
	public const int KEY_F6          = 0x010E0000;

	/// <summary>
	/// <para>F7 key code.</para>
	/// </summary>
	public const int KEY_F7          = 0x010F0000;

	/// <summary>
	/// <para>F8 key code.</para>
	/// </summary>
	public const int KEY_F8          = 0x01100000;

	/// <summary>
	/// <para>F9 key code.</para>
	/// </summary>
	public const int KEY_F9          = 0x01110000;

	/// <summary>
	/// <para>F10 key code.</para>
	/// </summary>
	public const int KEY_F10         = 0x01120000;

	/// <summary>
	/// <para>F11 key code.</para>
	/// </summary>
	public const int KEY_F11         = 0x01130000;

	/// <summary>
	/// <para>F12 key code.</para>
	/// </summary>
	public const int KEY_F12         = 0x01140000;

	/// <summary>
	/// <para>F13 key code.</para>
	/// </summary>
	public const int KEY_F13         = 0x01150000;

	/// <summary>
	/// <para>F14 key code.</para>
	/// </summary>
	public const int KEY_F14         = 0x01160000;

	/// <summary>
	/// <para>F15 key code.</para>
	/// </summary>
	public const int KEY_F15         = 0x01170000;

	/// <summary>
	/// <para>F16 key code.</para>
	/// </summary>
	public const int KEY_F16         = 0x01180000;

	/// <summary>
	/// <para>F17 key code.</para>
	/// </summary>
	public const int KEY_F17         = 0x01190000;

	/// <summary>
	/// <para>F18 key code.</para>
	/// </summary>
	public const int KEY_F18         = 0x011A0000;

	/// <summary>
	/// <para>F19 key code.</para>
	/// </summary>
	public const int KEY_F19         = 0x011B0000;

	/// <summary>
	/// <para>F20 key code.</para>
	/// </summary>
	public const int KEY_F20         = 0x011C0000;

	/// <summary>
	/// <para>F21 key code.</para>
	/// </summary>
	public const int KEY_F21         = 0x011D0000;

	/// <summary>
	/// <para>F22 key code.</para>
	/// </summary>
	public const int KEY_F22         = 0x011E0000;

	/// <summary>
	/// <para>F23 key code.</para>
	/// </summary>
	public const int KEY_F23         = 0x011F0000;

	/// <summary>
	/// <para>F24 key code.</para>
	/// </summary>
	public const int KEY_F24         = 0x01200000;

	/// <summary>
	/// <para>F25 key code.</para>
	/// </summary>
	public const int KEY_F25         = 0x01210000;

	/// <summary>
	/// <para>F26 key code.</para>
	/// </summary>
	public const int KEY_F26         = 0x01220000;

	/// <summary>
	/// <para>F27 key code.</para>
	/// </summary>
	public const int KEY_F27         = 0x01230000;

	/// <summary>
	/// <para>F28 key code.</para>
	/// </summary>
	public const int KEY_F28         = 0x01240000;

	/// <summary>
	/// <para>F29 key code.</para>
	/// </summary>
	public const int KEY_F29         = 0x01250000;

	/// <summary>
	/// <para>F30 key code.</para>
	/// </summary>
	public const int KEY_F30         = 0x01260000;

	/// <summary>
	/// <para>F31 key code.</para>
	/// </summary>
	public const int KEY_F31         = 0x01270000;

	/// <summary>
	/// <para>F32 key code.</para>
	/// </summary>
	public const int KEY_F32         = 0x01280000;

	/// <summary>
	/// <para>F33 key code.</para>
	/// </summary>
	public const int KEY_F33         = 0x01290000;

	/// <summary>
	/// <para>F34 key code.</para>
	/// </summary>
	public const int KEY_F34         = 0x012A0000;

	/// <summary>
	/// <para>F35 key code.</para>
	/// </summary>
	public const int KEY_F35         = 0x012B0000;

	/// <summary>
	/// <para>F36 key code.</para>
	/// </summary>
	public const int KEY_F36         = 0x012C0000;

	/// <summary>
	/// <para>F37 key code.</para>
	/// </summary>
	public const int KEY_F37         = 0x012D0000;

	/// <summary>
	/// <para>F38 key code.</para>
	/// </summary>
	public const int KEY_F38         = 0x012E0000;

	/// <summary>
	/// <para>F39 key code.</para>
	/// </summary>
	public const int KEY_F39         = 0x012F0000;

	/// <summary>
	/// <para>F40 key code.</para>
	/// </summary>
	public const int KEY_F40         = 0x01300000;

	/// <summary>
	/// <para>F41 key code.</para>
	/// </summary>
	public const int KEY_F41         = 0x01310000;

	/// <summary>
	/// <para>F42 key code.</para>
	/// </summary>
	public const int KEY_F42         = 0x01320000;

	/// <summary>
	/// <para>F43 key code.</para>
	/// </summary>
	public const int KEY_F43         = 0x01330000;

	/// <summary>
	/// <para>F44 key code.</para>
	/// </summary>
	public const int KEY_F44         = 0x01340000;

	/// <summary>
	/// <para>F45 key code.</para>
	/// </summary>
	public const int KEY_F45         = 0x01350000;

	/// <summary>
	/// <para>F46 key code.</para>
	/// </summary>
	public const int KEY_F46         = 0x01360000;

	/// <summary>
	/// <para>F47 key code.</para>
	/// </summary>
	public const int KEY_F47         = 0x01370000;

	/// <summary>
	/// <para>F48 key code.</para>
	/// </summary>
	public const int KEY_F48         = 0x01380000;

	/// <summary>
	/// <para>F49 key code.</para>
	/// </summary>
	public const int KEY_F49         = 0x01390000;

	/// <summary>
	/// <para>F50 key code.</para>
	/// </summary>
	public const int KEY_F50         = 0x013A0000;

	/// <summary>
	/// <para>F51 key code.</para>
	/// </summary>
	public const int KEY_F51         = 0x013B0000;

	/// <summary>
	/// <para>F52 key code.</para>
	/// </summary>
	public const int KEY_F52         = 0x013C0000;

	/// <summary>
	/// <para>F53 key code.</para>
	/// </summary>
	public const int KEY_F53         = 0x013D0000;

	/// <summary>
	/// <para>F54 key code.</para>
	/// </summary>
	public const int KEY_F54         = 0x013E0000;

	/// <summary>
	/// <para>F55 key code.</para>
	/// </summary>
	public const int KEY_F55         = 0x013F0000;

	/// <summary>
	/// <para>F56 key code.</para>
	/// </summary>
	public const int KEY_F56         = 0x01400000;

	/// <summary>
	/// <para>F57 key code.</para>
	/// </summary>
	public const int KEY_F57         = 0x01410000;

	/// <summary>
	/// <para>F58 key code.</para>
	/// </summary>
	public const int KEY_F58         = 0x01420000;

	/// <summary>
	/// <para>F59 key code.</para>
	/// </summary>
	public const int KEY_F59         = 0x01430000;

	/// <summary>
	/// <para>F60 key code.</para>
	/// </summary>
	public const int KEY_F60         = 0x01440000;

	/// <summary>
	/// <para>F61 key code.</para>
	/// </summary>
	public const int KEY_F61         = 0x01450000;

	/// <summary>
	/// <para>F62 key code.</para>
	/// </summary>
	public const int KEY_F62         = 0x01460000;

	/// <summary>
	/// <para>F63 key code.</para>
	/// </summary>
	public const int KEY_F63         = 0x01470000;

	/// <summary>
	/// <para>Delete line key code.</para>
	/// </summary>
	public const int KEY_DL          = 0x01480000;

	/// <summary>
	/// <para>Insert line key code.</para>
	/// </summary>
	public const int KEY_IL          = 0x01490000;

	/// <summary>
	/// <para>Delete character key code.</para>
	/// </summary>
	public const int KEY_DC          = 0x014A0000;

	/// <summary>
	/// <para>Insert character or enter insert mode key code.</para>
	/// </summary>
	public const int KEY_IC          = 0x014B0000;

	/// <summary>
	/// <para>Exit insert mode key code.</para>
	/// </summary>
	public const int KEY_EIC         = 0x014C0000;

	/// <summary>
	/// <para>Clear screen key code.</para>
	/// </summary>
	public const int KEY_CLEAR       = 0x014D0000;

	/// <summary>
	/// <para>Clear to end of screen key code.</para>
	/// </summary>
	public const int KEY_EOS         = 0x014E0000;

	/// <summary>
	/// <para>Clear to end of line key code.</para>
	/// </summary>
	public const int KEY_EOL         = 0x014F0000;

	/// <summary>
	/// <para>Scroll one line forward key code.</para>
	/// </summary>
	public const int KEY_SF          = 0x01500000;

	/// <summary>
	/// <para>Scroll one line backward key code.</para>
	/// </summary>
	public const int KEY_SR          = 0x01510000;

	/// <summary>
	/// <para>Next page key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is the key code that is normally reported for
	/// the "Page Down" key.  However, some systems may use
	/// <c>KEY_NEXT</c> instead, so both should be recognised.</para>
	/// </remarks>
	public const int KEY_NPAGE       = 0x01520000;

	/// <summary>
	/// <para>Previous page key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is the key code that is normally reported for
	/// the "Page Up" key.  However, some systems may use
	/// <c>KEY_PREVIOUS</c> instead, so both should be recognised.</para>
	/// </remarks>
	public const int KEY_PPAGE       = 0x01530000;

	/// <summary>
	/// <para>Set tab key code.</para>
	/// </summary>
	public const int KEY_STAB        = 0x01540000;

	/// <summary>
	/// <para>Clear tab key code.</para>
	/// </summary>
	public const int KEY_CTAB        = 0x01550000;

	/// <summary>
	/// <para>Clear all tabs key code.</para>
	/// </summary>
	public const int KEY_CATAB       = 0x01560000;

	/// <summary>
	/// <para>Enter or send key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This key code may be unreliable on some platforms.
	/// The key may be reported as <c>'\u000D'</c> or <c>'\u000A'</c>
	/// instead.</para>
	///
	/// <para>A good rule of thumb is to recognise <c>KEY_ENTER</c>,
	/// <c>'\u000D'</c>, and <c>'\u000A'</c> as all meaning "enter".
	/// This will catch most platforms, including those where the enter
	/// key has been mapped in a weird manner.</para>
	/// </remarks>
	public const int KEY_ENTER       = 0x01570000;

	/// <summary>
	/// <para>Soft reset key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This key code may be unreliable on some platforms.</para>
	/// </remarks>
	public const int KEY_SRESET      = 0x01580000;

	/// <summary>
	/// <para>Hard reset key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This key code may be unreliable on some platforms.</para>
	/// </remarks>
	public const int KEY_RESET       = 0x01590000;

	/// <summary>
	/// <para>Print key code.</para>
	/// </summary>
	public const int KEY_PRINT       = 0x015A0000;

	/// <summary>
	/// <para>Lower left of keypad key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The lower left of the keypad may be reported as either
	/// <c>KEY_LL</c> or <c>KEY_C3</c>, depending upon the system.</para>
	/// </remarks>
	public const int KEY_LL          = 0x015B0000;

	/// <summary>
	/// <para>Upper left of keypad key code.</para>
	/// </summary>
	public const int KEY_A1			 = 0x015C0000;

	/// <summary>
	/// <para>Upper right of keypad key code.</para>
	/// </summary>
	public const int KEY_A3			 = 0x015D0000;

	/// <summary>
	/// <para>Center of keypad key code.</para>
	/// </summary>
	public const int KEY_B2			 = 0x015E0000;

	/// <summary>
	/// <para>Lower left of keypad key code.</para>
	/// </summary>
	public const int KEY_C1			 = 0x015F0000;

	/// <summary>
	/// <para>Lower right of keypad key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The lower left of the keypad may be reported as either
	/// <c>KEY_LL</c> or <c>KEY_C3</c>, depending upon the system.</para>
	/// </remarks>
	public const int KEY_C3			 = 0x01600000;

	/// <summary>
	/// <para>Back tab key code.</para>
	/// </summary>
	public const int KEY_BTAB		 = 0x01610000;

	/// <summary>
	/// <para>Beginning of line key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This is how the "Home" key is reported on some platforms.</para>
	/// </remarks>
	public const int KEY_BEG		 = 0x01620000;

	/// <summary>
	/// <para>Cancel key code.</para>
	/// </summary>
	public const int KEY_CANCEL		 = 0x01630000;

	/// <summary>
	/// <para>Close key code.</para>
	/// </summary>
	public const int KEY_CLOSE		 = 0x01640000;

	/// <summary>
	/// <para>Command key code.</para>
	/// </summary>
	public const int KEY_COMMAND	 = 0x01650000;

	/// <summary>
	/// <para>Copy key code.</para>
	/// </summary>
	public const int KEY_COPY		 = 0x01660000;

	/// <summary>
	/// <para>Create key code.</para>
	/// </summary>
	public const int KEY_CREATE		 = 0x01670000;

	/// <summary>
	/// <para>End of line key code.</para>
	/// </summary>
	public const int KEY_END		 = 0x01680000;

	/// <summary>
	/// <para>Exit key code.</para>
	/// </summary>
	public const int KEY_EXIT		 = 0x01690000;

	/// <summary>
	/// <para>Find key code.</para>
	/// </summary>
	public const int KEY_FIND		 = 0x016A0000;

	/// <summary>
	/// <para>Help key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>Many applications use "F1" as their help key.  This usage
	/// is encouraged.  However, some keyboards have an explicit "Help"
	/// key which is separate from the function keys.  Therefore,
	/// good applications should recognise both <c>KEY_F1</c> and
	/// <c>KEY_HELP</c> to provide help to the user.</para>
	/// </remarks>
	public const int KEY_HELP		 = 0x016B0000;

	/// <summary>
	/// <para>Mark key code.</para>
	/// </summary>
	public const int KEY_MARK		 = 0x016C0000;

	/// <summary>
	/// <para>Message key code.</para>
	/// </summary>
	public const int KEY_MESSAGE	 = 0x016D0000;

	/// <summary>
	/// <para>Move key code.</para>
	/// </summary>
	public const int KEY_MOVE		 = 0x016E0000;

	/// <summary>
	/// <para>Next key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This may be reported instead of <c>KEY_NPAGE</c> for "Page Down"
	/// on some systems.</para>
	/// </remarks>
	public const int KEY_NEXT		 = 0x016F0000;

	/// <summary>
	/// <para>Open key code.</para>
	/// </summary>
	public const int KEY_OPEN		 = 0x01700000;

	/// <summary>
	/// <para>Options key code.</para>
	/// </summary>
	public const int KEY_OPTIONS	 = 0x01710000;

	/// <summary>
	/// <para>Previous key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This may be reported instead of <c>KEY_PPAGE</c> for "Page Up"
	/// on some systems.</para>
	/// </remarks>
	public const int KEY_PREVIOUS	 = 0x01720000;

	/// <summary>
	/// <para>Redo key code.</para>
	/// </summary>
	public const int KEY_REDO		 = 0x01730000;

	/// <summary>
	/// <para>Reference key code.</para>
	/// </summary>
	public const int KEY_REFERENCE	 = 0x01740000;

	/// <summary>
	/// <para>Refresh key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>You normally won't see this key code if you call
	/// <c>Input.GetNextChar</c>, because it will be intercepted
	/// by the <see cref="T:Curses.Input"/> class and handled
	/// internally.</para>
	/// </remarks>
	public const int KEY_REFRESH	 = 0x01750000;

	/// <summary>
	/// <para>Replace key code.</para>
	/// </summary>
	public const int KEY_REPLACE	 = 0x01760000;

	/// <summary>
	/// <para>Restart key code.</para>
	/// </summary>
	public const int KEY_RESTART	 = 0x01770000;

	/// <summary>
	/// <para>Resume key code.</para>
	/// </summary>
	public const int KEY_RESUME		 = 0x01780000;

	/// <summary>
	/// <para>Save key code.</para>
	/// </summary>
	public const int KEY_SAVE		 = 0x01790000;

	/// <summary>
	/// <para>Shifted beginning of line key code.</para>
	/// </summary>
	public const int KEY_SBEG		 = 0x017A0000;

	/// <summary>
	/// <para>Shifted cancel key code.</para>
	/// </summary>
	public const int KEY_SCANCEL	 = 0x017B0000;

	/// <summary>
	/// <para>Shifted command key code.</para>
	/// </summary>
	public const int KEY_SCOMMAND	 = 0x017C0000;

	/// <summary>
	/// <para>Shifted copy key code.</para>
	/// </summary>
	public const int KEY_SCOPY		 = 0x017D0000;

	/// <summary>
	/// <para>Shifted create key code.</para>
	/// </summary>
	public const int KEY_SCREATE	 = 0x017E0000;

	/// <summary>
	/// <para>Shifted delete character key code.</para>
	/// </summary>
	public const int KEY_SDC		 = 0x017F0000;

	/// <summary>
	/// <para>Shifted delete line key code.</para>
	/// </summary>
	public const int KEY_SDL		 = 0x01800000;

	/// <summary>
	/// <para>Select key code.</para>
	/// </summary>
	public const int KEY_SELECT		 = 0x01810000;

	/// <summary>
	/// <para>Shifted end of line key code.</para>
	/// </summary>
	public const int KEY_SEND		 = 0x01820000;

	/// <summary>
	/// <para>Shifted clear line key code.</para>
	/// </summary>
	public const int KEY_SEOL		 = 0x01830000;

	/// <summary>
	/// <para>Shifted exit key code.</para>
	/// </summary>
	public const int KEY_SEXIT		 = 0x01840000;

	/// <summary>
	/// <para>Shifted find key code.</para>
	/// </summary>
	public const int KEY_SFIND		 = 0x01850000;

	/// <summary>
	/// <para>Shifted help key code.</para>
	/// </summary>
	public const int KEY_SHELP		 = 0x01860000;

	/// <summary>
	/// <para>Shifted help key code.</para>
	/// </summary>
	public const int KEY_SHOME		 = 0x01870000;

	/// <summary>
	/// <para>Shifted insert character key code.</para>
	/// </summary>
	public const int KEY_SIC		 = 0x01880000;

	/// <summary>
	/// <para>Shifted left array key code.</para>
	/// </summary>
	public const int KEY_SLEFT		 = 0x01890000;

	/// <summary>
	/// <para>Shifted message key code.</para>
	/// </summary>
	public const int KEY_SMESSAGE	 = 0x018A0000;

	/// <summary>
	/// <para>Shifted move key code.</para>
	/// </summary>
	public const int KEY_SMOVE		 = 0x018B0000;

	/// <summary>
	/// <para>Shifted next key code.</para>
	/// </summary>
	public const int KEY_SNEXT		 = 0x018C0000;

	/// <summary>
	/// <para>Shifted options key code.</para>
	/// </summary>
	public const int KEY_SOPTIONS	 = 0x018D0000;

	/// <summary>
	/// <para>Shifted previous key code.</para>
	/// </summary>
	public const int KEY_SPREVIOUS	 = 0x018E0000;

	/// <summary>
	/// <para>Shifted print key code.</para>
	/// </summary>
	public const int KEY_SPRINT		 = 0x018F0000;

	/// <summary>
	/// <para>Shifted redo key code.</para>
	/// </summary>
	public const int KEY_SREDO		 = 0x01900000;

	/// <summary>
	/// <para>Shifted replace key code.</para>
	/// </summary>
	public const int KEY_SREPLACE	 = 0x01910000;

	/// <summary>
	/// <para>Shifted right array key code.</para>
	/// </summary>
	public const int KEY_SRIGHT		 = 0x01920000;

	/// <summary>
	/// <para>Shifted resume key code.</para>
	/// </summary>
	public const int KEY_SRSUME		 = 0x01930000;

	/// <summary>
	/// <para>Shifted save key code.</para>
	/// </summary>
	public const int KEY_SSAVE		 = 0x01940000;

	/// <summary>
	/// <para>Shifted suspend key code.</para>
	/// </summary>
	public const int KEY_SSUSPEND	 = 0x01950000;

	/// <summary>
	/// <para>Shifted undo key code.</para>
	/// </summary>
	public const int KEY_SUNDO		 = 0x01960000;

	/// <summary>
	/// <para>Suspend key code.</para>
	/// </summary>
	public const int KEY_SUSPEND	 = 0x01970000;

	/// <summary>
	/// <para>Undo key code.</para>
	/// </summary>
	public const int KEY_UNDO		 = 0x01980000;

	/// <summary>
	/// <para>Mouse event key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>You normally won't see this key code if you call
	/// <c>Input.GetNextChar</c>, because it will be intercepted
	/// by the <see cref="T:Curses.Input"/> class and diverted
	/// elsewhere.</para>
	/// </remarks>
	public const int KEY_MOUSE		 = 0x01990000;

	/// <summary>
	/// <para>Resize terminal key code.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>You normally won't see this key code if you call
	/// <c>Input.GetNextChar</c>, because it will be intercepted
	/// by the <see cref="T:Curses.Input"/> class and handled
	/// internally.</para>
	/// </remarks>
	public const int KEY_RESIZE		 = 0x019A0000;

	/// <summary>
	/// <para>Maximum curses key code.</para>
	/// </summary>
	public const int KEY_MAX		 = 0x01FF0000;

} // class KeyCode

} // namespace Curses
