/*
 * LogLine.cs - Object that represents a line of logs as a game piece.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Frogger
{

using System;
using Curses;

public class LogLine : GamePiece
{
	// Height of logs.
	public const int LogHeight = 4;

	// Internal state.
	private int phase;
	private int length;
	private int space;
	private int delta;
	private Frog sittingFrog;
	private Timeout timeout;

	// Create a new log line game piece at an initial position.
	public LogLine(MainBoard board, DrawingContext dc, int y, int phase,
				   int length, int space, double speed)
			: base(board, dc, 0, y, dc.Window.Width, LogHeight)
			{
				this.phase = phase;
				if(this.phase > 0)
				{
					this.phase -= (length + space);
				}
				this.length = length;
				this.space = space;
				if(speed < 0.0)
				{
					this.delta = -1;
					timeout = new LogTimeout(this, -speed);
				}
				else
				{
					this.delta = 1;
					timeout = new LogTimeout(this, speed);
				}
				sittingFrog = null;
			}

	// Draw a horizontal line, clipped to the screen boundaries.
	private void HLineClipped(DrawingContext dc, int x1, int x2, int y)
			{
				bool haveStart = true;
				bool haveEnd = true;
				int tempx1, tempx2;
				if(x1 < 0)
				{
					haveStart = false;
					x1 = 0;
				}
				if(x2 >= dc.Window.Width)
				{
					haveEnd = false;
					x2 = dc.Window.Width - 1;
				}
				if(x1 <= x2)
				{
					tempx1 = x1;
					if(haveStart)
					{
						dc.Move(x1, y);
						dc.Add('+');
						++tempx1;
					}
					if(haveEnd)
					{
						tempx2 = x2 - 1;
					}
					else
					{
						tempx2 = x2;
					}
					if(tempx1 <= tempx2)
					{
						dc.HLine(tempx1, tempx2, y);
					}
					if(haveEnd)
					{
						dc.Move(x2, y);
						dc.Add('+');
					}
				}
			}

	// Draw a vertical line, clipped to the screen boundaries.
	private void VLineClipped(DrawingContext dc, int x, int y1, int y2)
			{
				if(x >= 0 && x < dc.Window.Width)
				{
					dc.VLine(x, y1, y2);
				}
			}

	// Draw this game piece.
	public override void Draw()
			{
				// Set the color to use to draw the logs.
				dc.Foreground = Color.Yellow;

				// Draw the logs across the screen.
				int xposn = phase;
				while(xposn < width)
				{
					HLineClipped(dc, xposn, xposn + length - 1, y);
					VLineClipped(dc, xposn, y + 1, y + height - 2);
					HLineClipped(dc, xposn, xposn + length - 1, y + height - 1);
					VLineClipped(dc, xposn + length - 1, y + 1, y + height - 2);
					xposn += length + space;
				}
			}

	// Determine if the position a game piece intersects with
	// any of the logs on this log line.  For the purposes of
	// this class, intersections means that the piece is fully
	// enclosed in a log.
	public override bool IntersectsWith(GamePiece piece)
			{
				int pieceX = piece.X;
				int pieceY = piece.Y;
				int pieceWidth = piece.Width;
				int pieceHeight = piece.Height;
				int xposn = phase;
				if(pieceY < y || (pieceY + pieceHeight) > (y + height))
				{
					return false;
				}
				while(xposn < width)
				{
					if(pieceX >= (xposn + 1) &&
					   pieceX < (xposn + length - 1) &&
					   (pieceX + pieceWidth) <= (xposn + length - 1))
					{
						return true;
					}
					xposn += length + space;
				}
				return false;
			}

	// Stop all asynchronous events associated with this game piece.
	public override void StopEvents()
			{
				timeout.Remove();
			}

	// Move this game piece by a specified amount, and redraw.
	private void MoveLogs()
			{
				DrawingContext dc = this.dc;
				int xposn, yposn, right;

				dc.Foreground = Color.Yellow;
				if(delta < 0)
				{
					// Move the log line left.
					--phase;
					if(phase <= -(length + space))
					{
						phase += length + space;
					}
					for(yposn = 0; yposn < height; ++yposn)
					{
						dc.Move(0, y + yposn);
						dc.DeleteChar();
					}

					// Redraw the newly visible part of the last log.
					xposn = phase;
					while(xposn < width)
					{
						xposn += length + space;
					}
					xposn -= length + space;
					right = width - 1;
					if(xposn == right || (xposn + length - 1) == right)
					{
						dc.Move(right, y);
						dc.Add('+');
						for(yposn = 1; yposn < (height - 1); ++yposn)
						{
							dc.Move(right, y + yposn);
							dc.Add('|');
						}
						dc.Move(right, y + height - 1);
						dc.Add('+');
					}
					else if(xposn < right && (xposn + length) > right)
					{
						// Draw the horizontal section of a log.
						dc.Move(right, y);
						dc.Add('-');
						dc.Move(right, y + height - 1);
						dc.Add('-');
					}
				}
				else
				{
					// Move the log line right.
					++phase;
					if(phase > 0)
					{
						phase -= length + space;
					}

					// Redraw the newly visible part of the first log.
					xposn = phase;
					if(xposn == 0 || (xposn + length - 1) == 0)
					{
						// Draw a vertical side of the log.
						dc.Move(0, y);
						dc.InsertChar('+');
						for(yposn = 1; yposn < (height - 1); ++yposn)
						{
							dc.Move(0, y + yposn);
							dc.InsertChar('|');
						}
						dc.Move(0, y + height - 1);
						dc.InsertChar('+');
					}
					else if((xposn + length) > 0)
					{
						// Draw the horizontal section of a log.
						dc.Move(0, y);
						dc.InsertChar('-');
						for(yposn = 1; yposn < (height - 1); ++yposn)
						{
							dc.Move(0, y + yposn);
							dc.InsertChar(' ');
						}
						dc.Move(0, y + height - 1);
						dc.InsertChar('-');
					}
					else
					{
						// Draw the blank space between logs.
						for(yposn = 0; yposn < height; ++yposn)
						{
							dc.Move(0, y + yposn);
							dc.InsertChar(' ');
						}
					}
				}

				// If there is a frog sitting on this log line, then move it.
				if(sittingFrog != null)
				{
					sittingFrog.MoveWithLog(delta);
				}
			}

	// Get or set the frog that is sitting on this log line.
	public Frog SittingFrog
			{
				get
				{
					return sittingFrog;
				}
				set
				{
					sittingFrog = value;
				}
			}

	// Timeout handler for this log line.
	private sealed class LogTimeout : Timeout
	{
		private LogLine logLine;

		public LogTimeout(LogLine logLine, double speed)
			: base(TimeSpan.FromSeconds(speed), TimeSpan.FromSeconds(speed))
			{
				this.logLine = logLine;
			}

		protected override void OnFired(Object sender, EventArgs args)
			{
				logLine.MoveLogs();
			}

	} // LogTimeout

} // class LogLine

} // namespace Frogger
